import Tween from "./tween.js";

/**
 * 获取非百分比Dom高度
 * @param {HTMLBaseElement} elm DOM 对象
 * @returns
 */
const getHeight = (elm) => {
    if (getComputedStyle(elm).display !== "none") {
        return elm.offsetHeight;
    }
    let cssText = elm.style.cssText;
    elm.style.cssText +=
        "position:fixed;left:-9999px;display:block;visibility:hidden";
    let height = elm.offsetHeight;
    elm.style.cssText = cssText;
    return height;
};

/**
 * 获取非百分比Dom宽度
 * @param {HTMLBaseElement} elm DOM 对象
 * @returns
 */
const getWidth = (elm) => {
    if (getComputedStyle(elm).display !== "none") {
        return elm.offsetWidth;
    }
    let cssText = elm.style.cssText;
    elm.style.cssText +=
        "position:fixed;left:-9999px;display:block;visibility:hidden";
    let width = elm.offsetWidth;
    elm.style.cssText = cssText;
    return width;
};
/**
 * 获取DOM宽度不包含padding
 * @param {HTMLBaseElement} elm DOM 对象
 * @returns
 */
const getInnerWidth = (elm) => {
    let style = getComputedStyle(elm);
    let width = getWidth(elm);
    let padding =
        parseFloat(style.paddingLeft) + parseFloat(style.paddingRight);
    return width - padding;
};

/**
 * DOM 对象下拉动画
 * @param {HTMLBaseElement} elm DOM 对象
 * @param {Function} fnCallback 显示完成之后的回调
 * @returns
 */
const slideDown = (elm, fnCallback) => {
    var display = getComputedStyle(elm).display;
    var duration = 300;
    if (display !== "none") return;
    var csstext = elm.style.cssText;
    elm.style.height = "initial";
    var height = getHeight(elm);

    elm.style.display = "block";
    elm.style.height = 0;

    animate(elm, { height: height }, duration, "easeOut", endHandel);
    function endHandel() {
        elm.style.cssText = csstext + "display:block;";
        fnCallback && fnCallback.call(elm);
    }
};

/**
 * DOM 对象上拉动画
 * @param {HTMLBaseElement} elm DOM 对象
 * @param {Function} fnCallback 显示完成之后的回调
 * @returns
 */
const slideUp = (elm, fnCallback) => {
    if (getComputedStyle(elm).display === "none") return;
    var duration = 300;
    var csstext = elm.style.cssText;

    animate(elm, { height: 0 }, duration, "easeOut", endHandel);
    function endHandel() {
        elm.style.cssText = csstext + "display:none;";
        fnCallback && fnCallback.call(elm);
    }
};

/**
 * @description: 添加动画
 * @param {HTMLBaseElement} elm
 * @param {*} attrs css属性键值对
 * @param {*} times 动画持续时间
 * @param {*} fx 对象动画的过渡类型
 * @param {*} fnCallback
 * @return {*}
 */
const animate = (elm, attrs, times, fx, fnCallback) => {
    if (typeof times == "undefined") {
        times = 300;
        fx = "linear";
    }

    if (typeof times == "string") {
        if (typeof fx == "function") {
            fnCallback = fx;
        }
        fx = times;
        times = 300;
    } else if (typeof times == "function") {
        fnCallback = times;
        times = 300;
        fx = "linear";
    } else if (typeof times == "number") {
        if (typeof fx == "function") {
            fnCallback = fx;
            fx = "linear";
        } else if (typeof fx == "undefined") {
            fx = "linear";
        }
    }

    var iCur = {};

    for (var attr in attrs) {
        iCur[attr] = 0;

        if (attr == "opacity") {
            iCur[attr] = Math.round(getStyle(elm, attr) * 100);
        } else {
            iCur[attr] = parseInt(getStyle(elm, attr));
        }
    }

    var startTime = now();

    clearInterval(elm.timer);

    elm.timer = setInterval(function () {
        var changeTime = now();

        var t = times - Math.max(0, startTime - changeTime + times); //0鍒�2000

        for (var attr in attrs) {
            var value = Tween[fx](
                t,
                iCur[attr],
                attrs[attr] - iCur[attr],
                times
            );

            if (attr == "opacity") {
                elm.style.opacity = value / 100;
                elm.style.filter = "alpha(opacity=" + value + ")";
            } else {
                elm.style[attr] = value + "px";
            }
        }

        if (t == times) {
            clearInterval(elm.timer);
            if (fnCallback) {
                fnCallback.call(elm);
            }
        }
    }, 15);

    function getStyle(obj, attr) {
        return getComputedStyle(obj, false)[attr];
    }

    function now() {
        return new Date().getTime();
    }
};

/**
 * dom元素绑定只执行一次的事件对象
 * @param {HTMLBaseElement} elm DOM 对象
 * @param {String} eventType 事件类型
 * @param {Function} fnListener 事件 Listener 方法
 */
const onceEventHandler = (elm, eventType, fnListener) => {
    const f = () => {
        fnListener();
        elm.removeEventListener(eventType, f);
    };
    elm.addEventListener(eventType, f);
};

/**
 * @description:
 * @param {HTMLBaseElement} elm DOM 对象
 * @return {Array} 数组dom
 */
const siblings = (elm) => {
    let collection = [];
    let siblings = elm.parentNode.children;

    for (let i = 0; i < siblings.length; i++) {
        if (siblings[i] !== elm) collection.push(siblings[i]);
    }

    return collection;
};

/**
 * @description:获取dom元素的索引
 * @param {HTMLBaseElement} elm DOM 对象
 * @return {Number}
 */
const getIndex = (elm) => {
    let i = 0;
    while (elm.previousElementSibling) {
        elm = elm.previousElementSibling;
        i++;
    }
    return i;
};

/**
 * @description: 把元素插入到另一个、指定的元素元素后面。
 * @param {HTMLBaseElement} newElm DOM 对象
 * @param {HTMLBaseElement} referenceElm DOM 对象
 * @return
 */
const insertAfter = (newElm, referenceElm) => {
    let _parent = referenceElm.parentNode;
    referenceElm == _parent.lastChild
        ? _parent.appendChild(newElm)
        : _parent.insertBefore(newElm, referenceElm.nextSibling);
};

/**
 * @description: 把元素前置到另一个、指定的元素前。
 * @param {HTMLBaseElement} newElm DOM 对象
 * @param {HTMLBaseElement} referenceElm DOM 对象
 * @return
 */
const prependTo = (newElm, referenceElm) => {
    let firstChild = referenceElm.firstChild;
    if (firstChild) {
        referenceElm.insertBefore(newElm, firstChild);
    } else {
        referenceElm.appendChild(newElm);
    }
};

/**
 * @description:
 * @param {HTMLBaseElement} elm DOM 对象
 * @return {boolean}
 */
const isShow = (elm) => {
    return getComputedStyle(elm).display !== "none";
};

/**
 * @description: 在匹配的元素上触发某类事件
 * @param {HTMLBaseElement} elm DOM 对象
 * @param {String} eventType 事件类型
 * @return {*}
 */
const trigger = (elm, eventType) => {
    if (!elm) {
        return;
    }
    const event = new Event(eventType, { bubbles: true });
    elm.dispatchEvent(event);
};

const throttle = (callback, wait, options) => {
    var args, context;
    var opts = options || {};
    var runFlag = false;
    var isDestroy = false;
    var timeout = 0;
    var optLeading = "leading" in opts ? opts.leading : true;
    var optTrailing = "trailing" in opts ? opts.trailing : false;
    var runFn = function () {
        if (!isDestroy) {
            runFlag = true;
            callback.apply(context, args);
            timeout = setTimeout(endFn, wait);
        }
    };
    var endFn = function () {
        timeout = 0;
        if (!isDestroy && !runFlag && optTrailing === true) {
            runFn();
        }
    };
    var cancelFn = function () {
        var rest = timeout !== 0;
        clearTimeout(timeout);
        args = null;
        context = null;
        runFlag = false;
        timeout = 0;
        return rest;
    };
    var throttled = function () {
        args = arguments;
        context = this;
        runFlag = false;
        if (timeout === 0) {
            if (optLeading === true) {
                runFn();
            } else if (optTrailing === true) {
                timeout = setTimeout(endFn, wait);
            }
        }
    };
    throttled.cancel = cancelFn;
    return throttled;
}

export default {
    getHeight,
    getWidth,
    getInnerWidth,
    slideDown,
    slideUp,
    onceEventHandler,
    siblings,
    getIndex,
    insertAfter,
    prependTo,
    isShow,
    trigger,
    throttle
};
