package com.bokesoft.yes.mid.function;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.UUID;

import com.bokesoft.yes.common.util.DateUtil;
import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yes.log.LogSvr;
import com.bokesoft.yes.mid.io.doc.HisLoad;
import com.bokesoft.yes.mid.migration.process.merge.MigrationMergeProxy;
import com.bokesoft.yes.mid.oid.OIDAllocatorFactory;
import com.bokesoft.yigo.common.struct.PairItem;
import com.bokesoft.yigo.common.struct.PairItemList;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.commondef.MetaStatus;
import com.bokesoft.yigo.meta.commondef.MetaStatusCollection;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.solution.MetaLang;
import com.bokesoft.yigo.meta.solution.MetaLangConfig;
import com.bokesoft.yigo.meta.util.MetaUtil;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.mid.connection.DBType;
import com.bokesoft.yigo.mid.connection.IDBManager;
import com.bokesoft.yigo.mid.document.SaveData;
import com.bokesoft.yigo.mid.oid.IOIDAllocator;
import com.bokesoft.yigo.mid.parser.BaseMidFunctionImpl;
import com.bokesoft.yigo.parser.BaseFunImplCluster;
import com.bokesoft.yigo.parser.IExecutor;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.document.Document;

public class MidBasisFunction extends BaseFunImplCluster {
	public MidBasisFunction() {
		super();
	}

	class SetTableValueImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			String tableKey = TypeConvertor.toString(args[0]);
			String columnKey = TypeConvertor.toString(args[1]);
			Object value = args[2];
			DataTable dataTable = context.getDocument().get(tableKey);
			
			if(dataTable != null ){
				// 加这个处理的目的是为了方便处理头表
				if ( !dataTable.isValid() ) {
					dataTable.first();
				}
				dataTable.setObject(columnKey, value);
				context.getDocument().setModified();
				return true;
			}
			return false;
		}

	}

	/**
	 * 获取Document中的值 参数列表为[tableKey,columnKey],如果参数只有一个，则参数为columnKey，表格为主表
	 * <ul>
	 * <li>tableKey 数据表标识
	 * <li>columnKey 列标识
	 * </ul>
	 * @author 陈瑞
	 *
	 */
	class GetTableValueImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			String tableKey = null;
			String columnKey = null;
			Object value = null;
			Document document = context.getDocument();
			if (args.length == 1) {
				tableKey = document.getMetaDataObject().getMainTableKey();
				columnKey = TypeConvertor.toString(args[0]);
			} else if (args.length == 2) {
				tableKey = TypeConvertor.toString(args[0]);
				columnKey = TypeConvertor.toString(args[1]);
			} else
				return value;

			DataTable dataTable = document.get(tableKey);
			if (dataTable != null) {
				// 加这个处理的目的是为了方便处理头表
				if (!dataTable.isValid()) {
					dataTable.first();
				}
				value = context.isOrignalValue() ? dataTable.getOriginalObject(columnKey) : dataTable.getObject(columnKey);
			}
			return value;
		}

	}

	/**
	 * 打印调试信息
	 * <ul>
	 * <li>info 调试信息
	 * </ul>
	 */
	class OutputDebugImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			String info = TypeConvertor.toString(args[0]);
			return true;
		}

	}
	
	/**
	 * 获取服务器当前时间,精确到秒
	 * 
	 * @author 陈瑞
	 *
	 */
	class ServerDateImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			Date date = DateUtil.getNowTime();
			SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
			String val = format.format(date);
			return format.parse(val);
		}
		
	}
	
	/**
	 * 获取服务器数据库时间,精确到秒
	 * 
	 * @author 陈瑞
	 *
	 */
	class ServerDBDateImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor)
				throws Throwable {
			IDBManager manager = context.getDBManager();
			long time = manager.getCurTime();
			java.sql.Date date = new java.sql.Date(time);
			SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
			String val = format.format(date);
			return format.parse(val);
		}
		
	}
	
	
	/**
	 * 保存中间层数据，参数列表为[]
	 * <ul>
	 * <li>sysTopic 系统主题
	 * </ul>
	 * @author 王元和
	 *
	 */
	class SaveDataImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			MetaDataObject metaDataObject = context.getDataObject();
			Document document = context.getDocument();
			String sysTopic = "";
			if (args.length > 0){
				sysTopic = TypeConvertor.toString(args[0]);
			}
			if(sysTopic != null && sysTopic.length() > 0){
				context.setSysTopic(sysTopic);
			}
			SaveData save = new SaveData(metaDataObject, null, document);
			save.midLaunchSave(context);
			return true;
		}
		
	}
	
	/**
	 * 取得系统操作员ID
	 * @author wangyh
	 *
	 */
	class GetOperatorImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			return context.getVE().getEnv().getUserID();
		}
		
	}
	
	/**
	 * 取得用户操作员ID
	 * @author wangyh
	 *
	 */
	class GetUserOperatorImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor)
				throws Throwable {
			return context.getVE().getEnv().getGuestUserID();
		}
		
	}
	
	/**
	 * 取得表的数据行数，参数列表为[tableKey]
	 * <ul>
	 * <li>tableKey 表的列表，为数据源中定义的表的标识
	 * </ul>
	 * <p>如果tableKey定义的表存在，则返回其关联的数据表的行数，否则返回0
	 * </p>
	 * @author wangyh
	 *
	 */
	class GetRowCountImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			String tableKey = (String)args[0];
			Document document = context.getDocument();
			DataTable table = document.get(tableKey);
			return table != null ? table.size() : 0;
		}
		
	}
	
	/**
	 * 取得当前的数据库类型
	 * <p>取得当前的数据库类型
	 * </p>
	 *
	 */
	class GetDBTypeImpl extends BaseMidFunctionImpl {
		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			int dbType = context.getDBManager().getDBType();
			return DBType.toString(dbType);
		}
		
	}
	
	/**
	 * 取得数据对象的标识，无参数
	 * @author wangyh
	 *
	 */
	class GetOIDImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor)
				throws Throwable {
			long OID = -1;
			Document document = context.getDocument();
			if ( document != null ) {
				OID = document.getOID();
			}
			return OID;
		}
		
	}

	/**
	 * 分配一个新的序号，参数列表为[dataObjectKey, NoPrefix]
	 * <ul>
	 * <li>dataObjectKey 数据对象标识
	 * <li>NoPreifix 前缀
	 * </ul>
	 * 返回值：同NoPrefix关联的新的不重复序号
	 * @author wangyh
	 *
	 */
	class ApplyNewSequenceImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor) throws Throwable {
			String dataObjectKey = (String)args[0];
			String NoPrefix = (String)args[1];
			String tag = dataObjectKey;
			if (args.length > 2) {
				tag = (String) args[2];
			}
			
			int sequence = -1;
			IOIDAllocator allocator = OIDAllocatorFactory.getInstance().create(context);
			sequence = allocator.applyNewSequenceID(dataObjectKey, NoPrefix, tag);
			return sequence;

		}
		
	}
	
	/**
	 * 获取环境中的分数据簇ID
	 * 
	 * @author 陈瑞
	 *
	 */
	class GetClusterIDImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context,
				Object[] args, IExecutor executor)
				throws Throwable {			
			return context.getEnv().getClusterid();
		}
		
	}
	
	/**
	 * 载入指定版本号的数据
	 * <ul>
	 * <li>VERID 
	 * </ul>
	 * 返回值：无
	 * @author 刘翔翔
	 *
	 */
	class LoadHisImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			Integer VERID = TypeConvertor.toInteger(args[0]);
			HisLoad load=new HisLoad(context.getDataObject(), context.getDocument(), VERID);
			load.load(context);
			return true;
		}
		
	}
	
	/**
	 * 输出日志
	 * <ul>
	 * <li>Log 日志内容
	 * </ul>
	 * 
	 * @author 刘翔翔
	 *
	 */
	class LogInfoImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			LogSvr.getInstance().info(context.getEnv(), TypeConvertor.toString(args[0]));
			return true;
		}

	}
	/**
	 * 将迁移增量表中的数据合并到目标表中，原型：MigrationMerge(dataObjectKey);
	 * <ul>
	 * <li> dataObjectKey： 迁移目标表的数据对象标识
	 * </ul>
	 * @author wbh
	 *
	 */
	class MigrationMergeImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor)
				throws Throwable {
			String key = TypeConvertor.toString(args[0]);
			if (!StringUtil.isEmptyStr(key)) {
				MigrationMergeProxy proxy = new MigrationMergeProxy(key,false);
				proxy.process(context);
			}
			
			return true;
		}
		
	}

	static class GetStatusItemsImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor)
				throws Throwable {
			String formKey = "";

			if( args.length > 0 ) {
				formKey = TypeConvertor.toString(args[0]);
			} else {
				formKey = context.getFormKey();
			}

			IMetaFactory metaFactory = context.getVE().getMetaFactory();
			MetaForm metaForm = metaFactory.getMetaForm(formKey);

			MetaStatusCollection statusCollection = MetaUtil.getStatusCollection(metaFactory, metaForm);

			PairItemList list = new PairItemList();

			if(statusCollection != null){
                for (MetaStatus metaStatus : statusCollection) {
                    PairItem item = new PairItem(metaStatus.getValue(), metaStatus.getCaption());
                    list.add(item);
                }
			}
			return list;
		}

	}

	static class LocalDateImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			return DateUtil.getNowDate();
		}
	}

	static class GetLocaleImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			return context.getVE().getEnv().getLocale();
		}
	}

	static class GetFormCaptionImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			String formKey = context.getFormKey();
			MetaForm metaForm = context.getVE().getMetaFactory().getMetaForm(formKey);
			return metaForm.getCaption();
		}
	}

	static class GetFormAbbrCaptionImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			String formKey = context.getFormKey();
			MetaForm metaForm = context.getVE().getMetaFactory().getMetaForm(formKey);
			return metaForm.getAbbrCaption();
		}
	}

	static class ContainsKeyImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			String key = TypeConvertor.toString(args[0]);

			String formKey = context.getFormKey();
			MetaForm metaForm = context.getVE().getMetaFactory().getMetaForm(formKey);

			return metaForm.getAllUIComponents().containsKey(key);
		}
	}

	static class ApplyNewOIDImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			return context.applyNewOID();
		}
	}

	static class UUIDImpl extends BaseMidFunctionImpl{

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			return UUID.randomUUID();
		}
	}

	static class GetLangItemsImpl extends BaseMidFunctionImpl {

		@Override
		public Object evalImpl(String name, DefaultContext context, Object[] args, IExecutor executor) throws Throwable {
			MetaLangConfig metaLangConfig = MetaFactory.getGlobalInstance().getSolution().getLangConfig();
			String baseLang = TypeConvertor.toString(args.length == 1 ? args[0] : "");
			boolean hasAll = TypeConvertor.toBoolean(args.length == 2 ? args[1] : false);
			StringBuilder result = new StringBuilder();
			for (MetaLang metaLang : metaLangConfig) {
				if (baseLang != null && !baseLang.isEmpty()) {
					if (!hasAll && baseLang.equals(MetaLang.BaseLang_ZH) && metaLang.getKey().equals(MetaLang.BaseLang_EN)) continue;
					if (!metaLang.getBaseLang().equals(baseLang)) continue;
				}
				result.append(metaLang.getKey()).append(",").append(metaLang.getDescription()).append(";");
			}
			if (result.length() > 0) {
				result.deleteCharAt(result.length() - 1);
			}
			return result.toString();
		}
	}

	@Override
	public Object[][] getImplTable() {
		return new Object[][] {
				{ "SetTableValue", new SetTableValueImpl() },
				{ "GetTableValue", new GetTableValueImpl() },
				{ "OutputDebug", new OutputDebugImpl() },
				{ "SaveData", new SaveDataImpl() },
				{ "GetOperator",new GetOperatorImpl()},
				{ "GetRowCount", new GetRowCountImpl() },
				{ "GetDBType", new GetDBTypeImpl() },
				{ "ServerDate", new ServerDateImpl()},
				{ "ServerDBDate",new ServerDBDateImpl()},
				{ "GetOID", new GetOIDImpl() },
				{ "ApplyNewSequence", new ApplyNewSequenceImpl() },
				{ "GetClusterID", new GetClusterIDImpl() },
				{ "GetUserOperator", new GetUserOperatorImpl() },
				{ "LoadHis", new LoadHisImpl() },
				{ "LogInfo", new LogInfoImpl() },
				{ "LocalDate", new LocalDateImpl() },
				{ "MigrationMerge", new MigrationMergeImpl()},
				{ "GetStatusItems", new GetStatusItemsImpl()},
				{ "GetLocale", new GetLocaleImpl()},
				{ "GetFormCaption", new GetFormCaptionImpl()},
				{ "GetFormAbbrCaption", new GetFormAbbrCaptionImpl()},
				{ "ContainsKey", new ContainsKeyImpl()},
				{ "ApplyNewOID", new ApplyNewOIDImpl()},
				{ "UUID", new UUIDImpl()},
				{ "GetLangItems", new GetLangItemsImpl()},
		};
	}
}
