package com.bokesoft.yigo.meta.util;

import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yes.meta.i18n.StringTable;
import com.bokesoft.yes.meta.persist.dom.commondef.MetaDataElementFoldScanLoad;
import com.bokesoft.yes.meta.persist.dom.commondef.MetaDomainFoldScanLoad;
import com.bokesoft.yigo.common.def.*;
import com.bokesoft.yigo.common.struct.PairItem;
import com.bokesoft.yigo.common.struct.PairItemList;
import com.bokesoft.yigo.common.util.SimpleStringFormat;
import com.bokesoft.yigo.meta.base.AbstractMetaObject;
import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.base.MetaException;
import com.bokesoft.yigo.meta.common.MetaMacroCollection;
import com.bokesoft.yigo.meta.common.MetaScript;
import com.bokesoft.yigo.meta.common.MetaScriptCollection;
import com.bokesoft.yigo.meta.commondef.*;
import com.bokesoft.yigo.meta.dataelement.MetaDataElement;
import com.bokesoft.yigo.meta.dataelement.MetaDataElementCollection;
import com.bokesoft.yigo.meta.dataelement.MetaDataElementDef;
import com.bokesoft.yigo.meta.datamap.MetaMap;
import com.bokesoft.yigo.meta.datamap.target.MetaTargetField;
import com.bokesoft.yigo.meta.datamap.target.MetaTargetTable;
import com.bokesoft.yigo.meta.dataobject.*;
import com.bokesoft.yigo.meta.domain.MetaDomain;
import com.bokesoft.yigo.meta.domain.MetaDomainCollection;
import com.bokesoft.yigo.meta.domain.MetaDomainDef;
import com.bokesoft.yigo.meta.entry.MetaEntry;
import com.bokesoft.yigo.meta.entry.MetaEntryItem;
import com.bokesoft.yigo.meta.factory.GlobalI18N;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.IMetaResolverFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.form.MetaFormList;
import com.bokesoft.yigo.meta.form.MetaFormProfile;
import com.bokesoft.yigo.meta.form.MetaRightsProxyCollection;
import com.bokesoft.yigo.meta.form.component.MetaComponent;
import com.bokesoft.yigo.meta.form.component.control.editview.MetaEditViewColumn;
import com.bokesoft.yigo.meta.form.component.control.listview.MetaListViewColumn;
import com.bokesoft.yigo.meta.form.component.control.properties.MetaDictProperties;
import com.bokesoft.yigo.meta.form.component.control.properties.MetaUploadButtonProperties;
import com.bokesoft.yigo.meta.form.component.grid.MetaGridCell;
import com.bokesoft.yigo.meta.form.component.grid.MetaGridColumn;
import com.bokesoft.yigo.meta.form.component.grid.dynamic.MetaCellTypeDef;
import com.bokesoft.yigo.meta.form.component.grid.dynamic.MetaCellTypeTable;
import com.bokesoft.yigo.meta.intf.IMetaProject;
import com.bokesoft.yigo.meta.intf.IMetaSolution;
import com.bokesoft.yigo.meta.paratable.MetaParaGroup;
import com.bokesoft.yigo.meta.paratable.MetaParaTable;
import com.bokesoft.yigo.meta.setting.MetaSetting;
import com.bokesoft.yigo.meta.solution.MetaLang;
import com.bokesoft.yigo.meta.solution.MetaProject;
import com.bokesoft.yigo.meta.solution.MetaProjectCollection;
import com.bokesoft.yigo.meta.solution.MetaSolution;
import com.bokesoft.yigo.meta.strings.MetaStringTable;

import java.util.*;
import java.util.Map.Entry;

/**
 * 中间层配置对象工具类
 *
 * @author 陈瑞
 *
 */
public class MetaUtil {

	/**
	 * 获取参数组定义
	 * @param metaFactory 配置工厂
	 * @param groupKey 参数组标识
	 * @return 参数组定义
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaParaGroup getParaGroup(IMetaFactory metaFactory, String groupKey) throws Throwable {
		return  metaFactory.getParaTable().get(groupKey);
	}

	/**
	 * 获取参数组定义
	 * @param metaFactory 配置工厂
	 * @param solutionKey soltuion标识
	 * @return 参数组定义
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaParaTable getParaTable(IMetaFactory metaFactory, String solutionKey) throws Throwable {
		if (StringUtil.isBlankOrNull(solutionKey)) {
			solutionKey = metaFactory.getSolution().getKey();
		}
		
		MetaParaTable resultParaTable = new MetaParaTable();
		
		MetaParaTable paraTable = metaFactory.getParaTable();
		Iterator<Entry<String, MetaParaGroup>> itParaTable = paraTable.entryIterator();
		Entry<String, MetaParaGroup> entry = null;
		MetaParaGroup metaParaGroup = null;
		while (itParaTable.hasNext()) {
			entry = itParaTable.next();
			metaParaGroup = entry.getValue();
			if(metaParaGroup.getSolution().equals(solutionKey)) {
				resultParaTable.add(entry.getValue());
			}
		}
		return resultParaTable;
	}

	/**
	 * 获取引用的操作定义
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @param refKey 引用的操作标识
	 * @return 引用的操作定义
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaOperation getRefOperation(IMetaFactory metaFactory,MetaForm metaForm, String refKey) throws Throwable {
		MetaOperation metaOperation = null;

		// 取工程commonDef
		String projectKey = metaForm.getProject().getKey();
		MetaCommonDef commonDef = metaFactory.getCommonDef(projectKey);
		if (commonDef != null && commonDef.getOperationCollection() != null) {
			metaOperation = (MetaOperation) commonDef.getOperationCollection().get(refKey);
		}

		//取非主Solution中的commonDef
		if (metaOperation == null) {
			commonDef = metaFactory.getSolutionCommondDef(projectKey);
			if (commonDef != null && commonDef.getOperationCollection() != null) {
				metaOperation = (MetaOperation) commonDef.getOperationCollection().get(refKey);
			}
		}
		// 取最上层commonDef
		if (metaOperation == null) {
			commonDef = metaFactory.getCommonDef("");
			if (commonDef != null && commonDef.getOperationCollection() != null) {
				metaOperation = (MetaOperation) commonDef.getOperationCollection().get(refKey);
			}
		}
		return metaOperation;
	}

	/**
	 *  处理操作的引用,内部使用方法
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @param metaOperation 需要处理的操作
	 * @return 是否处理成功
	 * @throws Throwable
	 */
	@SuppressWarnings("deprecation")
	public static boolean processOperationRefKey(IMetaFactory metaFactory,MetaForm metaForm,MetaOperation metaOperation) throws Throwable {
		if( metaOperation.isRefKeyInited() ) {
			return true;
		}
		MetaOperation refMetaOperation = getRefOperation(metaFactory,metaForm, metaOperation.getRefKey());
		if (refMetaOperation != null) {
			metaOperation.setRefKeyInited(true);
			metaOperation.copyFrom(refMetaOperation);
			return true;
		}
		return false;
	}

	/**
	 * 获取状态项集合
	 * @param metaFactory 配置工厂
	 * @param metaForm 表单
	 * @return 状态项集合
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaStatusCollection getStatusCollection(IMetaFactory metaFactory,MetaForm metaForm) throws Throwable {
		MetaStatusCollection statusCollection = null;
		MetaDataObject dataObject = getDataObject(metaFactory,metaForm);
		if( dataObject == null ) {
			MetaProject project = (MetaProject) metaForm.getProject();
			return getStatusCollectionByProjectKey(metaFactory, project.getKey());
		}
		return getStatusCollection(metaFactory,dataObject);
	}

	/**
	 * 获取状态集合
	 * @param metaFactory 配置工厂
	 * @param dataObject 元数据对象
	 * @return 状态集合
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaStatusCollection getStatusCollection(IMetaFactory metaFactory,MetaDataObject dataObject) throws Throwable {
		if( dataObject == null )
			return null;
		MetaStatusCollection statusCollection = dataObject.getStatusCollection();
		if (statusCollection == null || statusCollection.size() == 0) {
			// 查询一下关联数据对象
			String relateObjectKey = dataObject.getRelateObjectKey();
			if ( relateObjectKey != null && !relateObjectKey.isEmpty() ) {
				MetaDataObject relateDataObject = metaFactory.getDataObject(relateObjectKey);
				if ( relateDataObject != null ) {
					statusCollection = relateDataObject.getStatusCollection();
				}
			}
			if ( statusCollection == null || statusCollection.size() == 0 ) {
				MetaProject project = (MetaProject) dataObject.getProject();
				statusCollection = getStatusCollectionByProjectKey(metaFactory, project.getKey());
			}
		}
		return statusCollection;
	}
	
	/**
	 * 通过工程标识，获取状态集合
	 * @param metaFactory 工程标识
	 * @param projectKey 如果工程在如果是非主Solution中，获取的状态集合是否需要合并主Solution和其它Solution的状态集合
	 * @return 状态集合
	 * @throws Throwable
	 */
	public static MetaStatusCollection getStatusCollectionByProjectKey(IMetaFactory metaFactory, String projectKey) throws Throwable {
		MetaStatusCollection statusCollection = metaFactory.getStatusCollectionByProjectKey(projectKey);
		// 保底方法，这里是考虑到上面调用的metaFactory.getStatusCollectionByProjectKey方法，其mergeOtherSolution参数有可能后续会传false，
		// 此时如果副Solution状态集合未定义，会导致取不到主Solution的状态集合，因此下面加一个保底方法
		if (statusCollection == null || statusCollection.size() == 0) {
			MetaCommonDef commonDef = metaFactory.getCommonDef("");
			if (commonDef != null) {
				statusCollection = commonDef.getStatusCollection();
			}
		}
		return statusCollection;
	}

	/**
	 * 获取状态集合,结果不为空
	 * @param metaFactory 配置工厂
	 * @param dataObject 元数据对象
	 * @param statusKey 状态标识
	 * @return 状态标识对应的状态
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaStatus getStatusNotNull(IMetaFactory metaFactory,MetaDataObject dataObject, String statusKey) throws Throwable {
		MetaStatusCollection statusCollection = getStatusCollection(metaFactory,dataObject);
		if (statusCollection != null) {
			MetaStatus item = statusCollection.get(statusKey);
			if (item != null)
				return item;
		}
		throw new MetaException(MetaException.NO_STATUS_KEY, SimpleStringFormat.format(StringTable.getString(null, "", StringTable.NoStatusKey), statusKey));
	}

	/**
	 * 获取commonDef定义的检查规则
	 * @param metaFactory 配置工厂
	 * @param dataObject 元数据对象
	 * @return 检查规则
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaCheckRuleCollection getMetaCheckRuleCollection(IMetaFactory metaFactory,MetaDataObject dataObject) throws Throwable {
		MetaCheckRuleCollection checkRuleCollection = null;
		MetaProject project = (MetaProject) dataObject.getProject();
		MetaCommonDef commonDef = metaFactory.getCommonDef(project.getKey());
		if (commonDef != null) {
			checkRuleCollection = commonDef.getCheckRuleCollection();
		}
		if (checkRuleCollection == null || checkRuleCollection.size() == 0) {
			commonDef = metaFactory.getSolutionCommondDef(project.getKey());
			if (commonDef != null) {
				checkRuleCollection = commonDef.getCheckRuleCollection();
			}
		}
		if (checkRuleCollection == null || checkRuleCollection.size() == 0) {
			commonDef = metaFactory.getCommonDef("");
			if (commonDef != null) {
				checkRuleCollection = commonDef.getCheckRuleCollection();
			}
		}
		return checkRuleCollection;
	}

	/**
	 * 获取数据对象
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @return 数据对象
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaDataObject getDataObject(IMetaFactory metaFactory,MetaForm metaForm) throws Throwable {
		MetaDataSource dataSource = metaForm.getDataSource();
		if(dataSource == null){
			return null;
		}
		String refObjectKey = dataSource.getRefObjectKey();
		MetaDataObject dataObject = dataSource.getDataObject();;
		if ( refObjectKey != null && !refObjectKey.isEmpty() ) {
			if (dataObject == null){
				dataObject = metaFactory.getDataObject(refObjectKey);
			}
		}
		return dataObject;
	}

	/**
	 * 通过Form类型获取Form列表
	 * @param metaFactory 配置工厂
	 * @param filter 过滤器
	 * @return 表单定义集合
	 * @throws Throwable 可能抛出的异常
	 */
	public static PairItemList getFormByType(IMetaFactory metaFactory,String filter) throws Throwable {
		MetaFormList formList = metaFactory.getMetaFormList();
		Iterator<MetaFormProfile> itFormProfile = formList.iterator();
		PairItemList itemList = new PairItemList();
		while (itFormProfile.hasNext()) {
			MetaFormProfile formProfile = itFormProfile.next();
			if (filter == null || filter.isEmpty()) {
				itemList.add(new PairItem(formProfile.getKey(), formProfile.getCaption()));
			} else if (StringUtil.instr(filter, FormType.toString(formProfile.getFormType()), ",")) {
				itemList.add(new PairItem(formProfile.getKey(), formProfile.getCaption()));
			}
		}
		return itemList;
	}

	/**
	 * 根据当前配置获取宏公式集合
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @return 宏公式集合
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaMacroCollection getMacroCollection(IMetaFactory metaFactory,MetaForm metaForm) throws Throwable{
		MetaMacroCollection macroCollection = metaForm.getMacroCollection();
		MetaProject metaProject = (MetaProject) metaForm.getProject();
		if (macroCollection == null || macroCollection.isEmpty()) {
			MetaCommonDef metaCommonDef = metaFactory.getCommonDef(metaProject.getKey());
			if (metaCommonDef != null) {
				macroCollection = metaCommonDef.getMacroCollection();
			}
		}
		if (macroCollection == null || macroCollection.isEmpty()) {
			MetaCommonDef metaCommonDef = metaFactory.getSolutionCommondDef(metaProject.getKey());
			if (metaCommonDef != null) {
				macroCollection = metaCommonDef.getMacroCollection();
			}
		}
		if (macroCollection == null || macroCollection.isEmpty()) {
			MetaCommonDef metaCommonDef = metaFactory.getCommonDef("");
			if (metaCommonDef != null) {
				macroCollection = metaCommonDef.getMacroCollection();
			}
		}
		return macroCollection;
	}

	/**
	 * 根据当前配置获取脚本集合
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @return 配置脚本集合
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaScriptCollection getScriptCollection(IMetaFactory metaFactory, MetaForm metaForm) throws Throwable{
		MetaScriptCollection scriptCollection = null;

		MetaScriptCollection metaScriptCollection = metaForm.getScriptCollection();

		if(metaScriptCollection != null) {
			scriptCollection = (MetaScriptCollection) metaScriptCollection.clone();
		}

		MetaProject metaProject = (MetaProject) metaForm.getProject();
		MetaCommonDef projectCommonDef = metaFactory.getCommonDef(metaProject.getKey());
		if (projectCommonDef != null) {
			MetaScriptCollection projectScriptCollection = projectCommonDef.getScriptCollection();
			scriptCollection = mergeScriptionCollection(scriptCollection, projectScriptCollection);
		}

		projectCommonDef = metaFactory.getSolutionCommondDef(metaProject.getKey());
		if (projectCommonDef != null) {
			MetaScriptCollection solutionScriptCollection = projectCommonDef.getScriptCollection();
			scriptCollection = mergeScriptionCollection(scriptCollection, solutionScriptCollection);
		}

		projectCommonDef = metaFactory.getCommonDef("");
		if (projectCommonDef != null) {
			MetaScriptCollection solutionScriptCollection = projectCommonDef.getScriptCollection();
			scriptCollection = mergeScriptionCollection(scriptCollection, solutionScriptCollection);
		}

		return scriptCollection;
	}

	private static MetaScriptCollection mergeScriptionCollection(MetaScriptCollection scriptCollection1, MetaScriptCollection scriptCollection2) {
		if(scriptCollection1 == null && scriptCollection2 != null) {
			scriptCollection1 = (MetaScriptCollection)scriptCollection2.clone();
			return scriptCollection1;
		}
		if(scriptCollection2 != null) {
			for(MetaScript metaScript : scriptCollection2) {
				if(scriptCollection1.containsKey(metaScript.getKey())) {
					continue;
				}
				if ( metaScript.getVerb() == FormVerb.Load ) {
					scriptCollection1.setLoad(metaScript);
				} else if ( metaScript.getVerb() == FormVerb.Save ) {
					scriptCollection1.setSave(metaScript);
				} else if ( metaScript.getVerb() == FormVerb.Show ) {
					scriptCollection1.setSave(metaScript);
				} else {
					scriptCollection1.add(metaScript);
				}
			}
		}
		return scriptCollection1;
	}

	/**
	 * 根据当前配置获取加载脚本
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @return 配置加载脚本
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaScript getLoadScript(IMetaFactory metaFactory,MetaForm metaForm) throws Throwable {
		MetaScriptCollection scriptCollection = metaForm.getScriptCollection();
		MetaScript load = null;
		if (scriptCollection != null) {
			load = scriptCollection.getLoad();
		}
		MetaProject metaProject = (MetaProject) metaForm.getProject();
		if (load == null) {
			MetaCommonDef projectCommonDef = metaFactory.getCommonDef(metaProject.getKey());
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					load = scriptCollection.getLoad();
				}
			}
		}
		if (load == null) {
			MetaCommonDef projectCommonDef = metaFactory.getSolutionCommondDef(metaProject.getKey());
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					load = scriptCollection.getLoad();
				}
			}
		}

		if (load == null) {
			MetaCommonDef projectCommonDef = metaFactory.getCommonDef("");
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					load = scriptCollection.getLoad();
				}
			}
		}
		return load;
	}

	/**
	 * 根据当前配置获取保存脚本
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @return 配置保存脚本
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaScript getSaveScript(IMetaFactory metaFactory,MetaForm metaForm) throws Throwable {
		MetaScriptCollection scriptCollection = metaForm.getScriptCollection();
		MetaScript save = null;
		if (scriptCollection != null) {
			save = scriptCollection.getSave();
		}
		MetaProject metaProject = (MetaProject) metaForm.getProject();
		if (save == null) {
			MetaCommonDef projectCommonDef = metaFactory.getCommonDef(metaProject.getKey());
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					save = scriptCollection.getSave();
				}
			}
		}
		if (save == null) {
			MetaCommonDef projectCommonDef = metaFactory.getSolutionCommondDef(metaProject.getKey());
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					save = scriptCollection.getSave();
				}
			}
		}
		if (save == null) {
			MetaCommonDef projectCommonDef = metaFactory.getCommonDef("");
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					save = scriptCollection.getSave();
				}
			}
		}
		return save;
	}

	/**
	 * 根据当前配置获取Show脚本
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @return 配置保存脚本
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaScript getShowScript(IMetaFactory metaFactory,MetaForm metaForm) throws Throwable {
		MetaScriptCollection scriptCollection = metaForm.getScriptCollection();
		MetaScript show = null;
		if (scriptCollection != null) {
			show = scriptCollection.getShow();
		}
		MetaProject metaProject = (MetaProject) metaForm.getProject();
		if (show == null) {
			MetaCommonDef projectCommonDef = metaFactory.getCommonDef(metaProject.getKey());
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					show = scriptCollection.getShow();
				}
			}
		}
		if (show == null) {
			MetaCommonDef projectCommonDef = metaFactory.getSolutionCommondDef(metaProject.getKey());
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					show = scriptCollection.getShow();
				}
			}
		}
		if (show == null) {
			MetaCommonDef projectCommonDef = metaFactory.getCommonDef("");
			if (projectCommonDef != null) {
				scriptCollection = projectCommonDef.getScriptCollection();
				if (scriptCollection != null) {
					show = scriptCollection.getShow();
				}
			}
		}
		return show;
	}

	/**
	 * 获取一个数据表对应的子表集合
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @param tableKey 表名
	 * @return 子表集合
	 * @throws Throwable 可能抛出的异常
	 */
	public static List<MetaTable> getChildrenTable(IMetaFactory metaFactory,MetaForm metaForm,String tableKey) throws Throwable {
		MetaDataObject metaDataObject = getDataObject(metaFactory,metaForm);

		MetaTableCollection metaTableCollection = null;
		if (metaDataObject != null) {
			metaTableCollection = metaDataObject.getTableCollection();
		}

		MetaTable metaTable = null;
		List<MetaTable> tables = new ArrayList<MetaTable>();
		if( metaTableCollection != null ) {
			Iterator<MetaTable> it = metaTableCollection.iterator();
			while( it.hasNext() ) {
				metaTable = it.next();
				if( metaTable.getParentKey() != null &&
						metaTable.getParentKey().equals(tableKey) )
					tables.add(metaTable);
			}
		}
		return tables;
	}

	/**
	 * 获取绑定的元数据表
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @param tableKey 表名
	 * @return 元数据表
	 * @throws Throwable 可能抛出的异常
	 */
	public static MetaTable getBindingMetaTable(IMetaFactory metaFactory,MetaForm metaForm,String tableKey) throws Throwable {

		MetaDataObject metaDataObject = getDataObject(metaFactory,metaForm);

		MetaTableCollection metaTableCollection = null;
		if (metaDataObject != null) {
			metaTableCollection = metaDataObject.getTableCollection();
		}

		MetaTable metaTable = null;
		if (tableKey != null && !tableKey.isEmpty() && metaTableCollection != null) {
			metaTable = metaTableCollection.get(tableKey);
		}

		return metaTable;
	}

	/**
	 * 获取数据表上的业务关键字定义
	 * @param metaFactory 配置工厂
	 * @param metaForm 元表单
	 * @param tableKey 数据表标识
	 * @return 业务关键字定义
	 * @throws Throwable 可能抛出的异常
	 */
	public static List<String> getPrimaryKeys(IMetaFactory metaFactory,MetaForm metaForm, String tableKey) throws Throwable {
		ArrayList<String> list = new ArrayList<String>();
		if(!tableKey.isEmpty()) {
			MetaTable metaTable = getBindingMetaTable(metaFactory, metaForm, tableKey);
			Iterator<MetaColumn> itColumn = metaTable.iterator();
			while(itColumn.hasNext()){
				MetaColumn metaColumn = itColumn.next();
				if(metaColumn.getIsPrimary()){
					list.add(metaColumn.getKey());
				}
			}
		}
		return list;
	}

	/**
	 * 获取权限代理集合
	 * @param metaFactory 配置工厂
	 * @param formKey 表单标识
	 * @return 权限代理集合
	 * @throws Throwable 处理异常
	 */
	public static MetaRightsProxyCollection getRightsProxyCollection(IMetaFactory metaFactory,String formKey) throws Throwable {
		MetaForm metaForm = metaFactory.getMetaForm(formKey);
		MetaRightsProxyCollection rightsProxyCollection = metaForm.getRightsProxyCollection();
		if( rightsProxyCollection != null ){
			return rightsProxyCollection;
		}

		MetaSetting metaSetting = metaFactory.getSetting();
		if ( metaSetting != null ) {
			rightsProxyCollection = metaSetting.getRightsProxyCollection();
		}

		return rightsProxyCollection;
	}

	public static String getLocalString(IMetaFactory metaFactory, MetaForm metaForm, String lang, String group, String word, String org, String s) throws Throwable{
		try {
			String key = group;

			MetaStringTable strings = null;
			try {
				strings = metaFactory.getMetaStringTable(metaForm.getKey(), lang);
			} catch (Exception e) {
			}
			if ( strings != null ) {
				s = strings.getString(lang, key, word);
			}
			
			MetaProject project = (MetaProject)metaForm.getProject();
			if ( s == null ) {
				if(project != null) {
					strings = project.getStrings();
					s = strings.getString(lang, key, word);
				}
				if ( s == null ) {
					strings = metaFactory.getStrings();
					s = strings.getString(lang, key, word);
				}
			}
			
			if ( s == null ) {
				String projectKey = project == null ? "" : project.getKey();
				s = getString(metaFactory, lang, projectKey, org);
			}
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
		return s;
	}

	/**
	 * project 或 solution 的i18n文件夹 中查找公共界面词条
	 * @param metaFactory 配置工厂
	 * @param lang 语种
	 * @param projectKey 工程标识
	 * @param group 组标识
	 * @param word 词条标识
	 * @param org 默认值
	 * @return 词文
	 * @throws Throwable 处理异常
	 */
	public static String getString(IMetaFactory metaFactory, String lang, String projectKey, String group, String word, String org) throws Throwable{
		if(!metaFactory.getSolution().isEnableMultiLang()){
			return org;
		}
		String s = null;
		try {
			String key = group;
			MetaStringTable strings = null;

			if ( projectKey != null && !projectKey.isEmpty() ) {
				MetaProject metaProject = metaFactory.getMetaProject(projectKey);
				if ( metaProject != null ) {
					strings = metaProject.getStrings();
					if (strings != null) {
						s = strings.getString(lang, key, word);
					}
				}
			}

			if ( s == null ) {
				strings = metaFactory.getStrings();
				s = strings.getString(lang, key, word);
			}

			if ( s == null ) {
				s = getString(metaFactory, lang, projectKey, org);
				if (s == null) {
					s = org;
				}
			}
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
		return s;
	}

	public static String getString(IMetaFactory metaFactory, String lang, String projectKey, String org) throws Throwable{
		if(!metaFactory.getSolution().isEnableMultiLang()){
			return org;
		}
		String s = null;
		try {
			MetaStringTable strings = null;
			if(projectKey != null && !projectKey.isEmpty()){
				MetaProject metaProject = metaFactory.getMetaProject(projectKey);
				
				//project找词条
				if(metaProject != null) {
					strings = metaProject.getStrings();
					s = strings.getString(lang, GroupKeyTable.STR_STRINGS, GroupKeyTable.STR_STRING + org);
				}
				//project所在的solution拿词条
				if ( s == null ) {
					GlobalI18N i18n = metaFactory.getI18N(projectKey);
					if(i18n != null){
						strings = i18n.getSolutionStrings();
						s = strings.getString(lang, GroupKeyTable.STR_STRINGS, GroupKeyTable.STR_STRING + org);
					}
				}
			}
			
			//主solution拿词条
			if ( s == null ) {
				strings = metaFactory.getI18N("").getSolutionStrings();
				s = strings.getString(lang, GroupKeyTable.STR_STRINGS, GroupKeyTable.STR_STRING + org);
			}

			if ( s == null ) {
				s = org;
			}
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
		return s;
	}
	

	/**
	 * form 或 project 或 solution 的i18n文件夹 中查找公共词条
	 * @param metaFactory 配置工厂
	 * @param lang 语种
	 * @param formKey 表单标识
	 * @param group 组标识
	 * @param word 词条标识
	 * @param org 默认值
	 * @return 词文
	 * @throws Throwable 处理异常
	 */
	public static String getFormLocaleString(IMetaFactory metaFactory, String lang, String formKey, String group, String word, String org) throws Throwable{
		if(!metaFactory.getSolution().isEnableMultiLang()){
			return org;
		}

		String s = null;
		MetaForm metaForm = metaFactory.getMetaForm(formKey);
		return MetaUtil.getLocalString(metaFactory, metaForm, lang, group, word, org, s);
	}


	/**
	 * 获取数据源词条
	 * @param metaFactory 配置工厂
	 * @param lang 语种
	 * @param dataObjectKey 数据对象标识
	 * @param org 默认值
	 * @return 词文
	 * @throws Throwable 处理异常
	 */
	public static String getDataObjectString(IMetaFactory metaFactory, String lang, String dataObjectKey, String org) throws Throwable{
		if(!metaFactory.getSolution().isEnableMultiLang()){
			return org;
		}

		MetaDataObject metaDataObj = metaFactory.getDataObject(dataObjectKey);
		if( metaDataObj == null ) {
			return org;
		}

		String s = null;
		try {
			MetaStringTable strings = null;

			String formKey = metaFactory.getDataObjectList().get(dataObjectKey).getFormKey();
			if (formKey == null || formKey.isEmpty()) {
				strings = metaFactory.getDataObjectStrings(dataObjectKey);
			} else {
				strings = metaFactory.getFormStrings(formKey);
			}

			if (strings != null) {
				s = strings.getString(lang, GroupKeyTable.STR_DATAOBJ, dataObjectKey);
			}
			
			IMetaProject imetaProject = metaDataObj.getProject();
			String projectKey = imetaProject != null ? imetaProject.getKey() : "";
			if (s == null) {
				//project中找词条
				if ( projectKey != null && !projectKey.isEmpty() ) {
					MetaProject metaProject = metaFactory.getMetaProject(projectKey);
					if ( metaProject != null ) {
						strings = metaProject.getStrings();
						s = strings.getString(lang, GroupKeyTable.STR_DATAOBJ, dataObjectKey);
					}
				}

				//solution中找词条
				if ( s == null ) {
					strings = metaFactory.getStrings();
					s = strings.getString(lang, GroupKeyTable.STR_DATAOBJ, dataObjectKey);
				}
			}

			if ( s == null ) {
				s = getString(metaFactory, lang, projectKey, org);
			}
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
		
		return s;
	}

	/**
	 * 获取数据源中列的词条
	 * @param metaFactory 配置工厂
	 * @param lang 语种
	 * @param dataObjectKey 数据对象标识
	 * @param tableKey 表标识
	 * @param colKey 列标识
	 * @param org 默认值
	 * @return 词文
	 * @throws Throwable 处理异常
	 */
	public static String getDataObjectString(IMetaFactory metaFactory, String lang, String dataObjectKey, String tableKey, String colKey, String org) throws Throwable{
		if(!metaFactory.getSolution().isEnableMultiLang()){
			return org;
		}

		MetaDataObject metaDataObj = metaFactory.getDataObject(dataObjectKey);
		if( metaDataObj == null ) {
			return org;
		}

		String s = null;
		try {
			MetaStringTable strings = null;

			// 从form 或 外置配置中取
			String formKey = metaFactory.getDataObjectList().get(dataObjectKey).getFormKey();
			if (formKey == null || formKey.isEmpty()) {
				strings = metaFactory.getDataObjectStrings(dataObjectKey);
			} else {
				strings = metaFactory.getFormStrings(formKey);
			}

			if (strings != null) {
				s = strings.getString(lang, GroupKeyTable.STR_DATAOBJ_COLUMN + "_"+ tableKey , colKey);
			}
			
			IMetaProject imetaProject = metaDataObj.getProject();
			String projectKey = imetaProject != null ? imetaProject.getKey() : "";
			if (s == null) {
				//project中找词条
				if ( projectKey != null && !projectKey.isEmpty() ) {
					MetaProject metaProject = metaFactory.getMetaProject(projectKey);
					if ( metaProject != null ) {
						strings = metaProject.getStrings();
						s = strings.getString(lang, GroupKeyTable.STR_DATAOBJ_COLUMN, colKey);
					}
				}

				//solution中找词条
				if ( s == null ) {
					strings = metaFactory.getStrings();
					s = strings.getString(lang, GroupKeyTable.STR_DATAOBJ_COLUMN, colKey);
				}
			}

			if ( s == null ) {
				s = getString(metaFactory, lang, projectKey, org);
			}
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
		return s;
	}

	/**
	 * 获取流程多语言
	 * @param metaFactory
	 * @param projectKey
	 * @param processKey
	 * @param lang
	 * @param group
	 * @param key
	 * @param org
	 * @return
	 * @throws Throwable
	 */
	public static String getProcessString(IMetaFactory metaFactory,String projectKey,String processKey,String lang, String group,String key, String org) throws Throwable{
		if(!metaFactory.getSolution().isEnableMultiLang()){
			return org;
		}
		
		String s = null;
		try {
			MetaStringTable strings = metaFactory.getProcessStrings(processKey);
			
			if ( strings != null ) {
				s = strings.getString(lang, group, key);
			}

			if ( s == null) {
				if(projectKey != null && projectKey.length() > 0){
					MetaProject metaProject = metaFactory.getMetaProject(projectKey);
					if ( metaProject != null ) {
						strings = metaProject.getStrings();
						s = strings.getString(lang, group, key);
					}
				}
				if ( s == null ) {
					strings = metaFactory.getStrings();
					s = strings.getString(lang, group, key);
				}
			}

			if ( s == null ) {
				s = getString(metaFactory, lang, projectKey, org);
			}
			
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
		
		return s;
	}

	/**
	 * 获取下推的字段绑定的组件key,在计算默认值时忽略
	 * @param metaFactory 配置工厂
	 * @param metaForm 目标表单配置
	 * @param mapKey 下推标识
	 * @return 下推的字段绑定的组件key集合
	 * @throws Throwable
	 */
	public static List<String> getMapComs(IMetaFactory metaFactory,MetaForm metaForm,String mapKey) throws Throwable {
		MetaMap metaMap = metaFactory.getDataMap(mapKey).getDataMapParas(metaFactory).getMeta();
		String dataObjectKey = metaMap.getTgtDataObjectKey();
		MetaDataObject metaObject = metaFactory.getDataObject(dataObjectKey);

		List<String> list = new ArrayList<String>();

		Iterator<MetaTargetTable> itTables = metaMap.getTargetTableCollection().iterator();
		while (itTables.hasNext()) {
			MetaTargetTable table = itTables.next();
			String tableKey = table.getKey();
			MetaTable metaTable = metaObject.getMetaTable(tableKey);
			Iterator<MetaTargetField> itFields = table.iterator();
			while (itFields.hasNext()) {
				MetaTargetField field = itFields.next();
				switch (metaTable.getTableMode()) {
					case TableMode.HEAD:
						MetaComponent com = metaForm.getComponentByDataBinding(tableKey, field.getDefinition());
						if (com != null) {
							list.add(com.getKey());
						}
						break;
					case TableMode.DETAIL:
						MetaGridCell metaCell = metaForm.getCellByDataBinding(tableKey, field.getDefinition());
						if (metaCell != null) {
							list.add(metaCell.getKey());
						}
						break;
					default:
						break;
				}
			}
		}
		return list;
	}

	/**
	 * 获取下推目标明细表绑定的表格标识,用于计算sequence等
	 * @param metaFactory 配置工厂
	 * @param metaForm 目标表单配置
	 * @param mapKey 下推标识
	 * @return 下推目标明细表绑定的表格集合
	 * @throws Throwable
	 */
	public static List<String> getMapGrids(IMetaFactory metaFactory,MetaForm metaForm,String mapKey) throws Throwable {
		MetaMap metaMap = metaFactory.getDataMap(mapKey).getDataMapParas(metaFactory).getMeta();
		String dataObjectKey = metaMap.getTgtDataObjectKey();
		MetaDataObject metaObject = metaFactory.getDataObject(dataObjectKey);

		List<String> keys = new ArrayList<String>();

		Iterator<MetaTargetTable> itTables = metaMap.getTargetTableCollection().iterator();

		MetaTargetTable tgtTable = null;
		MetaTable metaTable = null;
		while (itTables.hasNext()) {
			tgtTable = itTables.next();
			metaTable = metaObject.getMetaTable(tgtTable.getKey());

			if( metaTable.getTableMode() == TableMode.DETAIL ) {
				MetaComponent meta = metaForm.findComponentByTable(tgtTable.getKey());
				if( meta != null ) {
					keys.add(meta.getKey());
				}
			}
		}
		return keys;
	}

	/**
	 * 根据每个表的列定义获取对应的控件定义
	 * @param metaForm
	 * @param keysMap
	 * @return
	 * @throws Throwable
	 */
	public static List<String> getIgnoreKeys(MetaForm metaForm, Map<String, List<String>> keysMap) throws Throwable {
		ArrayList<String> list = new ArrayList<String>();
		MetaDataObject dataObject = metaForm.getDataSource().getDataObject();
		Iterator<String> itTables = keysMap.keySet().iterator();
		while (itTables.hasNext()) {
			String tableKey = itTables.next();
			MetaTable metaTable = dataObject.getMetaTable(tableKey);
			Iterator<String> itFields = keysMap.get(tableKey).iterator();
			while (itFields.hasNext()) {
				String fieldKey = itFields.next();
				switch (metaTable.getTableMode()) {
					case TableMode.HEAD:
						MetaComponent component = metaForm.getComponentByDataBinding(tableKey, fieldKey);
						if (component != null) {
							list.add(component.getKey());
						}
						break;
					case TableMode.DETAIL:
						MetaGridCell metaGridCell = metaForm.getCellByDataBinding(tableKey, fieldKey);
						if (metaGridCell != null) {
							list.add(metaGridCell.getKey());
						}
						break;
					default:
						break;
				}
			}
		}
		return list;
	}

	/**
	 * 根据数据对象获取忽略修改的列
	 * @param dataObject 数据对象
	 * @return 忽略修改的列
	 * @throws Throwable 异常
	 */
	public static Map<String, List<String>> getIgnoreModilfyColumns(MetaDataObject dataObject) throws Throwable {
		if(dataObject == null) {
			return null;
		}

		Map<String, List<String>> map = new HashMap<String, List<String>>();
		List<String> list = null;
		MetaTableCollection tableCollection = dataObject.getTableCollection();
		for(MetaTable metaTable : tableCollection) {
			Iterator<MetaColumn> it = metaTable.iterator();
			MetaColumn column = null;
			list = new ArrayList<String>();
			while(it.hasNext()){
				column = it.next();
				if(column.isIgnoreModify()){
					list.add(column.getKey());
				}
			}
			map.put(metaTable.getKey(), list);
		}

		return map;
	}

	/**
	 * 根据组件标识获取字典配置属性对象
	 * @param metaFactory 配置工厂
	 * @param formKey 表单Key
	 * @param fieldKey 字段Key(表头字段或单元格字段)
	 * @return 字典配置属性对象
	 * @throws Throwable
	 */
	public static MetaDictProperties getDictProperties(IMetaFactory metaFactory, String formKey, String fieldKey) throws Throwable {
		return getDictProperties(metaFactory, formKey, fieldKey, "");
	}

	/**
	 * 获取字典配置属性对象的完整方法
	 * @param metaFactory 配置工厂
	 * @param formKey 表单Key
	 * @param fieldKey 字段Key(表头字段或单元格字段)
	 * @param typeDefKey 动态单元格类型定义
	 * @return 字典配置属性对象
	 * @throws Throwable
	 */
	public static MetaDictProperties getDictProperties(IMetaFactory metaFactory, String formKey, String fieldKey, String typeDefKey) throws Throwable {
		MetaDictProperties properties = null;
		if(formKey != null && !formKey.isEmpty()){
			// 从组件中寻找
			if( fieldKey != null && !fieldKey.isEmpty() ) {
				MetaForm metaForm = metaFactory.getMetaForm(formKey);
				MetaComponent meta = metaForm.componentByKey(fieldKey);
				if( meta != null ){ // 如果头组件有找到的字典组件
					if(meta.getControlType() == ControlType.DICT || meta.getControlType() == ControlType.DYNAMICDICT ){
						properties = (MetaDictProperties) meta.getProperties();
					}
				}

				if( properties == null ) {
					MetaGridCell metaCell = metaForm.metaGridCellByKey(fieldKey);
					if( metaCell != null ) { //如果表格中有找到的字典单元格
						if(metaCell.getCellType() == ControlType.DICT || metaCell.getCellType() == ControlType.DYNAMICDICT) {
							properties = (MetaDictProperties) metaCell.getProperties();
						}
					}
				}
				if(properties == null){
					MetaListViewColumn metaColumn = metaForm.metaListViewColumnByKey(fieldKey);
					if(metaColumn != null){ //如果列表中有找到的字典列
						if(metaColumn.getColumnType() == ControlType.DICT || metaColumn.getColumnType() == ControlType.DYNAMICDICT){
							properties = (MetaDictProperties) metaColumn.getProperties();
						}
					}
				}
				if(properties == null){
					MetaEditViewColumn metaColumn = metaForm.metaEditViewColumnByKey(fieldKey);
					if(metaColumn != null){ //如果列表中有找到的字典列
						if(metaColumn.getColumnType() == ControlType.DICT || metaColumn.getColumnType() == ControlType.DYNAMICDICT){
							properties = (MetaDictProperties) metaColumn.getProperties();
						}
					}
				}
				if(properties == null){
					MetaComponent metaComp = metaForm.metaDetailComponentByKey(fieldKey);
					if(metaComp != null){ //如果TableView明细行中有找到的字典组件
						if(metaComp.getControlType() == ControlType.DICT || metaComp.getControlType() == ControlType.DYNAMICDICT){
							properties = (MetaDictProperties) metaComp.getProperties();
						}
					}
				}
			}

			// 从动态单元格定义中寻找
			if( properties == null && ( typeDefKey != null && !typeDefKey.isEmpty() ) ) {
				MetaCellTypeTable table = metaFactory.getCellTypeTable();
				MetaCellTypeDef def = table.get(formKey).get(typeDefKey);
				if( def.getType() == ControlType.DICT || def.getType() == ControlType.DYNAMICDICT ) {
					properties = (MetaDictProperties) def.getProperties();
				}
			}
		}

		return properties;
	}


	public static MetaUploadButtonProperties getUploadButtonProperties(IMetaFactory metaFactory, String formKey, String fieldKey) throws Throwable{
		MetaUploadButtonProperties properties = null;
		if(formKey != null && !formKey.isEmpty()){
			// 从组件中寻找
			if( fieldKey != null && !fieldKey.isEmpty() ) {
				MetaForm metaForm = metaFactory.getMetaForm(formKey);
				MetaComponent meta = metaForm.componentByKey(fieldKey);
				if( meta != null ){ // 如果头组件有找到的字典组件
					if(meta.getControlType() == ControlType.UPLOADBUTTON){
						properties = (MetaUploadButtonProperties) meta.getProperties();
					}
				}

				if( properties == null ) {
					MetaGridCell metaCell = metaForm.metaGridCellByKey(fieldKey);
					if( metaCell != null ) { //如果表格中有找到的字典单元格
						if(metaCell.getCellType() == ControlType.UPLOADBUTTON) {
							properties = (MetaUploadButtonProperties) metaCell.getProperties();
						}
					}
				}
			}
		}
		return properties;
	}
	
	public static MetaTable getChildMetaTable(MetaDataObject metaDataObject, String parentKey) {
		for (MetaTable metaTable : metaDataObject.getTableCollection()) {
			if (metaTable.getLevelID() > 2) {
				String pk = metaTable.getParentKey();
				if(pk.equals(parentKey)) {
					return metaTable;
				}
			}
		}
		return null;
	}
	
	/**
	 * 获取域
	 * @param metaFactory 工厂
	 * @param key 标识
	 * @return 域
	 * @throws Throwable 异常
	 */
	public static MetaDomain getDomain(IMetaFactory metaFactory, String key) throws Throwable{
		return metaFactory.getDomainDef().getDomain(key);
	}

	
	/**
	 * 获取数据元素
	 * @param metaFactory 工厂
	 * @param key 数据元素标识
	 * @return 数据元素
	 * @throws Throwable 异常
	 */
	public static MetaDataElement getDataElement(IMetaFactory metaFactory, String key) throws Throwable{
		return  metaFactory.getDataElementDef().getDataElement(key);
	}
	
	/**
	 * 获取域
	 * @param metaFactory 域
	 * @param solution solution标识
	 * @param projectKey project标识
	 * @param key 域标识
	 * @return 域
	 * @throws Throwable 异常
	 */
	public static MetaDomain getDomain(IMetaFactory metaFactory, String solution, String projectKey, String key) throws Throwable{
		MetaDomain metaDomain = null;
		MetaDomainDef domainDef = null;
		if(metaDomain != null) {
			return metaDomain;
		}
		
		String solutionKey;
		if(solution == null || solution.isEmpty()) {
			solutionKey = getSolutionKey(metaFactory, projectKey);
		}else {
			solutionKey = solution;
		}
		
		// 从project对应的solution中获取
		if(solutionKey != null && !solutionKey.isEmpty()) {
			domainDef = metaFactory.getDomainDef(solutionKey);	
			if(domainDef != null) {
				metaDomain = domainDef.getDomain(key);
			}
		}
		if(metaDomain != null) {
			return metaDomain;
		}
		// 从primary solution中获取
		domainDef = metaFactory.getDomainDef("");	
		if(domainDef != null) {
			metaDomain = domainDef.getDomain(key);
		}

		return metaDomain;
	}

	public static String getSolutionKey(IMetaFactory metaFactory, String project) throws Throwable {
		if(project == null || project.isEmpty()) {
			return "";
		}
		MetaProject metaProject = metaFactory.getMetaProject(project);
        IMetaSolution iMetaSolution = metaProject == null ? null : metaProject.getSolution();

        String solutionKey = "";
        if(iMetaSolution != null){
            solutionKey = ((MetaSolution) iMetaSolution).getKey();
        }
        return solutionKey;
    }
	
	/**
	 * 刷修域及相关配置
	 * @param metaFactory 工厂
	 * @param metaDomainDef 域
	 * @throws Throwable 异常
	 */
	public static void reloadMetaDomain(IMetaFactory metaFactory, MetaDomainDef metaDomainDef) throws Throwable{
		
		if(metaDomainDef == null) {
            return;
        }
		Collection<MetaSolution> solutions = metaFactory.getMetaSolutions();
		if(solutions == null) {
			return;
		}
		MetaDataElementDef newDataElementDef = new MetaDataElementDef();
		for(MetaSolution solution : solutions) {
			MetaProjectCollection metaProjectCollection = solution.getProjectCollection();	
			if(metaProjectCollection == null) {
				continue;
			}
			// 根据solutionKey 获取
			MetaDataElementDef metaDataElementDef  =  metaFactory.getDataElementDef(solution.getKey());
			MetaDataElementDef tmp = _reloadDataElement(metaFactory, metaDomainDef, metaDataElementDef);
			newDataElementDef.toMerge(tmp);
		}
		reloadMetaDataElement(metaFactory, newDataElementDef);
	}

	private static MetaDataElementDef _reloadDataElement(IMetaFactory metaFactory, MetaDomainDef metaDomainDef,
								  MetaDataElementDef metaDataElementDef) throws Throwable {
		MetaDataElementDef newDataElementDef = new MetaDataElementDef();
		MetaDataElementCollection collection = new MetaDataElementCollection();
		if(metaDomainDef == null || metaDataElementDef == null) {
			return null;
		}

		MetaDataElementCollection dataElementCollection = metaDataElementDef.getDataElementCollection();
		if(dataElementCollection == null) {
			return null;
		}

		MetaDomainCollection domainCollection = metaDomainDef.getDomainCollection();
		if(domainCollection == null) {
			return null;
		}


		Iterator<MetaDataElement> it2 = dataElementCollection.iterator();
		MetaDataElement item = null;
		MetaDomain newDomain = null;
		String domainKey = null;
		while(it2.hasNext()) {
			item = it2.next();
			if(item == null) {
				continue;
			}
			domainKey = item.getDomainKey();
			if(domainCollection.containsKey(domainKey)){
				newDomain = domainCollection.get(domainKey);
				// 更新域
				item.setDomain(newDomain);
				// 合并数据元素
				MetaAnnotationUtil.mergeProperty(item, newDomain);
				collection.add(item);
			}
		}

		if(collection.size()>0){
			newDataElementDef.setDataElementCollection(collection);
			return newDataElementDef;
		}
		return null;
	}

		/**
         * 重载数据元素及相关配置
         * @param metaFactory 工厂
         * @param dataElementDef 数据元素标识
         * @throws Throwable 异常
         */
	public static LinkedHashSet<AbstractMetaObject> reloadMetaDataElement(IMetaFactory metaFactory,
	MetaDataElementDef dataElementDef) throws Throwable{
		LinkedHashSet<AbstractMetaObject> objects = new LinkedHashSet<>();
		if(dataElementDef == null){
			return objects;
		}

		MetaDataElementCollection dataElementCollection =   dataElementDef.getDataElementCollection();
		  if(dataElementCollection == null){
			  return objects;
		  }
		  Iterator<MetaDataElement> it ;

		MetaFormList metaFormList = metaFactory.getMetaFormList();
		MetaFormProfile profile = null;
		MetaForm metaForm = null;
		MetaDataSource dataSource = null;
		MetaDataObject dataObject = null;
		MetaDataObjectProfile dataObjectProfile = null;
		boolean bExists = false;
		String dataObjectKey = null;
		String mergeToSourceMapKey = null;
		for(int i=0;i<metaFormList.size();i++){
			profile = metaFormList.get(i);
			if(profile==null){
				continue;
			}
			metaForm = profile.getForm();
			if(metaForm == null) {
				continue;
			}
			dataSource = metaForm.getDataSource();
		    if (dataSource == null) {
		    	continue;
		    }
			dataObject = dataSource.getDataObject();
			if(dataObject == null) {
				continue;
			}
			// 查看数据对象是否包含数据元素
			it = dataElementCollection.iterator();
			while(it.hasNext()) {
				MetaDataElement dataElement = it.next();
				if (dataElement == null) {
					continue;
				}
				// 判断是否包含数据元素
				bExists = checkDataElementKey(dataObject, dataElement.getKey());
				if(bExists) {
					objects.add(profile);
					String refObjectKey = dataSource.getRefObjectKey();
					dataObjectKey = dataObject.getKey();
					mergeToSourceMapKey = dataObject.getMergeToSourceMapKey();

					if(refObjectKey != null && !refObjectKey.isEmpty()) {
                        // 更新数据对象
						dataObjectProfile = metaFactory.getDataObjectList().get(dataObjectKey);
						if(dataObjectProfile != null) {
							// 缓存清空
						}
						if(mergeToSourceMapKey != null && !mergeToSourceMapKey.isEmpty()) {
							dataObjectProfile = metaFactory.getDataObjectList().get(mergeToSourceMapKey);
							if(dataObjectProfile != null) {
								// 缓存清空
							}
						}
					}else{
					}
					break;
				}
			}
		}
		return objects;
	}

	/**
	 * 检查当前表单是否包含DataElementKey
	 * @param dataObject 数据对象
	 * @param key 数据元素标识
	 * @return 是 or 否
	 */
	private static boolean checkDataElementKey(MetaDataObject dataObject, String key) {

		MetaTableCollection tableCollection =  dataObject.getTableCollection();
		if(tableCollection == null) {
			return false;
		}
		
		for (MetaTable table : tableCollection) {
			Iterator<MetaColumn> it = table.iterator();
			MetaColumn metaColumn = null;
			while (it.hasNext()) {
				metaColumn = it.next();
				String dataElementKey = metaColumn.getDataElementKey();
				if(dataElementKey != null && dataElementKey.equals(key)) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * 加载solution的数据元素
	 * @param metaFactory 工厂
	 * @param solutionKey 解决方案标识
	 * @return 数据元素
	 * @throws Throwable
	 */
	public static MetaDataElementDef loadDataElementDef(IMetaFactory metaFactory, String solutionKey) throws Throwable {
		IMetaResolverFactory resolverFactory = metaFactory.getMetaResolverFactoryBySolution(solutionKey);
		IMetaResolver resolver = resolverFactory.newMetaResolver("");
		return loadDataElementDef(metaFactory, resolver, solutionKey, null);     
	}
	public static MetaDataElementDef loadDataElementDef(IMetaFactory metaFactory, IMetaResolver resolver,
														String solutionKey, MetaProject metaProject,
														String filePath) throws Throwable {
		MetaDataElementFoldScanLoad loader = new MetaDataElementFoldScanLoad(resolver, metaProject, null, AppRunType.App);
		loader.loadMetaData(null, "", filePath, "");
		MetaDataElementDef dataElementDef = loader.getMetaDataElementDef();
		dataElementDef = processDataElementDef(metaFactory, dataElementDef, solutionKey);
		return dataElementDef;
	}

    /**
     * 根据MetaResolver 加载数据元素
     * @param metaFactory 工厂
     * @param resolver 配置读取接口
     * @param solutionKey 解决方案标识
     * @param metaProject 项目
     * @return 数据元素
     * @throws Throwable 异常
     */
	public static MetaDataElementDef loadDataElementDef(IMetaFactory metaFactory, IMetaResolver resolver, String solutionKey, MetaProject metaProject) throws Throwable {
		// dataElementDef
        MetaDataElementFoldScanLoad loader = new MetaDataElementFoldScanLoad(resolver, metaProject, null, AppRunType.App);
        loader.load();
        MetaDataElementDef dataElementDef = loader.getMetaDataElementDef();
		dataElementDef = processDataElementDef(metaFactory, dataElementDef, solutionKey);
		return dataElementDef;     
	}

	private static MetaDataElementDef processDataElementDef(IMetaFactory metaFactory, MetaDataElementDef dataElementDef,
													String solutionKey){
		if(dataElementDef != null) {
			dataElementDef.setSolutionKey(solutionKey);
			dataElementDef.doPostProcess(0, new MetaDataElementDefProcess(metaFactory, solutionKey));
		}
		return dataElementDef;
	}
	/**
	 * 根据solution加载域
	 * @param metaFactory 工厂
	 * @param solutionKey 解决方案标识
	 * @return 域
	 * @throws Throwable 异常
	 */
    public static MetaDomainDef loadDomainDef(IMetaFactory metaFactory, String solutionKey) throws Throwable{
		IMetaResolverFactory resolverFactory = metaFactory.getMetaResolverFactoryBySolution(solutionKey);
		IMetaResolver resolver = resolverFactory.newMetaResolver("");
    	return loadDomainDef(resolver, solutionKey, null);
	}
    
    /**
     * 根据MetaResolver 加载域
     * @param resolver 配置读取接口
     * @param solutionKey 解决方案标识
     * @param metaProject 项目
     * @return 域
     * @throws Throwable 异常
     */
    public static MetaDomainDef loadDomainDef(IMetaResolver resolver, String solutionKey, MetaProject metaProject) throws Throwable{
        //domainDef
        MetaDomainFoldScanLoad loader = new MetaDomainFoldScanLoad(resolver, metaProject, null, AppRunType.App);
        loader.load();
        MetaDomainDef domainDef = loader.getMetaDomainDef();
        if(domainDef != null) {
        	domainDef.setSolutionKey(solutionKey);
        }
		return domainDef;
	}

	public static MetaDomainDef loadDomainDef(IMetaFactory metaFactory, IMetaResolver resolver, String solutionKey,
											  MetaProject metaProject, String filePath) throws Throwable{
		//domainDef
		MetaDomainFoldScanLoad loader = new MetaDomainFoldScanLoad(resolver, metaProject, null, AppRunType.App);
		loader.loadMetaData(null,"", filePath, null);
		MetaDomainDef domainDef = loader.getMetaDomainDef();
		if(domainDef != null) {
			domainDef.setSolutionKey(solutionKey);
		}
		return domainDef;
	}

	/**
	 * 根据数据元素标识获取对应的字典标识
	 *
	 * @param metaFactory    配置工厂类
	 * @param dataElementKey 数据元素标识
	 * @return 字典标识
	 * @throws Throwable 异常
	 */
	public static String getItemKeyByDataElementKey(IMetaFactory metaFactory, String dataElementKey) throws Throwable {
		String itemKey = null;
		if (!StringUtil.isBlankOrNull(dataElementKey)) {
			MetaDataElement metaDataElement = getDataElement(metaFactory, dataElementKey);
			if (metaDataElement != null) {
				itemKey = metaDataElement.getDomain().getItemKey();
			}
		}
		return itemKey;
	}
}
