package com.bokesoft.yigo.tools.document;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.bokesoft.yes.common.struct.MultiKey;
import com.bokesoft.yes.common.struct.MultiKeyNode;
import com.bokesoft.yes.common.struct.report.MultiDimNode;
import com.bokesoft.yes.common.struct.report.MultiDimValue;
import com.bokesoft.yes.common.util.DBTypeUtil;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.dataobject.MetaColumn;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.struct.datatable.ColumnInfo;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.datatable.DataTableMetaData;

public class DataTableUtil {

	/** 缓存的DataTableMetaData数据 */
	public static Map<MetaTable, DataTableMetaData> cacheMetaDataMap = new HashMap<MetaTable, DataTableMetaData>();
	
	/**
	 * 根据配置对象创建一个DataTable数据对象（无数据）
	 * 
	 * @param metaTable 表定义
	 * @return 数据表
	 * @throws Throwable 处理异常
	 */
	public static DataTable newEmptyDataTable(MetaTable metaTable) {
		if (metaTable == null) {
			throw new RuntimeException("MetaTable不能为空！");
		}
		DataTableMetaData dataTableMetaData = cacheMetaDataMap.get(metaTable);
		// 如果传入过来的结构跟缓存的结构不一致，则清除缓存
		boolean isDiff = checkMetaTableColumnInfo(metaTable, dataTableMetaData);
		if (isDiff) {
			cacheMetaDataMap.remove(metaTable);
			dataTableMetaData = null;
		} else {
			DataTable dataTable = new DataTable(dataTableMetaData);
			dataTable.setKey(metaTable.getKey());
			dataTable.setCheckLength(metaTable.isPersist());
			return dataTable;
		}
		
		DataTable table = new DataTable();
		table.setKey(metaTable.getKey());
		table.setCheckLength(metaTable.isPersist());
		Iterator<MetaColumn> it = metaTable.iterator();
		MetaColumn metaColumn = null;
		int index = 0;
		while (it.hasNext()) {
			metaColumn = it.next();
			if (metaColumn.isHidden())
				continue;
			if (metaColumn.isSupportI18n())
				continue;

			String columnKey = metaColumn.getKey();
			ColumnInfo columnInfo = new ColumnInfo(columnKey, metaColumn.getDataType());
			columnInfo.setAccessControl(metaColumn.isAccessControl());
			columnInfo.setPrimary(metaColumn.getIsPrimary());
			columnInfo.setDefaultValue(metaColumn.getDefaultValue());
			columnInfo.setScale(metaColumn.getScale());
			columnInfo.setLength(metaColumn.getLength());
			columnInfo.setCodeColumnKey(metaColumn.getCodeColumnKey());
			columnInfo.setCheckLength(metaColumn.isPersist());
			columnInfo.setIgnoreChangeState(metaColumn.isIgnoreQuery() && !metaColumn.isPersist());
			if (metaColumn.isIgnoreQuery()) {
				table.addColumn(columnInfo);
				continue;
			} else {
				table.addColumn(index, columnInfo);
				index++;
			}
		}
		cacheMetaDataMap.put(metaTable, table.getMetaData());
		return table;
	}
	
	/**
	 * 检查metaTable列和缓存中是否一致
	 * @param metaTable
	 * @param dataTableMetaData
	 * @return
	 */
	private static boolean checkMetaTableColumnInfo(MetaTable metaTable, DataTableMetaData dataTableMetaData) {
		if (dataTableMetaData == null) {
			return true;
		}

		if (metaTable.size() != dataTableMetaData.getColumnCount()) {
			return true;
		}

		Iterator<MetaColumn> iterator = metaTable.iterator();
		while (iterator.hasNext()) {
			MetaColumn column = iterator.next();
			String columnKey = column.getKey();
			int columnIndex = dataTableMetaData.findColumnIndexByKey(columnKey);
			if (columnIndex < 0) {
				return true;
			}
		}

		return false;
	}
	
	/**
	 * 获取获取业务主键在数据表中的索引和数据类型
	 * 
	 * @param table 数据表
	 * @param columnKeys 列标志集合
	 * @param indexes 列序号集合
	 * @param types 类型集合
	 * @throws Throwable 列不存在的异常
	 */
	public static void getIndexesAndTypes(DataTable table, List<String> columnKeys, int[] indexes, int[] types) throws Throwable {
		DataTableMetaData tableMetaData = table.getMetaData();
		for ( int i = 0; i < columnKeys.size(); ++i ) {
			ColumnInfo columnInfo = tableMetaData.getColumnInfo(columnKeys.get(i));
			indexes[i] = tableMetaData.findColumnIndexByKey(columnInfo.getColumnKey());
			types[i] = columnInfo.getUserDataType();
		}
	}
	
	/**
	 * 根据维度值定位到数据表唯一的一行
	 * @param table 
	 * @param dimValue
	 * @return 如果定位到,返回数据表位置,否则返回-1
	 */
	public static int locate(DataTable table,MultiDimValue dimValue){
		MultiDimNode node = null;
		MultiDimValue other = null;
		table.beforeFirst();
		while( table.next() ) {
			other = new MultiDimValue();
			for( int i = 0,size = dimValue.size();i < size;i++ ) {
				node = dimValue.getValue(i);
				other.addValue(makeDimNode(table, node.getColumnKey()));
			}
			if( dimValue.equals(other) ) {
				return table.getBookmark();
			}
		}
		return -1;
	}
	
	/**
	 * 根据数据表当前行,生成一个维度值
	 * @param table 数据表
	 * @param columnKeys 维度标识
	 * @return 维度值
	 */
	public static MultiDimValue makeDimValue(DataTable table,List<String> columnKeys){
		MultiDimValue dimValue = new MultiDimValue();
		for( String columnKey : columnKeys ) {
			dimValue.addValue(makeDimNode(table, columnKey));
		}
		return dimValue;
	}
	
	private static MultiDimNode makeDimNode(DataTable table,String columnKey) {
		ColumnInfo info = table.getMetaData().getColumnInfo(columnKey);
		Object value = table.getObject(columnKey);
		MultiDimNode node = new MultiDimNode(columnKey, info.getDataType(), value);
		return node;
	}
	
	/**
	 * 根据给定的列索引和类型,构造出一个MultiKey
	 * 
	 * @param table 数据表
	 * @param indexes 列序号集合
	 * @param types 类型集合
	 * @return 多值主键
	 */
	public static MultiKey makeMultiKey(DataTable table, int[] indexes, int[] types) {
		MultiKey value = new MultiKey();
		int length = indexes.length;
		for ( int i = 0; i < length; ++i ) {
			value.addValue(new MultiKeyNode(types[i], table.getObject(indexes[i])));
		}
		return value;
	}
	
	/**
	 * 将维度值转换成MultiKey
	 * @param dimValue
	 * @return
	 */
	public static MultiKey convertDimValue(MultiDimValue dimValue){
		MultiKey key = new MultiKey();
		for( int i = 0,size = dimValue.size();i < size;i++ ) {
			MultiDimNode dimNode = dimValue.getValue(i);
			MultiKeyNode node = new MultiKeyNode(DBTypeUtil.dataType2JavaDataType(dimNode.getDataType()), dimNode.getValue());
			key.addValue(node);
		}
		return key;
	}
	
	/**
	 * 将源表的行附加到目标表中
	 * @param src 源表
	 * @param tgt 目标表
	 * @param metaTable 目标配置表
	 * @throws Throwable 
	 */
	public static void append(DataTable src, DataTable tgt, MetaTable metaTable) throws Throwable {
		append(src, tgt,metaTable, -1);
	}
	
	/**
	 * 将源表的行附加到目标表中,做类型转换
	 * @param src 源表
	 * @param tgt 目标表
	 * @param metaTable 目标配置表
	 * @param parentBkmk 父行标识
	 * @throws Throwable 
	 */
	public static void append(DataTable src, DataTable tgt, MetaTable metaTable, int parentBkmk) throws Throwable {
		DataTableMetaData srcMetaData = src.getMetaData();
		DataTableMetaData tgtMetaData = tgt.getMetaData();

		int colIndex = -1;
		ColumnInfo tgtCoInfo = null;
		
		Iterator<MetaColumn> it = null;
		String key = null;
		src.beforeFirst();
		while ( src.next() ) {
			DocumentUtil.newRow(metaTable, tgt);
			it = metaTable.iterator();
			while( it.hasNext() ){
				MetaColumn metaColumn =  it.next();
				if (metaColumn.isSupportI18n()) {
					continue;
				}
				key = metaColumn.getKey();
				colIndex = srcMetaData.findColumnIndexByKey(key);
				tgtCoInfo = tgtMetaData.getColumnInfo(key);
				if(tgtCoInfo != null  && colIndex >= 0){
					tgt.setObject(key, TypeConvertor.toDataType(tgtCoInfo.getDataType(), src.getObject(key)));
				}
			}
			if( parentBkmk != -1 ) {
				tgt.setParentBookmark(parentBkmk);
			}
		}
	}

	/**
	 * 合并数据,要求两个表格结构一致
	 * @param src 源表
	 * @param tgt 目标表
	 */
	public static void appendAll(DataTable src, DataTable tgt) {
		src.beforeFirst();
		int count = src.getMetaData().getColumnCount();
		while (src.next()) {
			tgt.append();
			tgt.setState(tgt.getState());
			for (int i = 0; i < count; i++) {
				tgt.setObject(i, src.getObject(i));
			}
		}
	}
	
	/**
	 * 类似于sql的join语句，生成一个新表。例如sql语句 “SELECT c1, t2.c2 c2_2, t1.c2, t2.c4 FROM t1
	 * JOIN t2 ON t1.c1 = t2.OID and t1.c3 = t2.c1” 可以替换为本接口的输入参数 (table1, table2,
	 * new String[]{"c1", "t2.c2 c2_2", "t1.c2", "t2.c4"}, new String[][]{{"c1",
	 * "OID"}, {"c3", "c1"}})
	 * 
	 * @param table1              第一个DataTable
	 * @param table2              第二个DataTable
	 * @param selectColumnKeys    选取table1和table2中哪些列的列标识数组。
	 *                            支持别名（用空格分隔，前面为列名，后面为别名）和”表名.列名“的格式
	 *                            参数为null代表新表取两张表的所有列，此时table2的列放在table1的列后面
	 * @param conditionColumnKeys 查询列名数组，只支持分别在两张表中的两列对应的值相等。
	 *                            数组中列名，不要带上表名。数组的长度为查询条件的数量；
	 *                            元素为长度为2的字符串数组，即[table1中的列名，table1中的列名]。
	 *                            注意列名不可使用selectColumnKeys中的别名
	 * @return 生成的新表
	 */
	public static DataTable join(DataTable table1, DataTable table2, String[] selectColumnKeys,
			String[][] conditionColumnKeys) {
		// 给较大的table建立索引
		int indexTableIndex, driverTableIndex;
		if (table2.size() >= table1.size()) {
			indexTableIndex = 1;
			driverTableIndex = 0;
		} else {
			indexTableIndex = 0;
			driverTableIndex = 1;
		}

		return baseJoin(table1, table2, selectColumnKeys, conditionColumnKeys, indexTableIndex, driverTableIndex,
				false);
	}

	/**
	 * 类似于sql的left join语句，生成一个新表。例如sql语句 “SELECT c1, t2.c2 c2_2, t1.c2, t2.c4 FROM
	 * t1 LEFT JOIN t2 ON t1.c1 = t2.OID and t1.c3 = t2.c1” 可以替换为本接口的输入参数 (table1,
	 * table2, new String[]{"c1", "t2.c2 c2_2", "t1.c2", "t2.c4"}, new
	 * String[][]{{"c1", "OID"}, {"c3", "c1"}})
	 * 
	 * @param table1              第一个DataTable
	 * @param table2              第二个DataTable
	 * @param selectColumnKeys    选取table1和table2中哪些列的列标识数组。
	 *                            支持别名（用空格分隔，前面为列名，后面为别名）和”表名.列名“的格式
	 *                            参数为null代表新表取两张表的所有列，此时table2的列放在table1的列后面
	 * @param conditionColumnKeys 查询列名数组，只支持分别在两张表中的两列对应的值相等。
	 *                            数组中列名，不要带上表名。数组的长度为查询条件的数量；
	 *                            元素为长度为2的字符串数组，即[table1中的列名，table1中的列名]。
	 *                            注意列名不可使用selectColumnKeys中的别名
	 * @return 生成的新表
	 */
	public static DataTable leftJoin(DataTable table1, DataTable table2, String[] selectColumnKeys,
			String[][] conditionColumnKeys) {
		// 给table2建立索引
		return baseJoin(table1, table2, selectColumnKeys, conditionColumnKeys, 1, 0, true);
	}

	/**
	 * 类似于sql的right join语句，生成一个新表。例如sql语句 “SELECT c1, t2.c2 c2_2, t1.c2, t2.c4 FROM
	 * t1 RIGHT JOIN t2 ON t1.c1 = t2.OID and t1.c3 = t2.c1” 可以替换为本接口的输入参数 (table1,
	 * table2, new String[]{"c1", "t2.c2 c2_2", "t1.c2", "t2.c4"}, new
	 * String[][]{{"c1", "OID"}, {"c3", "c1"}})
	 * 
	 * @param table1              第一个DataTable
	 * @param table2              第二个DataTable
	 * @param selectColumnKeys    选取table1和table2中哪些列的列标识数组。
	 *                            支持别名（用空格分隔，前面为列名，后面为别名）和”表名.列名“的格式
	 *                            参数为null代表新表取两张表的所有列，此时table2的列放在table1的列后面
	 * @param conditionColumnKeys 查询列名数组，只支持分别在两张表中的两列对应的值相等。
	 *                            数组中列名，不要带上表名。数组的长度为查询条件的数量；
	 *                            元素为长度为2的字符串数组，即[table1中的列名，table1中的列名]。
	 *                            注意列名不可使用selectColumnKeys中的别名
	 * @return 生成的新表
	 */
	public static DataTable rightJoin(DataTable table1, DataTable table2, String[] selectColumnKeys,
			String[][] conditionColumnKeys) {
		// 给table1建立索引
		return baseJoin(table1, table2, selectColumnKeys, conditionColumnKeys, 0, 1, true);
	}

	/**
	 * 类似于sql的join语句，生成一个新表的公共方法。
	 * 
	 * @param table1                 第一个DataTable
	 * @param table2                 第二个DataTable
	 * @param selectColumnKeys       选取table1和table2中哪些列的列标识数组。
	 *                               支持别名（用空格分隔，前面为列名，后面为别名）和”表名.列名“的格式
	 *                               参数为null代表新表取两张表的所有列，此时table2的列放在table1的列后面
	 * @param conditionColumnKeys    查询列名数组，只支持分别在两张表中的两列对应的值相等。
	 *                               数组中列名，不要带上表名。数组的长度为查询条件的数量；
	 *                               元素为长度为2的字符串数组，即[table1中的列名，table1中的列名]。
	 *                               注意列名不可使用selectColumnKeys中的别名
	 * @param indexTableIndex        索引表的序号，0为table1，1为table2
	 * @param driverTableIndex       驱动表的序号，0为table1，1为table2
	 * @param addDriverRowIfNotIndex 遍历驱动表的行时，如果索引表中没找到对应的行， 是否在新表中插入只包含驱动表数据的一行
	 * @return 生成的新表
	 */
	private static DataTable baseJoin(DataTable table1, DataTable table2, String[] selectColumnKeys,
			String[][] conditionColumnKeys, int indexTableIndex, int driverTableIndex, boolean addDriverRowIfNotIndex) {
		if (conditionColumnKeys == null || conditionColumnKeys.length == 0) {
			throw new RuntimeException("查询列名不可为空");
		}
		Object[] objectArray = mergeEmpty(table1, table2, selectColumnKeys);
		DataTable newTable = (DataTable) objectArray[0];
		int[][] indexArray = (int[][]) objectArray[1];
		HashMap<ObjectArray, List<Integer>> indexMap = new HashMap<>();
		DataTable indexTable = indexTableIndex == 0 ? table1 : table2;
		DataTable driverTable = driverTableIndex == 0 ? table1 : table2;
		indexTable.beforeFirst();
		while (indexTable.next()) {
			ObjectArray value = newObjectArray(indexTable, conditionColumnKeys, indexTableIndex);
			List<Integer> list = indexMap.get(value);
			if (list == null) {
				list = new ArrayList<>();
				indexMap.put(value, list);
			}
			list.add(indexTable.getPos());
		}

		driverTable.beforeFirst();
		while (driverTable.next()) {
			ObjectArray value = newObjectArray(driverTable, conditionColumnKeys, driverTableIndex);
			List<Integer> list = indexMap.get(value);
			if (list != null) {
				for (Integer pos : list) {
					indexTable.setPos(pos);
					newTable.append();
					for (int i = 0, size = indexArray.length; i < size; i++) {
						int[] childArray = indexArray[i];
						DataTable columnTable = childArray[0] == 0 ? table1 : table2;
						int columnIndex = childArray[1];
						newTable.setObject(i, columnTable.getObject(columnIndex));
					}
				}
			} else if (addDriverRowIfNotIndex) {
				newTable.append();
				for (int i = 0, size = indexArray.length; i < size; i++) {
					int[] childArray = indexArray[i];
					DataTable columnTable = childArray[0] == 0 ? table1 : table2;
					if (columnTable != driverTable) {
						continue;
					}
					int columnIndex = childArray[1];
					newTable.setObject(i, columnTable.getObject(columnIndex));
				}
			}
		}

		return newTable;
	}

	/**
	 * 根据查询列，新建{@link ObjectArray}
	 * 
	 * @param table               表
	 * @param conditionColumnKeys 查询列列名集合
	 * @param conditionIndex      查询位置
	 * @return 生成的{@link ObjectArray}
	 */
	private static ObjectArray newObjectArray(DataTable table, String[][] conditionColumnKeys, int conditionIndex) {
		Object[] valueArray = new Object[conditionColumnKeys.length];
		for (int i = 0, size = valueArray.length; i < size; i++) {
			valueArray[i] = table.getObject(conditionColumnKeys[i][conditionIndex]);
		}
		return new ObjectArray(valueArray);
	}

	/**
	 * 可作为{@link HashMap}的key的{@link Object}数组
	 */
	public static class ObjectArray {

		/**
		 * 数组值
		 */
		private final Object[] value;

		private ObjectArray(Object[] value) {
			this.value = value;
		}

		@Override
		public boolean equals(Object o) {
			if (!(o instanceof ObjectArray)) {
				return false;
			}
			return Arrays.deepEquals(value, ((ObjectArray) o).value);
		}

		@Override
		public int hashCode() {
			return Arrays.deepHashCode(value);
		}

		@Override
		public String toString() {
			return Arrays.toString(value);
		}
	}

	/**
	 * 生成一个包含两个DataTable的表结构的新表，不包含表数据
	 * 
	 * @param table1           第一个DataTable
	 * @param table2           第二个DataTable
	 * @param selectColumnKeys 选取table1中哪些列的列标识数组。
	 *                         支持别名（用空格分隔，前面为列名，后面为别名）和”表名.列名“的格式
	 *                         参数为null代表新表取两张表的所有列，此时table2的列放在table1的列后面
	 * @return [合并后的新DataTable,合并后列在合并前的位置数组]。
	 *         位置数组的序号，代表列在合并后的表中的列序号，也就是在selectColumnKeys中的序号；
	 *         位置数组的元素，代表[列在合并前表的序号，列在合并前表中的列序号]。
	 *         上述所有序号从0开始计算。例如某列在合并前，在table1中列序号为3，合并后的表中列序号为5， 则indexArray[5]=new
	 *         int[2]{0, 3}
	 */
	private static Object[] mergeEmpty(DataTable table1, DataTable table2, String[] selectColumnKeys) {
		DataTable newTable = new DataTable();
		int[][] indexArray;
		DataTableMetaData metaData1 = table1.getMetaData();
		DataTableMetaData metaData2 = table2.getMetaData();
		if (selectColumnKeys == null) {
			int columnCount1 = metaData1.getColumnCount();
			int columnCount2 = metaData2.getColumnCount();
			indexArray = new int[columnCount1 + columnCount2][2];
			for (int i = 0; i < columnCount1; i++) {
				ColumnInfo columnInfo = metaData1.getColumnInfo(i);
				if (metaData2.constains(columnInfo.getColumnKey())) {
					throw new RuntimeException("查询列名" + columnInfo.getColumnKey() + "在两张表中同时存在");
				}
				newTable.addColumn(columnInfo.deepClone());
				indexArray[i] = new int[2];
				indexArray[i][0] = 0;
				indexArray[i][1] = i;
			}
			for (int i = 0; i < columnCount2; i++) {
				ColumnInfo columnInfo = metaData2.getColumnInfo(i);
				newTable.addColumn(columnInfo.deepClone());
				int index = i + columnCount1;
				indexArray[index] = new int[2];
				indexArray[index][0] = 1;
				indexArray[index][1] = i;
			}
		} else {
			boolean tableKeyEqual = Objects.equals(table1.getKey(), table2.getKey());
			indexArray = new int[selectColumnKeys.length][2];
			for (int i = 0, size = selectColumnKeys.length; i < size; i++) {
				String columnKey = selectColumnKeys[i].trim();
				indexArray[i] = new int[2];
				ColumnInfo columnInfo = null;
				String[] columnKeyArray = columnKey.split(" +", 2);
				String aliasKey = null, tableKey = null;
				if (columnKeyArray.length >= 2) {
					columnKey = columnKeyArray[0];
					aliasKey = columnKeyArray[1];
				}
				columnKeyArray = columnKey.split("\\.", 2);
				if (columnKeyArray.length >= 2) {
					columnKey = columnKeyArray[1];
					tableKey = columnKeyArray[0];
				}

				if (tableKey != null) {
					if (tableKey.equals(table1.getKey())) {
						if (tableKeyEqual && metaData2.constains(columnKey)) {
							throw new RuntimeException("查询列名" + columnKey + "在两张表中同时存在");
						}
						int columnIndex = metaData1.findColumnIndexByKey(columnKey, true);
						columnInfo = metaData1.getColumnInfo(columnIndex);
						indexArray[i][0] = 0;
						indexArray[i][1] = columnIndex;
					} else if (tableKey.equals(table2.getKey())) {
						int columnIndex = metaData2.findColumnIndexByKey(columnKey, true);
						columnInfo = metaData2.getColumnInfo(columnIndex);
						indexArray[i][0] = 1;
						indexArray[i][1] = columnIndex;
					} else {
						throw new RuntimeException("找不到对应的表" + tableKey);
					}
				} else {
					int columnIndex = metaData1.findColumnIndexByKey(columnKey);
					if (columnIndex >= 0) {
						if (metaData2.constains(columnKey)) {
							throw new RuntimeException("查询列名" + columnKey + "在两张表中同时存在");
						}
						columnInfo = metaData1.getColumnInfo(columnIndex);
						indexArray[i][0] = 0;
					} else {
						columnIndex = metaData2.findColumnIndexByKey(columnKey, true);
						columnInfo = metaData2.getColumnInfo(columnIndex);
						indexArray[i][0] = 1;
					}
					indexArray[i][1] = columnIndex;
				}

				columnInfo = columnInfo.deepClone();
				if (aliasKey != null) {
					columnInfo.setColumnKey(aliasKey);
				}
				newTable.addColumn(columnInfo);
			}
		}

		return new Object[] { newTable, indexArray };
	}
}
