package com.bokesoft.yes.design.utils;

import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;

/**
 * 与项目使用的 {@link IMetaResolver} 相关的工具方法, 主要涉及使用 MetaResolver 方式的资源读写
 */
public class ProjectMetaResolverUtils {
    private static final Logger log = LoggerFactory.getLogger(ProjectMetaResolverUtils.class);

	private static IMetaResolver getProjectMetaResolver(String projectKey) {
		IMetaFactory metaFactory = MetaFactory.getGlobalInstance();
		IMetaResolver pResolver = metaFactory.getProjectResolver(projectKey);
		if (null==pResolver){
			throw new IllegalArgumentException("Project 'projectKey' 无效: 找不到相应的 MetaResolver");
		}
		return pResolver;
	}

    /**
     * 尝试获取项目资源的文件路径.
     * <p>FIXME IMetaResolver 接口各个方法的行为定义不够明确, 导致基本上没有完善的办法获取资源的路径</p>
     * @param projectKey
     * @param resourcePath
     * @return 如果对应项目的 {@link IMetaResolver} 不支持 {@link IMetaResolver#getPath(String)}, 返回 null
     */
	public static String tryGetProjectFilePath(String projectKey, String resourcePath){
		IMetaResolver pResolver = getProjectMetaResolver(projectKey);

        try{
            return pResolver.getPath(resourcePath);
        }catch(Exception ex){
            log.warn("Project '"+projectKey+"' 获取资源文件 '"+resourcePath+"' 路径出错, 将返回 null", ex);
            return null;
        }
	}

    /**
     * 读取指定的项目资源
     * @param projectKey
     * @param resourcePath
     * @return
     */
	public static String readProjectFile(String projectKey, String resourcePath) {
		IMetaResolver pResolver = getProjectMetaResolver(projectKey);

		try (InputStream ins = pResolver.read(resourcePath, 0)){
            if (null==ins){
                log.info("在 Project '"+projectKey+"' 中读取 '"+resourcePath+"' 返回 null, 原因: 获取的 InputStream 为空");
                return null;
            }
			String result = IOUtils.toString(ins, "UTF-8");
			return result;
		}catch(Exception ex){
			log.error("在 Project '"+projectKey+"' 中读取 '"+resourcePath+"' 返回 null, 原因: "+ex.getMessage(), ex);
			return null;
		}
	}

    /**
     * 将字符串内容写入指定的项目资源
     * @param projectKey
     * @param resourcePath
     * @param content
     * @throws Exception
     */
	public static void writeProjectFile(String projectKey, String resourcePath, String content) throws Exception {
		IMetaResolver pResolver = getProjectMetaResolver(projectKey);

		pResolver.write(resourcePath, content.getBytes("UTF-8"));
	}
}
