package com.bokesoft.distro.tech.yigosupport.extension.system.request;

import com.bokesoft.yigo.common.trace.TraceRecorder;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * 系统请求持有者类，用于存储和管理当前活动的HTTP请求
 * 通过TraceUtil工具类获取追踪ID来关联请求，支持并发操作
 */
public class SystemRequestHolder {

    // 存储当前活动的HTTP请求，键为追踪ID，值为HttpServletRequest对象
    private static final Map<String, HttpServletRequest> aliveRequests = new ConcurrentHashMap<>();

    /**
     * 获取当前活动的HTTP请求映射
     *
     * @return 包含当前活动HTTP请求的映射
     */
    public static Map<String, HttpServletRequest> getAliveRequests() {
        return aliveRequests;
    }

    /**
     * 添加当前HTTP请求到映射中，使用TraceUtil生成的追踪ID作为键
     * HttpServletRequest,地址引用寄存
     * @param request 当前的HTTP请求对象
     */
    public static void addCurrentRequest(HttpServletRequest request) {
        String traceId = TraceRecorder.getTraceId();
        if(StringUtils.isNotBlank(traceId)) {
            aliveRequests.remove(traceId);
        }
    }

    /**
     * 从映射中移除当前HTTP请求，使用TraceUtil生成的追踪ID作为键
     */
    public static void removeCurrentRequest() {
        String traceId = TraceRecorder.getTraceId();
        if(StringUtils.isNotBlank(traceId)) {
            aliveRequests.remove(traceId);
        }
    }

    /**
     * 获取当前活动的HTTP请求的数量
     *
     * @return 当前活动HTTP请求的数量
     */
    public static int getAliveRequestCount() {
        return aliveRequests.size();
    }

    /**
     * 根据追踪ID获取特定的HTTP请求
     *
     * @param traceId 追踪ID
     * @return 对应追踪ID的HTTP请求对象，如果不存在则返回null
     */
    public static HttpServletRequest getRequest(String traceId) {
        return aliveRequests.get(traceId);
    }

    /**
     * 获取当前活动的HTTP请求，使用TraceUtil生成的追踪ID作为键
     *
     * @return 当前活动的HTTP请求对象，如果不存在则返回null
     */
    public static HttpServletRequest getCurrentRequest() {
        return getRequest(TraceRecorder.getTraceId());
    }

}