package com.bokesoft.distro.tech.yigosupport.extension;

import java.util.ServiceLoader;

import com.bokesoft.distro.tech.yigosupport.extension.impl.BaseStaticWrapperMidFuncImplCluster;
import com.bokesoft.distro.tech.yigosupport.extension.intf.IFunctionProviderService;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.parser.IFunImplCluster;
import com.bokesoft.yigo.parser.IFunctionProvider;

/**
 * 将指定的类中的静态方法包装成 Yigo 公式的通用 {@link IFunctionProviderService} 实现基类;
 * <hr/>
 * 基本规则如下:
 * <ol>
 *  <li>直接继承当前抽象类 {@link BaseStaticWrapperMidFuncRegistryService} 即可定义一组公式,
 *      这组公式实际上是由 {@link #getWrappers()} 返回的多个 Java 类中的静态方法实现的 —— 这组
 *      Class 中的静态方法都可以被包装成 Yigo 中间层公式.</li>
 *  <li>每个公式具有一个 "prefix", 公式可以使用 <code>"prefix"."类名"."方法名"</code> 调用,
 *      也可以使用不包含"."的 <code>"prefix""类名""方法名"</code> 调用(称为 "narrow"), 或者
 *      使用下划线("_")代替"."(主要是因为在 Yigo 中, 对于使用"."分割的公式名称，不会去检查公式
 *      是否存在，使用"_"可以确保公式不存在的情况下系统能够报错) ; <br/>
 *      例如 <code>cms.PageExp.Var(...)</code> 还可以写成 <code>cmsPageExpVar(...)</code>,
 *      <code>cms_PageExp_Var(...)</code> 等. <br/>
 *      注意: 公式名称中包含类名是因为 {@link #getBuildOptions()} 默认实现所确定的, 具体实现时,
 *      子类通过重载此方法可以改变产生 Yigo 公式的默认行为;
 *  </li>
 *  <li>公式的参数支持通过两种方式直接传递到相应的静态方法, 1)直接逐个对应到方法的参数；2)如果该方法
 *      的第一个参数是 {@link DefaultContext} 类型, 那么公式的参数将对应该方法的第二个以后的各个
 *      参数, 第一个参数将传入当前系统的上下文({@link DefaultContext}).</li>
 * </ol>
 * <hr/>
 * 补充和注意事项:
 * <ol>
 *  <li>支持方法重载.</li>
 *  <li>注意 <code>"prefix"."类名"."方法名"</code> 中的 "类名" 并不包括 package name, 所以,
 *      公式与实际的 Class 全名并不会完全一致; 比如 <code>yigoext.EncryptExp.Encrypt(...)</code>
 *      对应的 class 可能是
 *      <code>com.bokesoft.distro.tech.yigosupport.extension.exttools.ExtFuncRegistryService.EncryptExp</code>.</li>
 *  <li>在 Yigo 2.0 以上版本中使用时, 本类的子类实现 需要依赖 {@link ServiceLoaderMidFunctionProvider},
 *      通过 Java 的 {@link ServiceLoader} 机制实现公式加载，因此需要手工在 solution 的 `Enhance.xml` 中
 *      注册, 具体注册方法参考 {@link ServiceLoaderMidFunctionProvider} 的说明.</li>
 *  <li><code>/META-INF/com.bokesoft.distro.tech.yigosupport.extension.intf.IFunctionProviderService</code> 文件:
 *      Java 的 {@link ServiceLoader} 机制要求在此文件中写入本类的子类实现的 className .</li>
 *  <li>静态方法实现建议: 为了符合 Yigo "公式区分大小写" 的惯例, 以及方便区分出 "类名" 和 "方法名",
 *      建议用于公式的静态类的静态方法的方法名采用 "首字母大写" 的规则.</li>
 * </ol>
 */
public abstract class BaseStaticWrapperMidFuncRegistryService implements IFunctionProviderService{
	/**
	 * 获取Yigo公式自定义前缀
	 * @return 获取Yigo公式自定义前缀
	 */
	protected abstract String getFormulaPrefix();

	/**
	 * 获取对应扩展开发处理类的类名集合
	 * @return 对应扩展开发处理类的类名集合
	 */
	protected abstract Class<?>[] getWrappers();

	/**
	 * 获取Yigo公式的构建选项. 默认选项为 {@link FunctionBuildOptions#createDefault()} .
	 * @return
	 */
	protected FunctionBuildOptions getBuildOptions(){
		return FunctionBuildOptions.createDefault();
	}
	
	@Override
	public IFunctionProvider getFunctionProvider() {
		return new IFunctionProvider() {
			@Override
			public IFunImplCluster[] getClusters() {
				return new IFunImplCluster[] {
					new BaseStaticWrapperMidFuncImplCluster(
							BaseStaticWrapperMidFuncRegistryService.this.getFormulaPrefix(),
							BaseStaticWrapperMidFuncRegistryService.this.getWrappers(),
							BaseStaticWrapperMidFuncRegistryService.this.getBuildOptions())
				};
			}
		};
	}

	/**
	 * 构建 Yigo 公式时的选项
	 */
	public static class FunctionBuildOptions {
		/** 是否在公式中包含类名(Class 的 Short Name) */
		private boolean withClassName;

		/**
		 * 创建默认的 Yigo 公式选项
		 * @return
		 */
		public static FunctionBuildOptions createDefault(){
			FunctionBuildOptions options = new FunctionBuildOptions();
			return options.withClassName(true);
		}

		/**
		 * 设置是否在公式中包含类名(参考 {@link #withClassName})
		 * @param withClassName
		 * @return
		 */
		public FunctionBuildOptions withClassName(boolean withClassName){
			this.setWithClassName(withClassName);
			return this;
		}

		public boolean isWithClassName() {
			return withClassName;
		}
		public void setWithClassName(boolean withClassName) {
			this.withClassName = withClassName;
		}
	}
}
