package com.bokesoft.distro.tech.commons.basis.jdbc.mssql;

import java.util.List;

import org.apache.commons.lang3.StringUtils;

import com.bokesoft.distro.tech.commons.basis.MiscUtil;

import net.sf.jsqlparser.JSQLParserException;
import net.sf.jsqlparser.expression.Alias;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.statement.Statement;
import net.sf.jsqlparser.statement.select.OrderByElement;
import net.sf.jsqlparser.statement.select.PlainSelect;
import net.sf.jsqlparser.statement.select.Select;
import net.sf.jsqlparser.statement.select.SelectBody;
import net.sf.jsqlparser.statement.select.SelectExpressionItem;
import net.sf.jsqlparser.statement.select.SetOperationList;
import net.sf.jsqlparser.statement.select.WithItem;

/**
 * SQL Server 特有的一些 SQL 处理
 */
public class SQLServerUtils {

	public static class SQLParserResult{
		private Select selectStmt;
		private List<OrderByElement> orderBy;
	
		public SQLParserResult(Select selectStmt, List<OrderByElement> orderBy){
			this.selectStmt = selectStmt;
			this.orderBy = orderBy;
		}
		public Select getSelectStmt() {
			return selectStmt;
		}
		public List<OrderByElement> getOrderBy() {
			return orderBy;
		}
		public String getSelectSql(){
			return selectStmt.toString();
		}
		public String getOrderBySql(){
			return PlainSelect.orderByToString(orderBy);
		}
	}

	/**
	 * 将 Order By 部分从 SELECT 语句中分离出来, 如果语句中有 TOP 子句, 那么不清除 Order By 部分, 否则清除 Order By 部分 
	 * @param sql
	 * @return
	 */
	protected static SQLParserResult splitOuterOrderBy(String sql){
		try {
			Statement stmt = CCJSqlParserUtil.parse(sql);
			MiscUtil.$assert(!(stmt instanceof Select), "Unsupported SQL Statement(MUST be SELECT statement): \n"+sql);
			Select selStmt = (Select) stmt;
			SelectBody body = selStmt.getSelectBody();
			if (body instanceof PlainSelect){
				PlainSelect ps = (PlainSelect) body;
				List<OrderByElement> orderBy = ps.getOrderByElements();
				if (null==ps.getTop()){
					ps.setOrderByElements(null);
				}
				return new SQLParserResult(selStmt, orderBy);
			}else if (body instanceof SetOperationList){
				throw new UnsupportedOperationException("UNION,INTERSECT,MINUS,EXCEPT is not supported, use it as subquery: \n"+sql);
			}else if (body instanceof WithItem){
				throw new UnsupportedOperationException("WITH clause MUST be a part of a SELECT statement: \n"+sql);
			}else{
				throw new UnsupportedOperationException("Unknown SELECT SQL: \n"+sql);
			}
		} catch (JSQLParserException e) {
			throw MiscUtil.toRuntimeException(e);
		}
	}

	/**
	 * 除去 SQL 中最后的 Order By 部分并包装成一个 SELECT COUNT(*) 的 SQL 语句
	 * @param sql 原 SQL
	 * @param countColumnName
	 * @return
	 */
	public static String buildCountSqlWithoutOrderBy(String sql, String countColumnName) {
		SQLParserResult pr = splitOuterOrderBy(sql);
		String countSql = "SELECT COUNT(*) AS "+countColumnName+" FROM ("+pr.getSelectSql()+") ALL_DATA";
		return countSql;
	}
	
	public static String adaptSqlRange(String sql, int from, int size){
		try {
			SQLParserResult spr = splitOuterOrderBy(sql);
			
			List<OrderByElement> orderBy = spr.getOrderBy();
			//检查 - 不支持没有 Order By 的 SQL 语句
			if (null==orderBy || orderBy.size()<=0){
				throw new UnsupportedOperationException(
						"The ranking function 'ROW_NUMBER' must have an ORDER BY clause: \n"+sql);
			}
			//检查 - 开窗函数不支持整数索引作为 ORDER BY 子句表达式
			for(OrderByElement oe: orderBy){
				String exp = oe.getExpression().toString();
				if (StringUtils.isNumeric(exp)){
					throw new UnsupportedOperationException(
							"Windowed functions do not support integer indices as ORDER BY clause expressions: \n"+sql);
				}
			}

			//创建 ROW_NUMBER() OVER(ORDER BY ...) AS rownum__$PAGING 这一列
			String rownum = "ROW_NUMBER() OVER("+spr.getOrderBySql()+")";
			SelectExpressionItem selItem = new SelectExpressionItem(CCJSqlParserUtil.parseExpression(rownum));
			selItem.setAlias(new Alias("rownum__$PAGING", true));
			
			PlainSelect ps = (PlainSelect)spr.getSelectStmt().getSelectBody();
			ps.addSelectItems(selItem);
			
			String sqlWithRownum = spr.getSelectSql();
			
			String result = "SELECT * FROM ("+sqlWithRownum+") AS T__$PAGING"
					      + " WHERE rownum__$PAGING>"+from+" AND rownum__$PAGING<="+(from+size)
					      + " ORDER BY rownum__$PAGING";
			return result;
		} catch (JSQLParserException e) {
			throw MiscUtil.toRuntimeException(e);
		}
	}

}
