package com.bokesoft.distro.tech.commons.basis.auth.annotation;

import com.bokesoft.distro.tech.commons.basis.auth.exception.NoPermissionException;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * 标记某个方法需要进行权限校验。
 *
 * <p>此注解通常用于控制器方法或服务层方法上，表示调用该方法前需要进行用户权限验证。
 * 如果用户没有相应的权限，则抛出 {@link NoPermissionException}。</p>
 *
 * <p><strong>示例,最简便用法：</strong></p>
 * <pre>{@code
 * @RequiresPermission()
 * public User getUser(int id) {
 *     return userService.getUserById(id);
 * }
 * }</pre>
 * <p><strong>示例,完整用法：</strong></p>
 * <pre>{@code
 * @RequiresPermission(
 *    value= {
 *        @PermissionCheck(permission = "user:read_write", strategy = Strategy.USER),
 *        @PermissionCheck(permission = "user:read", strategy = Strategy.TOKEN)
 *    },
 *    logical = Logical.ANY
 * )
 * public User getUser(int id) {
 *     return userService.getUserById(id);
 * }
 * }</pre>
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
public @interface RequiresPermission {

    /**
     * 这里的默认值是 TOKEN
     * token 的 权限校验标志 * 暂时是无实际意义的
     */
    PermissionCheck[] value() default @PermissionCheck(permission = "*", strategy = Strategy.TOKEN);

    /**
     * 逻辑关系，默认为 {@link Logical#ALL},也就是所有都匹配模式
     */
    Logical logical() default Logical.ALL;
}

