package com.bokesoft.distro.tech.commons.basis;

import java.net.*;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 网络信息相关的工具方法
 * @author guyadong
 * 参考资料: https://cloud.tencent.com/developer/article/1011721
 */
public class NetworkUtil {
    private static Logger logger = LoggerFactory.getLogger(NetworkUtil.class);
    private static String cachedHostName ;
    private static long hostNameLoadTime = 0;
    private static String cachedIPv4 ;
    private static long ipv4LoadTime = 0;
    private static String cachedMacAddress ;
    private static long macAddressLoadTime = 0;

    private static final long INTERVAL_TIME = 28_800_000;

    public static String getHost() {
        long curTime = System.currentTimeMillis();

        if(StringUtils.isNotBlank(cachedHostName) && curTime < hostNameLoadTime){
            return cachedHostName;
        }else {
            cachedHostName = System.getenv("BK_INSTANCE_HOSTNAME");
            if(StringUtils.isBlank(cachedHostName)) {
                cachedHostName = System.getenv("HOSTNAME");
            }
            if (StringUtils.isBlank(cachedHostName)) {
                // 如果可以通过InetAddress.getLocalHost()获取hostname,则尝试获取下
                try {
                    cachedHostName = InetAddress.getLocalHost().getHostName();
                }catch (UnknownHostException e){
                    logger.error("查找网络主机名称错误: " + e.getMessage(), e);
                    // 如果报错则hostname变为UNKNOWN,表示无从得知
                    cachedHostName = "unknown";
                }
            }
            if (StringUtils.isBlank(cachedHostName)) {
                cachedHostName = "unknown";
            }

            //相应延长缓存时间
            hostNameLoadTime = System.currentTimeMillis() + INTERVAL_TIME;

            return cachedHostName;
        }
    }

    public static String getIp() {
        long curTime = System.currentTimeMillis();
        if(StringUtils.isNotBlank(cachedIPv4) && curTime < ipv4LoadTime){
            return cachedIPv4;
        }else {
            cachedIPv4 = getRealIp();
            if(StringUtils.isNotBlank(cachedIPv4)) {
                ipv4LoadTime = System.currentTimeMillis() + INTERVAL_TIME;
            }
        }
        return cachedIPv4;
    }

    public static String getMacAddress() {

        long curTime = System.currentTimeMillis();
        if(StringUtils.isNotBlank(cachedMacAddress) && curTime < macAddressLoadTime){
            return cachedMacAddress;
        }else {
            cachedMacAddress = getRealMacAddress();
            if(StringUtils.isNotBlank(cachedMacAddress)) {
                macAddressLoadTime = System.currentTimeMillis() + INTERVAL_TIME;
            }
        }
        return cachedMacAddress;
    }

    private static String getRealIp() {
        String vmMacNetIP = "";
        String virtualNetIP = "";
        try {
            Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
            while(networkInterfaces.hasMoreElements()){
                NetworkInterface networkInterface = networkInterfaces.nextElement();
                if(isPhyicalOnlyNet(networkInterface)){
                    String phyicalIP = formatIPv4(networkInterface.getInetAddresses());
                    // 如果没有ipv4地址,则跳过进行下一个轮询
                    if(StringUtils.isNotBlank(phyicalIP)) {
                        logger.debug("发现物理网卡 {}, 返回 IPv4 地址 {} 作为默认的应用实例 ID", networkInterface.toString(), phyicalIP);
                        return phyicalIP;
                    }
                }else if(isVMMacNet(networkInterface)){
                    if(StringUtils.isBlank(vmMacNetIP)){
                        vmMacNetIP = formatIPv4(networkInterface.getInetAddresses());
                        logger.debug("发现 VM 网卡 {}, IPv4 地址为 {}", networkInterface.toString(), vmMacNetIP);
                    }
                }else if(isVirtualNet(networkInterface)){
                    if(StringUtils.isBlank(virtualNetIP)) {
                        virtualNetIP = formatIPv4(networkInterface.getInetAddresses());
                        logger.debug("发现虚拟网卡 {}, IPv4 地址为 {}", networkInterface.toString(), vmMacNetIP);
                    }
                }
            }
            if(StringUtils.isNotBlank(vmMacNetIP)){
                logger.debug("返回 VM 网卡 IPv4 地址为 {} 作为默认的应用实例 ID", vmMacNetIP);
                return vmMacNetIP;
            }else if(StringUtils.isNotBlank(virtualNetIP)){
                logger.debug("返回虚拟网卡 IPv4 地址为 {} 作为默认的应用实例 ID", virtualNetIP);
                return virtualNetIP;
            }
        }catch (Exception e){
            logger.error("网卡遍历过程出错: " + e.getMessage(), e);
        }
        return null;
    }

    private static String getRealMacAddress() {
        String vmMacNet = "";
        String virtualNet = "";
        try {
            Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
            while(networkInterfaces.hasMoreElements()){
                NetworkInterface networkInterface = networkInterfaces.nextElement();
                if(isPhyicalOnlyNet(networkInterface)){
                    String mac = format(networkInterface.getHardwareAddress());
                    logger.debug("发现物理网卡 {}, 返回 MAC 地址 {} 作为默认的应用实例 ID", networkInterface.toString(), mac);
                    return mac;
                }else if(isVMMacNet(networkInterface)){
                    if(StringUtils.isBlank(vmMacNet)){
                        vmMacNet = format(networkInterface.getHardwareAddress());
                        logger.debug("发现 VM 网卡 {}, MAC 地址为 {}", networkInterface.toString(), vmMacNet);
                    }
                }else if(isVirtualNet(networkInterface)){
                    if(StringUtils.isBlank(virtualNet)) {
                        virtualNet = format(networkInterface.getHardwareAddress());
                        logger.debug("发现虚拟网卡 {}, MAC 地址为 {}", networkInterface.toString(), virtualNet);
                    }
                }
            }
            if(StringUtils.isNotBlank(vmMacNet)){
                logger.debug("返回 VM 网卡 MAC 地址为 {} 作为默认的应用实例 ID", vmMacNet);
                return vmMacNet;
            }else if(StringUtils.isNotBlank(virtualNet)){
                logger.debug("返回虚拟网卡 MAC 地址为 {} 作为默认的应用实例 ID", virtualNet);
                return virtualNet;
            }
        }catch (Exception e){
            logger.error("网卡遍历过程出错: " + e.getMessage(), e);
        }
        return null;
    }

    private static boolean isVirtualNet(NetworkInterface input){
        return input.isVirtual();
    }

    private static boolean isPhyicalOnlyNet(NetworkInterface input) throws SocketException {
        byte[] hardwareAddress = input.getHardwareAddress();
        return null != hardwareAddress
                && hardwareAddress.length > 0
                && !input.isVirtual()
                && !isVMMac(hardwareAddress);
    }

    private static boolean isVMMacNet(NetworkInterface input) throws SocketException {
        byte[] hardwareAddress = input.getHardwareAddress();
        return null != hardwareAddress
                && hardwareAddress.length > 0
                && !input.isVirtual()
                && isVMMac(hardwareAddress);
    }

    private static final byte invalidMacs[][] = {
            {0x00, 0x05, 0x69},             // VMWare
            {0x00, 0x1C, 0x14},             // VMWare
            {0x00, 0x0C, 0x29},             // VMWare
            {0x00, 0x50, 0x56},             // VMWare
            {0x08, 0x00, 0x27},             // Virtualbox
            {0x0A, 0x00, 0x27},             // Virtualbox
            {0x00, 0x03, (byte)0xFF},       // Virtual-PC
            {0x00, 0x15, 0x5D}              // Hyper-V
    };

    private static boolean isVMMac(byte[] mac) {
        if(null == mac) {
            return false;
        }

        for (byte[] invalid: invalidMacs){
            if (invalid[0] == mac[0] && invalid[1] == mac[1] && invalid[2] == mac[2]) {
                return true;
            }
        }

        return false;
    }

    /**
     * 将{@code byte[]} 转换为{@code radix}指定格式的字符串
     *
     * @param source 网卡mac二进制数组
     * @return {@code source}为{@code null}时返回空字符串
     */
    public static String format(byte[] source) {
        if (null == source){
            return "";
        }
        List<String> hex = new ArrayList<>(source.length);
        for(byte input:source){
            hex.add(String.copyValueOf(new char[]{
                    Character.forDigit((input & 240) >> 4, 16),
                    Character.forDigit(input & 15, 16)
            }));
        }
        return StringUtils.join(hex,"-");
    }

    /**
     * 将{@code inetAddressEnumeration} 中IPV4的的网络地址提取出来
     *
     * @param inetAddressEnumeration 网卡网路地址
     */
    private static String formatIPv4(Enumeration<InetAddress> inetAddressEnumeration) {
        while (inetAddressEnumeration.hasMoreElements()){
            InetAddress inetAddress = inetAddressEnumeration.nextElement();
            if(null != inetAddress && inetAddress instanceof Inet4Address){
                return inetAddress.getHostAddress();
            }
        }
        return null;
    }
}

