package com.bokesoft.distro.tech.bootsupport.starter.filter;


import com.bokesoft.distro.tech.bootsupport.starter.config.YigoInstanceReadyConfig;
import com.bokesoft.distro.tech.bootsupport.starter.i18n.StringTable;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.YigoInstanceManager;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.model.InstanceStatus;
import com.bokesoft.distro.tech.bootsupport.starter.utils.ServletUtil;
import com.bokesoft.yigo.common.i18n.ILocale;
import com.bokesoft.yigo.common.util.SimpleStringFormat;
import jodd.util.Wildcard;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 检查 Yigo 实例是否启动完成, 在未启动完成的情况下, 拦截依赖于 Yigo 的 URL 请求.
 */
public class YigoInstanceReadyFilter implements Filter {

    private static final Logger logger = LoggerFactory.getLogger(YigoInstanceReadyFilter.class);

    private final String pcWaitingPage;
    private final String mobileWaitingPage;
    private final YigoInstanceManager yigoInstanceManager;
    private final YigoInstanceReadyConfig yigoInstanceReadyConfig;
    private static final Map<String,Boolean> independUrlsCache = new ConcurrentHashMap<>();

    public YigoInstanceReadyFilter(YigoInstanceManager yigoInstanceManager,String pcWaitingPage,
        String mobileWaitingPage, YigoInstanceReadyConfig yigoInstanceReadyConfig){
        this.yigoInstanceManager = yigoInstanceManager;
        this.pcWaitingPage = pcWaitingPage;
        this.mobileWaitingPage = mobileWaitingPage;
        this.yigoInstanceReadyConfig = yigoInstanceReadyConfig;
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest httpReq = (HttpServletRequest)request;
        HttpServletResponse httpResp = (HttpServletResponse)response;
        String url = httpReq.getServletPath();

        if(!isIndependUrl(url)) {
            InstanceStatus is = yigoInstanceManager.getInstanceStatus();
            if (!is.getStatus().equals(InstanceStatus.Status.READY)) {
                if (yigoInstanceReadyConfig.getAppPcPageUrls().contains(url)) {
                    // 如果是pc访问
                    httpReq.getRequestDispatcher(pcWaitingPage).forward(request, response);
                    return;
                } else if(yigoInstanceReadyConfig.getAppMobilePageUrls().contains(url)){
                    // 如果是手持访问
                    httpReq.getRequestDispatcher(mobileWaitingPage).forward(request, response);
                    return;
                } else {
                    // 如果是错误或不可用状态
                    if(is.getStatus().equals(InstanceStatus.Status.ERROR)
                    || is.getStatus().equals(InstanceStatus.Status.UNAVALIABLE)) {
                        logger.error("Yigo 应用目前处于{}状态, 原因:'{}', '{}' 请求将直接返回 500",
                                is.getStatus(),is.getMessage(), ServletUtil.getFullServletPath(httpReq));
                        ILocale locale = getLocaleByCookie(httpReq);
                        String errMsg = SimpleStringFormat.format(
                                StringTable.getString(locale, StringTable.MSG_GROUP, StringTable.MSG_YIGO_STATUS_UNAVALIABLE),
                                is.getStatus(),is.getMessage());
                        sendError(httpResp, "-1", errMsg);
                        return;
                    }else{
                        // 剩下状态为: PREPARING, WAITING, LOADING, UNLOADING, 认为在初始化过程中
                        logger.warn("Yigo 应用未初始化完成,状态:{}, '{}' 请求将直接返回 500",
                                is.getStatus(), ServletUtil.getFullServletPath(httpReq));
                        ILocale locale = getLocaleByCookie(httpReq);
                        String errMsg = SimpleStringFormat.format(
                                StringTable.getString(locale, StringTable.MSG_GROUP, StringTable.MSG_YIGO_STATUS_UNREADY),
                                is.getStatus());
                        sendError(httpResp, "-1", errMsg);
                        return;
                    }
                }
            }
        }
        chain.doFilter(request, response);
    }

    private ILocale getLocaleByCookie(HttpServletRequest httpReq) {
        Cookie[] cookies = httpReq.getCookies();
        if(null != cookies) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals("locale")) {
                    String localeStr = cookie.getValue();
                    return new ILocale() {
                        @Override
                        public String getLocale() {
                            return localeStr;
                        }
                    };
                }
            }
        }
        return null;
    }


    /**
     * 原有 {@link HttpServletResponse#sendError(int, String)} 方法会被spring-boot框架拦截处理,无法输出有效message消息
     * 所以使用 {@link HttpServletResponse#getWriter()} 对象来输出内容
     * @param httpResp httpResponse实例
     * @param errCode 错误代码
     * @param errMsg 错误消息
     * @throws IOException
     */
    private void sendError(HttpServletResponse httpResp, String errCode, String errMsg) throws IOException {
        JSONObject json = new JSONObject();
        json.put("error_info", errMsg);
        json.put("error_code", errCode);
        httpResp.setCharacterEncoding("UTF-8");
        httpResp.setContentType("application/json;charset=UTF-8");
        httpResp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        httpResp.getWriter().write(json.toString());
    }

    private boolean isIndependUrl(String url) {
        boolean result = independUrlsCache.computeIfAbsent(url, testUrl->{
            List<String> indepUrls = yigoInstanceReadyConfig.getYigoIndepUrls();
            if( null != indepUrls && !indepUrls.isEmpty()){
                for(String urlPattern: indepUrls){
                    //如果命中独立请url。放入缓存,并之直接返回
                    if(Wildcard.match(testUrl, urlPattern)){
                        return true;
                    }
                }
            }
            return false;
        });
        
        return result;
    }
}