package com.bokesoft.distro.tech.bootsupport.starter.datasource;

import java.sql.Connection;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.slf4j.event.Level;

import com.bokesoft.distro.tech.bootsupport.starter.utils.DataSourceAwareDBManagerBuilder;
import com.bokesoft.distro.tech.bootsupport.starter.wrapper.WrappedConnection;
import com.bokesoft.yigo.mid.connection.IConnectionProfile;
import com.bokesoft.yigo.mid.connection.IDBManager;
import com.bokesoft.yigo.mid.connection.IDBManagerFactory;

/**
 * 支持 SpringBoot 数据源的 DBManagerFactory
 */
public class DataSourceAwareDBManagerFactory implements IDBManagerFactory {
    private static Log log = LogFactory.getLog(DataSourceAwareDBManagerFactory.class);
    private static Log tracePrintLog = LogFactory.getLog(DataSourceAwareDBManagerFactory.class.getName()+".tracePrint");

    private static long expectPrintTraceTime = -1;
    private static long intervalMs = 300_000;

    private static DataSource datasource;
    private static Level logLevel = Level.DEBUG;

    private static boolean dbTraceCacheEnabled;
    private static boolean execTimeoutControlEnabled;

    public static void setDataSource(DataSource datasource) {
        DataSourceAwareDBManagerFactory.datasource = datasource;
    }
    public static void setLogLevel(String logLevelStr){
        if(null != logLevelStr) {
            switch (logLevelStr.toUpperCase()) {
                case "INFO":
                    logLevel = Level.INFO;
                    break;
                case "WARN":
                    logLevel = Level.WARN;
                    break;
                case "ERROR":
                    logLevel = Level.ERROR;
                default:
                    logLevel = Level.DEBUG;
                    break;
            }
        }
    }
    public static void setDbTraceCacheEnabled(boolean dbTraceCacheEnabled) {
        DataSourceAwareDBManagerFactory.dbTraceCacheEnabled = dbTraceCacheEnabled;
    }
    public static void setExecTimeoutControlEnabled(boolean execTimeoutControlEnabled) {
        DataSourceAwareDBManagerFactory.execTimeoutControlEnabled = execTimeoutControlEnabled;
    }
    public static void setPrintDBTraceIntervalTime(long intervalMs) {
        DataSourceAwareDBManagerFactory.intervalMs = intervalMs;
    }

    @Override
    public IDBManager getDBManager(IConnectionProfile cp) throws Throwable {
        try{
        return DataSourceAwareDBManagerBuilder.getDBManager(cp,datasource,logLevel,dbTraceCacheEnabled,execTimeoutControlEnabled);}
        catch (Exception ex){
            log.error("通过 Bean 创建 IConnectionFactory 失败", ex);
            if (dbTraceCacheEnabled) {
                if(canPrint()){
                    tracePrintLog.error(buildDBTraceContent());
                }
            }
            throw ex;
        }
    }

    /**
     * 检测是否满足打印当前所有未关闭dbconn的调用堆栈信息
     * @return
     */
    private boolean canPrint() {
        // 当前时间大于预期时间
        if(System.currentTimeMillis() > expectPrintTraceTime){
            // 同步后再次判断,防止多线程并发执行
            synchronized (this){
                // 当前时间大于预期时间
                if(System.currentTimeMillis() > expectPrintTraceTime) {
                    // 将下一次允许打印堆栈信息的时间设置指定间隔时间之后
                    expectPrintTraceTime = System.currentTimeMillis() + intervalMs;
                    return true;
                }
            }
        }
        return false;
    }

    private static String buildDBTraceContent() {
        StringBuffer result = new StringBuffer("DB Connection create Trace Info");
        result.append("\n");
        result.append("-----------------------------------------------------------------\n");
        int pos = 1;
        for (Connection conn : WrappedConnection.dbTraceCache.keySet()) {
            result.append("index[" + pos + "],connection:" + conn.toString() + " call by trace");
            result.append(WrappedConnection.dbTraceCache.get(conn));
            result.append("\n");
            result.append("-----------------------------------------------------------------\n");
            pos++;
        }
        return result.toString();
    }
}