(function () {
    'use strict';

    const funs = {};

    /**
     * 将用户的界面设置转化为 JSON 对象
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args
     */
    funs.ux_Settings2JSON = function (evl, name, ctx, args) {
        let settings = _YigoUX_Utils.getSettings();
        var themeName = settings.themeOption.themeName;
        var themeColorGroup = settings.themeOption.colorGroupName;
        var themeFontScheme = settings.themeOption.fontSchemeName;
        var navbarName = settings.navbarOption.navbarName;
        return {
            themeName, themeColorGroup, themeFontScheme, navbarName
        }
    }
    /**
     * 获取指定主题默认设置
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args
     */
    funs.ux_ThemeDefaultValue = function (evl, name, ctx, args) {
        // check
        if ((args.length != 1) || (typeof args[0] != 'string') || (args[0].trim() == '')) {
            return {};
        }
        // Get theme setting
        let themeName = args[0].trim();
        let themeSetting = findThemePackageByName(themeName);
        if (themeSetting == null) {
            return {};
        }
        // Analyze default value
        let result = {};
        let fields = ['colorGroups', 'fontSchemes'];
        for (let i = 0; i < fields.length; i++) {
            let fieldName = fields[i];
            let items = themeSetting[fieldName];
            // invalid list
            if (!items || (items.length == 0)) {
                result[fieldName] = null;
                continue;
            }
            // set first item as default item
            result[fieldName] = items[0].name;
            // search fallback item
            for (let j = 0; j < items.length; j++) {
                if (items[j].fallback) {
                    result[fieldName] = items[j].name;
                    break;
                }
            }
        }
        return result;
    }
    /**
     * 准备主题下拉框内容
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args
     */
    funs.ux_ThemeComboboxItems = function (evl, name, ctx, args) {
        let settings = _YigoUX_Utils.getSettings();
        var avaliableThemes = settings.avaliableThemes;
        var items = [];
        for (var i = 0; i < avaliableThemes.length; i++) {
            var t = avaliableThemes[i];
            items.push(t.name + "," + t.caption);
        }
        return items.join(";");
    }
    /**
     * 准备色调下拉框内容
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args 0: themeName
     */
    funs.ux_ThemeColorGroupComboboxItems = function (evl, name, ctx, args) {
        const themeName = args[0];
        var theme = findThemePackageByName(themeName);
        if (!theme) {
            return "";
        }
        if (theme.colorPalette == "Monochromatic") {
            var items = [];
            var cgs = theme.colorGroups || [];
            for (var i = 0; i < cgs.length; i++) {
                var colorGroup = cgs[i];
                items.push(colorGroup.name + "," + colorGroup.caption);
            }
            return items.join(";");
        } else {
            //其他类型暂时都不支持
            return "";
        }
    }
    /**
     * 准备字体下拉框内容
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args 0: themeName
     */
    funs.ux_ThemeFontSchemeComboboxItems = function (evl, name, ctx, args) {
        const themeName = args[0];
        var theme = findThemePackageByName(themeName);
        if (!theme) {
            return "";
        }
        var items = [];
        var fss = theme.fontSchemes || [];
        // 排序
        fss.sort((a, b) => {
            return a.baseFontSize <= b.baseFontSize ? -1 : 0;
        });
        // 生成
        for (var i = 0; i < fss.length; i++) {
            var fs = fss[i];
            items.push(fs.name + "," + fs.caption);
        }
        return items.join(";");
    }
    /**
     * 准备主菜单下拉框内容
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args
     */
    funs.ux_NavbarComboboxItems = function (evl, name, ctx, args) {
        let settings = _YigoUX_Utils.getSettings();
        var avaliableNavbars = settings.avaliableNavbars;
        var items = [];
        for (var i = 0; i < avaliableNavbars.length; i++) {
            var t = avaliableNavbars[i];
            items.push(t.name + "," + t.caption);
        }
        return items.join(";");
    }
    /**
     * 保存用户设置
     * @param {*} evl
     * @param {*} name
     * @param {*} ctx
     * @param {*} args 0: JSON 数据 {themeName, themeColorGroup, navbarName}
     */
    funs.ux_SaveUserPreference = async function (evl, name, ctx, args) {
        await savePreference(args[0]);
        setTimeout(() => {
            location.reload();
        }, 200)
    }

    function findThemePackageByName(themeName) {
        let settings = _YigoUX_Utils.getSettings()
        var avaliableThemes = settings.avaliableThemes;
        for (var i = 0; i < avaliableThemes.length; i++) {
            var t = avaliableThemes[i];
            if (t.name == themeName) {
                return t;
            }
        }
        return null;
    }

    async function savePreference(pref) {
        saveThemeToCookie(pref);
        let settings = _YigoUX_Utils.getSettings();
        var formkey = settings.clientParameters.individualityFormKey;
        var obj = {
            themeName: pref.themeName,
            colorGroupName: pref.themeColorGroup,
            fontSchemeName: pref.themeFontScheme
        }
        var prefData = {
            navbarOption: {
                navbarName: pref.navbarName
            }
        };
        if (_YigoUX_Utils.isMobile()) {
            prefData['themeOptionMobile'] = obj;
        } else {
            prefData['themeOption'] = obj;
        }

        //FIXME 额外的 JSON.stringify() 是为了保证与旧版 savePreference 的兼容性
        prefData = {data: JSON.stringify(prefData)};
        var service = new YIUI.RemoteService();
        //FIXME 兼容 API, formKey 在后端实际作为 PreferenceKey
        return service.savePreference2(formkey, formkey, prefData);
    }

    function saveThemeToCookie(pref) {
        var obj = {
            themeName: pref.themeName,
            colorGroupName: pref.themeColorGroup,
            fontSchemeName: pref.themeFontScheme
        }
        var localData = {}

        if (_YigoUX_Utils.isMobile()) {
            localData['themeOptionMobile'] = obj;
        } else {
            localData['themeOption'] = obj;
        }
        window.dispatchEvent(new CustomEvent('YIUI-ux-theme-changed', {
            detail: localData
        }));
    }

    function waitYIUIReady() {
        if (window.Expr && window.View) {
            window.Expr.regCluster(window.View.FuncMap, funs);
        } else {
            //FIXME 目前无法准确判定 YIUI 加载完成
            setTimeout(waitYIUIReady, 100);
        }
    }
    waitYIUIReady();
})()
