package com.bokesoft.yigo.ux.bootstarter.impl;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Base64;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.util.WebUtils;

import com.bokesoft.distro.tech.bootsupport.starter.beans.PagesBuildService;
import com.bokesoft.distro.tech.commons.basis.dependency.DependencySortCore;
import com.bokesoft.yigo.common.def.PlatformType;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.struct.preference.Preference;
import com.bokesoft.yigo.ux.bootstarter.data.SpringUxConfigData;
import com.bokesoft.yigo.ux.bootstarter.data.additional.AdditionalThemeSetting;
import com.bokesoft.yigo.ux.defination.classic.YigoClassicUxSetting;
import com.bokesoft.yigo.ux.defination.classic.YigoClassicUxSetting.ClientParameters;
import com.bokesoft.yigo.ux.defination.classic.config.UxSystemParameters;
import com.bokesoft.yigo.ux.defination.classic.decorations.DefaultLoginDecoration;
import com.bokesoft.yigo.ux.defination.classic.decorations.DefaultMainFrameDecoration;
import com.bokesoft.yigo.ux.defination.classic.decorations.DefaultSiteDecoration;
import com.bokesoft.yigo.ux.defination.classic.navbar.NavbarOption;
import com.bokesoft.yigo.ux.defination.classic.navbar.NavbarPackage;
import com.bokesoft.yigo.ux.defination.classic.navigation.NavActionPackage;
import com.bokesoft.yigo.ux.defination.classic.navigation.NavActionSetting;
import com.bokesoft.yigo.ux.defination.classic.navigation.NavigationBuildin;
import com.bokesoft.yigo.ux.defination.classic.navigation.NavigationOption;
import com.bokesoft.yigo.ux.defination.common.themes.ThemeDecoration;
import com.bokesoft.yigo.ux.defination.common.themes.ThemeOption;
import com.bokesoft.yigo.ux.defination.common.themes.ThemePackage;
import com.bokesoft.yigo.ux.defination.common.themes.ThemePackage.ColorGroup;
import com.bokesoft.yigo.ux.defination.common.themes.ThemePackage.FontScheme;
import com.bokesoft.yigo.ux.defination.mobile.YigoMobileUxSetting;
import com.bokesoft.yigo.ux.defination.mobile.YigoMobileUxSetting.ClientParametersMobile;
import com.bokesoft.yigo.ux.impl.DefaultUxFactory;
import com.bokesoft.yigo.ux.intf.ext.IClassicDecorationProvider;
import com.bokesoft.yigo.ux.intf.providers.IUxPackageProvider;
import com.bokesoft.yigo.ux.intf.providers.IUxSettingProvider;
import com.bokesoft.yigo.ux.preference.classic.UxUserLocalData;
import com.bokesoft.yigo.ux.preference.classic.UxUserPreferenceData;
import com.bokesoft.yigo.ux.utils.ExtendedFieldsUtils;
import com.bokesoft.yigo.ux.utils.PackagePathUtils;

import jodd.util.Wildcard;

/**
 * Spring Web 方式下的 {@link IUxSettingProvider} 实现, 综合处理来自 application 配置和用户自定义的 Ux 设置
 */
public class SpringWebUxSettingProvider implements IUxSettingProvider{
    private static final Logger log = LoggerFactory.getLogger(SpringWebUxSettingProvider.class);

    private SpringUxConfigData configData;
    private ResourceLoader resourceLoader;

    private List<IClassicDecorationProvider> additionalDecorationProviders;

    public SpringWebUxSettingProvider(SpringUxConfigData configData, ResourceLoader resourceLoader,
                                      List<IClassicDecorationProvider> additionalDecorationProviders) {
        this.configData = configData;
        this.resourceLoader = resourceLoader;

        if (null!=additionalDecorationProviders){
            this.additionalDecorationProviders = DependencySortCore.sort(additionalDecorationProviders);
        }
    }

    private IUxPackageProvider getPackageProvider() {
        return DefaultUxFactory.getUxPackageProvider();
    }

    /**
     * FIXME 补充 {@link PagesBuildService} 在 buildPage 时没有处理 locale 的问题
     * @param context
     */
    private static void patchLocaleCookie(DefaultContext context){
        if (null==context || null==context.getEnv()){
            return;
        }

        RequestAttributes ra = RequestContextHolder.getRequestAttributes();
        if (null==ra){
            return;
        }
        ServletRequestAttributes sra = (ServletRequestAttributes)ra;
        HttpServletRequest req = sra.getRequest();
        if (null==req){
            return;
        }

        Cookie[] cookies = req.getCookies();
        if (null==cookies){
            return;
        }
        for(int i=0; i<cookies.length; i++){
            Cookie c = cookies[i];
            if ("locale".equals(c.getName())){
                context.getEnv().setLocale(c.getValue());
                break;
            }
        }
    }

    @Override
    public YigoClassicUxSetting getClassicUxSetting(DefaultContext context) {
        patchLocaleCookie(context);

        YigoClassicUxSetting setting = new YigoClassicUxSetting();

        // 处理返回客户端的变量
        UxSystemParameters sysPara = configData.getSystemParameters();
        // 通过配置定义客户端的参数
        setting.setClientParameters(getClientParameters(sysPara));
        // 客户端自定义设置
        setting.setClientSettings(configData.getClientSettings());
        // 获取用户参数
        UxUserPreferenceData userPref = getUserPref(context,sysPara.getIndividualityFormKey());
        // 通过配置获取主菜单定义
        setting.setNavbarOption(getNavbarOption(userPref));
        // 通过配置获取主题定义
        ThemeOption themeOpt = getThemeOption(userPref, context);
        setting.setThemeOption(themeOpt);
        // 处理导航栏区域的设置
        setting.setNavigationOption(getNavigationOption());

        //处理可用的模块(系统信息, 主要用于用户界面自定义)
        setting.setAvaliableNavbars(this.getPackageProvider().getNavbars());
        setting.setAvaliableNavActions(mergeAvaliableNavActions(this.getPackageProvider().getNavActions())); // 合并

        setting.setAvaliableThemes(getAvaliableThemePackagesByPlatform(PlatformType.PC, context));

        processThemeExtension(themeOpt,
            vars->setting.setAdditionalThemeVariables(vars),
            urls->setting.setExtendThemeCssUrls(urls));

        mergeSimpleDecorations(context, setting, configData);

        // 处理 siteDecoration 的 faviconResourceUrl, 主要是将后端资源(classpath:...)调整为前端资源
        DefaultSiteDecoration siteDecoration = setting.getSiteDecoration();
        if (null!=siteDecoration && null!=siteDecoration.getFaviconResourceUrl()){
            siteDecoration.setFaviconResourceUrl(processImageResourceData(siteDecoration.getFaviconResourceUrl()));
        }

        // 处理扩展字段相关，主要是处理下拉框类型配置自定义来源于二开。
        ExtendedFieldsUtils.processClassicUxSetting(setting);
        
        return setting;
    }

    @Override
    public YigoMobileUxSetting getMobileUxSetting(DefaultContext context) {
        patchLocaleCookie(context);

        YigoMobileUxSetting setting = new YigoMobileUxSetting();
        // 处理返回客户端的变量
        UxSystemParameters sysPara = configData.getSystemParameters();
        // 通过配置定义客户端的参数
        setting.setClientParameters(getClientParametersMobile(sysPara));
        // 获取用户参数
        UxUserPreferenceData userPref = getUserPref(context,sysPara.getMobileIndividualityFormKey());
        // 通过配置获取主题定义
        ThemeOption themeOpt = getThemeOptionMobile(userPref, context);
        setting.setThemeOption(themeOpt);

        setting.setAvaliableThemes(getAvaliableThemePackagesByPlatform(PlatformType.Mobile, context));

        processThemeExtension(themeOpt,
            vars->setting.setAdditionalThemeVariables(vars),
            urls->setting.setExtendThemeCssUrls(urls));

        return setting;
    }

    /**
     * 根据匹配规则计算当前主题的附加变量和扩展 css 地址
     * @param themeOpt
     * @param vars
     * @param urls
     */
    private void processThemeExtension (ThemeOption themeOpt, Consumer<Map<String, String>> vars , Consumer<List<String>> urls) {
        if (null!= themeOpt) {
            String themePath = PackagePathUtils.buildThemePackagePath(themeOpt, false);
            List<AdditionalThemeSetting> addiThemeSettings = configData.getAdditionalThemeSettings();

            Map<String, String> addiThemeVars = buildAddiThemeVars(themePath, addiThemeSettings);
            if (null != addiThemeVars && !addiThemeVars.isEmpty()) {
                vars.accept(addiThemeVars);
            }

            List<String> extCssUrls = buildExtThemeCssUrls(themePath, addiThemeSettings);
            if (null != extCssUrls && !extCssUrls.isEmpty()) {
                urls.accept(extCssUrls);
            }
        }
    }

    //处理导航栏区域的设置(通常不由用户决定)
    private NavigationOption getNavigationOption() {
        //处理导航栏区域的设置(通常不由用户决定)
        NavigationOption navOpts = configData.getNavigation();
        if (null==navOpts){
            navOpts = new NavigationOption();
        }
        List<String> userActions = navOpts.getUserActions();
        if (null==userActions){
            userActions = new ArrayList<>();
            navOpts.setUserActions(userActions);
        }
        if (userActions.isEmpty()){
            userActions.addAll(NavigationBuildin.DEFAULT_USER_ACTIONS);
        }
        List<String> settingActions = navOpts.getSettingActions();
        if (null==settingActions){
            settingActions = new ArrayList<>();
            navOpts.setSettingActions(settingActions);
        }
        if (settingActions.isEmpty()){
            settingActions.addAll(NavigationBuildin.DEFAULT_SETTING_ACTIONS);
        }
        return navOpts;
    }

    private ThemeOption getThemeOption(UxUserPreferenceData userPref, DefaultContext context) {
        //通过配置获取主题定义
        ThemeOption themeOpt = configData.getThemeOption();
        //获取根据用户或者客户端 cookie 所设置的主题定义
        if (null!= userPref && null!= userPref.getThemeOption()){
            themeOpt = fillupUserOpt(userPref.getThemeOption(), themeOpt);
        }
        // 检查配置内容有效性，并进行调整
        return getAvailableThemeOption(themeOpt, context);
    }

    private ThemeOption getThemeOptionMobile(UxUserPreferenceData userPref, DefaultContext context) {
        //通过配置获取移动端主题定义
        ThemeOption themeOpt = configData.getThemeOptionMobile();
        //获取根据用户或者客户端 cookie 所设置的主题定义
        if (null!= userPref && null!= userPref.getThemeOptionMobile()){
            themeOpt = fillupUserOpt(userPref.getThemeOptionMobile(), themeOpt);
        }
        // 检查配置内容有效性，并进行调整
        return getAvailableThemeOption(themeOpt, context);
    }

    /**
     * 尝试使用系统配置的 Theme 信息补全用户的 Theme 配置信息
     * @param userOpt
     * @param themeOpt
     * @return
     */
    private static ThemeOption fillupUserOpt(ThemeOption userOpt, ThemeOption themeOpt) {
        if (null==themeOpt){
            return userOpt;
        }
        if (! StringUtils.equals(userOpt.getThemeName(), themeOpt.getThemeName())){
            return userOpt;
        }

        //特殊处理 - 如果系统配置中定义了主题的 colorGroupName 等选择, 可以影响用户配置中没有定义的选择
        if (StringUtils.isBlank(userOpt.getColorGroupName())
                && StringUtils.isNotBlank(themeOpt.getColorGroupName())){
            userOpt.setColorGroupName(themeOpt.getColorGroupName());
        }
        if (StringUtils.isBlank(userOpt.getFontSchemeName())
                && StringUtils.isNotBlank(themeOpt.getFontSchemeName())){
            userOpt.setFontSchemeName(themeOpt.getFontSchemeName());
        }
        
        return userOpt;
    }

    private NavbarOption getNavbarOption(UxUserPreferenceData userPref) {
        //通过配置获取主菜单定义
        NavbarOption navbarOption = configData.getNavbarOption();
        //获取根据用户或者客户端 cookie 所设置的主题定义
        if (null!= userPref && null!= userPref.getNavbarOption()){
            NavbarOption userOpt = userPref.getNavbarOption();
            //用户配置覆盖系统配置
            navbarOption = userOpt;
        }
        // 检查配置内容有效性，并进行调整
        navbarOption = getAvailableNavbarOption(navbarOption);
        return navbarOption;
    }
    // 定义客户端的参数
    private static ClientParameters getClientParameters(UxSystemParameters sysPara) {
        ClientParameters toClient = new ClientParameters();

        if (StringUtils.isNotBlank(sysPara.getIndividualityFormKey())) {
            toClient.setIndividualityFormKey(sysPara.getIndividualityFormKey());
        }else{
            throw new UnsupportedOperationException("未正确配置 IndividualityFormKey");
        }

        if (StringUtils.isNotBlank(sysPara.getMenuHelpFormula())) {
            toClient.setMenuHelpFormula(sysPara.getMenuHelpFormula());
        }
        return toClient;
    }

    private YigoClassicUxSetting mergeSimpleDecorations(DefaultContext context, YigoClassicUxSetting setting, SpringUxConfigData configData) {
        DefaultLoginDecoration loginDecoration = new DefaultLoginDecoration();
        DefaultMainFrameDecoration mainFrameDecoration = new DefaultMainFrameDecoration();
        DefaultSiteDecoration siteDecoration = new DefaultSiteDecoration();

        List<IClassicDecorationProvider> providers = this.additionalDecorationProviders;
        if (null!=providers){
            String providerClasses = providers.stream().map(p -> p.getClass().getName()).collect(Collectors.joining(", "));
            log.info("开始处理附加 Decoration: {}", providerClasses);

            DefaultLoginDecoration currLoginDecoration = configData.getLogin();
            DefaultMainFrameDecoration currMainFrameDecoration = configData.getMainframe();
            DefaultSiteDecoration currSiteDecoration = configData.getSite();
            
            for (IClassicDecorationProvider p: providers){
                DefaultLoginDecoration addiLoginDecoration = p.getLoginDecoration(context);
                mergeSimpleConfigObject(loginDecoration, addiLoginDecoration, currLoginDecoration);
                currLoginDecoration = loginDecoration;
        
                DefaultMainFrameDecoration addiMainFrameDecoration = p.getMainFrameDecoration(context);
                mergeSimpleConfigObject(mainFrameDecoration, addiMainFrameDecoration, currMainFrameDecoration);
                currMainFrameDecoration = mainFrameDecoration;
        
                DefaultSiteDecoration addiSiteDecoration = p.getSiteDecoration(context);
                mergeSimpleConfigObject(siteDecoration, addiSiteDecoration, currSiteDecoration);
                currSiteDecoration = siteDecoration;
            }
        }

        setting.setLoginDecoration(loginDecoration);
        setting.setMainFrameDecoration(mainFrameDecoration);
        setting.setSiteDecoration(siteDecoration);
        return setting;
    }

    private String processImageResourceData(String url) {
         if (StringUtils.isBlank(url)) {
            return null;
        }

        if (url.startsWith("data:")) { // 已经是数据了，直接返回
            return url;
        }

        Resource res = resourceLoader.getResource(url);

        String filename = res.getFilename();
        String ext = null;
        if (StringUtils.isNotBlank(filename)) {
            int pointIndex = filename.indexOf(".");
            if ((pointIndex > 0) /* 存在点，且不以点开头 */
                && (pointIndex < (filename.length() - 1)) /* 不以点结尾 */) {
                ext = filename.substring(filename.lastIndexOf(".") + 1);
            }
        }
        if (StringUtils.isBlank(ext)) {
            ext = "png";
        }
        ext = ext.toLowerCase();

        try (InputStream in = res.getInputStream()) {
            byte[] bytes = IOUtils.toByteArray(in);
            String data = "data:image/" + ext + ";base64," + Base64.getEncoder().encodeToString(bytes);
            return data;
        } catch (IOException ex) {
            log.error("读取图片资源 '" + url + "' 失败：" + ex.getMessage(), ex);
            return null;
        }
    }
    private static void mergeSimpleConfigObject(Object result, Object additional, Object current){
        List<Field> flds = FieldUtils.getAllFieldsList(result.getClass());
        for (Field fld : flds){
            setNotBlankField(result, additional, current, fld);
        }
    }
    private static void setNotBlankField(Object result, Object additional, Object current, Field fld){
        try {
            Object value = null;
            if (null != additional) { // Additional 内容优先
                value = FieldUtils.readField(fld, additional, true);
            }
            if (null==value || StringUtils.isBlank(value.toString())){
                if (null!=current){
                    value = FieldUtils.readField(fld, current, true);
                }
            }
            if (null!=value){
                FieldUtils.writeField(fld, result, value, true);
            }
        } catch (IllegalAccessException e) {
            ExceptionUtils.rethrow(e);
        }

    }

    private NavbarOption getAvailableNavbarOption(NavbarOption option) {
        List<NavbarPackage> nps = this.getPackageProvider().getNavbars();
        if (null == nps || nps.isEmpty()) {
            return null;
        }

        String navbarName = null!=option ? option.getNavbarName() : null;
        NavbarPackage np = null;
        if (StringUtils.isNotBlank(navbarName)) {
            for (NavbarPackage p: nps) {
                if (navbarName.equals(p.getName())) {
                    np = p;
                    break;
                }            
            }
        }
        if (null == np) { // 当前navbar无效, 取第一个有效navbar
            navbarName = nps.get(0).getName();
        }

        NavbarOption result = new NavbarOption();
        result.setNavbarName(navbarName);

        return result;
    }

    private ColorGroup getThemeDefaultColorGroup(ThemePackage tp) {
        if (null != tp) {
            List<ColorGroup> cgs = tp.getColorGroups();
            if (null != cgs && !cgs.isEmpty()){
                for (int i = 0; i < cgs.size(); i++) {
                    ColorGroup cg = cgs.get(i);
                    if (cg.isFallback()) return cg; // 返回带有默认属性的项
                }

                return cgs.get(0); // 返回第一项
            }    
        }

        return null; // 无默认颜色
    }

    private FontScheme getThemeDefaultFontSchema(ThemePackage tp) {
        if (null != tp) {
            List<FontScheme> fss = tp.getFontSchemes();
            if (null != fss && !fss.isEmpty()){
                for (int i = 0; i < fss.size(); i++) {
                    FontScheme fs = fss.get(i);
                    if (fs.isFallback()) return fs; // 返回带有默认属性的项
                }

                return fss.get(0); // 返回第一项
            }    
        }

        return null; // 无默认字体
    }

    private List<ThemePackage> getAvaliableThemePackages(DefaultContext context) {
        List<ThemePackage> avaliableThemes = this.getPackageProvider().getThemes();

        //处理 Theme 的修饰
        List<IClassicDecorationProvider> providers = this.additionalDecorationProviders;
        if (null!=providers){
            for (IClassicDecorationProvider p: providers){
                mergeThemeDecorations(avaliableThemes, p.getThemeDecorations(context), p.getClass().getName());
            }
        }

        return avaliableThemes;
    }

    private ThemeOption getAvailableThemeOption(ThemeOption option, DefaultContext context) {
        List<ThemePackage> tps = getAvaliableThemePackages(context);
        if (null == tps || tps.isEmpty()) {
            return null;
        }

        // 主题
        String themeName = null!=option ? option.getThemeName() : null;
        ThemePackage tp = null;
        if (StringUtils.isNotBlank(themeName)) {
            for (ThemePackage p: tps) {
                if (themeName.equals(p.getName())) {
                    tp = p;
                    break;
                }            
            }
        }
        if (null == tp) { // 当前主题无效, 取第一个有效主题
            tp = tps.get(0);
            ThemeOption result = new ThemeOption();

            result.setThemeName(tp.getName());

            // 默认颜色
            ColorGroup cg = getThemeDefaultColorGroup(tp);
            if (null != cg) {
                result.setColorGroupName(cg.getName());
            }

            // 默认字体
            FontScheme fs = getThemeDefaultFontSchema(tp);
            if (null != fs){
                result.setFontSchemeName(fs.getName());
            }
            
            return result;
        }

        // 颜色
        String colorGroupName = null!=option ? option.getColorGroupName() : null;
        List<ColorGroup> cgs = tp.getColorGroups();
        if (StringUtils.isNotBlank(colorGroupName) && null!=cgs && !cgs.isEmpty()) {
            boolean found = false;
            for (ColorGroup cg: cgs) {
                if (colorGroupName.equals(cg.getName())) {
                    found = true;
                    break;
                }
            }

            if (!found) { // 颜色无效，取默认颜色
                ColorGroup cg = getThemeDefaultColorGroup(tp);
                if (null != cg) {
                    colorGroupName = cg.getName();
                }
            }
        }

        // 字体
        String fontSchemeName = null!=option ? option.getFontSchemeName() : null;
        List<FontScheme> fss = tp.getFontSchemes();
        if (StringUtils.isNotBlank(fontSchemeName) && null!=fss && !fss.isEmpty()) {
            boolean found = false;
            for (FontScheme fs: fss) {
                if (fontSchemeName.equals(fs.getName())) {
                    found = true;
                    break;
                }
            }

            if (!found) { // 字体无效，取默认字体
                FontScheme fs = getThemeDefaultFontSchema(tp);
                if (null != fs) {
                    fontSchemeName = fs.getName();
                }
            }
        }

        // 返回数据
        ThemeOption result = new ThemeOption();
        result.setThemeName(themeName);
        result.setColorGroupName(colorGroupName);
        result.setFontSchemeName(fontSchemeName);

        return result;
    }

    private Map<String, String> buildAddiThemeVars(String themePath, List<AdditionalThemeSetting> settings){
        if (null==themePath){
            return null;
        }
        if (null==settings || settings.isEmpty()){
            return null;
        }

        Map<String, String> result = new LinkedHashMap<>();
        settings.forEach(setting -> {
            String namePattern = setting.getTheme();
            if (Wildcard.matchPath(themePath, namePattern)){
                putAddiThemeVars(result, setting.getLessVariables(), "@");
                putAddiThemeVars(result, setting.getCssVariables(), "--");
            }
        });
        return result;
    }
    private void putAddiThemeVars(Map<String, String> result, Map<String, String> vars, String prefix){
        if (null==vars){
            return;
        }
        vars.forEach((key, value)->{
            if (! key.startsWith(prefix)){
                key = prefix + key;
            }
            result.put(key, value);
        });
    }
    private List<String> buildExtThemeCssUrls(String themePath, List<AdditionalThemeSetting> settings){
        if (null==themePath){
            return null;
        }
        if (null==settings || settings.isEmpty()){
            return null;
        }

        Set<String> result = new LinkedHashSet<>();
        settings.forEach(setting -> {
            String namePattern = setting.getTheme();
            if (Wildcard.matchPath(themePath, namePattern)){
                List<String> urls = setting.getExtendCssUrls();
                if (null!=urls && !urls.isEmpty()){
                    result.addAll(urls);
                }
            }
        });

        return new ArrayList<>(result);
    }

    private UxUserPreferenceData getUserPref(DefaultContext context,String individualityFormKey) {
        long uid = -1;
        if( null != context ) {
            uid = context.getUserID();
        }
        if (uid <= 0){
            //未登录, 从 cookie 获取
            return readByCookie();
        }else{
            //已登录, 从用户喜好获取
            UxUserPreferenceData pref = readByPreference(context,individualityFormKey);
            if (null==pref){
                //如果用户喜好中没有, 仍然尝试从 cookie 中读取
                pref = readByCookie();
            }
            return pref;
        }
    }
    private UxUserPreferenceData readByPreference(DefaultContext context,String individualityFormKey) {
        if (StringUtils.isBlank(individualityFormKey)) {
            throw new UnsupportedOperationException("未正确配置 IndividualityFormKey");
        }
        try {
            Preference pref;
            try {
                String prefKey = individualityFormKey;
                String formKey = prefKey; // FIXME: Yigo 3 中因为兼容原因, 这里 FormKey 始终为空, 使用 prefKey 当作 formKey,
                // 20231207将prefKey赋值给fromKey
                // FIXME Preference API 需要特殊处理材料兼容 E3.1.6
                /*
                 * IPreferenceIO prefIO = PreferenceIOFactory.getPreferenceIO(context);
                 * Preference pref = prefIO.load(prefKey, formKey, context);
                 */
                Class<?> prefFac = Class.forName("com.bokesoft.yigo.mid.preference.PreferenceIOFactory");
                Object prefIO = MethodUtils.invokeStaticMethod(prefFac, "getPreferenceIO", context);
                pref = (Preference) MethodUtils.invokeMethod(prefIO, "load", prefKey, formKey, context);
            } catch (ClassNotFoundException ex) {
                // 说明是 Yigo 2 版本的 API
                String formKey = individualityFormKey;
                /*
                 * IPreferenceIO prefIO =
                 * com.bokesoft.yes.mid.cmd.form.preference.PreferenceIOFactory.getPreferenceIO(
                 * context);
                 * Preference pref = prefIO.load(formKey, context);
                 */
                Class<?> prefFac = Class.forName("com.bokesoft.yes.mid.cmd.form.preference.PreferenceIOFactory");
                Object prefIO = MethodUtils.invokeStaticMethod(prefFac, "getPreferenceIO", context);
                pref = (Preference)MethodUtils.invokeMethod(prefIO, "load", formKey, context);
            }

            if (null==pref){
                return null;
            }
            return UxUserPreferenceData.fromPreference(pref);
        } catch (Throwable e) {
            log.error("从用户喜好获取 UX 设置错误", e);
            return null;
        }
    }

    private UxUserPreferenceData readByCookie() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes ();
        if (null==requestAttributes){
            return null;
        }
        HttpServletRequest req = ((ServletRequestAttributes)requestAttributes).getRequest ();
        Cookie cookie = WebUtils.getCookie(req, UxUserLocalData.CLIENT_COOKIE_KEY);
        if (null==cookie){
            return null;
        }
        String val = cookie.getValue();
        try{
            UxUserLocalData uld = UxUserLocalData.fromCookieValue(val);
            return UxUserPreferenceData.fromUserLocalData(uld);
        } catch (Throwable e) {
            log.error("从 Cookie 获取 UX 设置错误", e);
            return null;
        }
    }

    private List<NavActionPackage> mergeAvaliableNavActions(List<NavActionPackage> defaultSettings) {
        Map<String, NavActionSetting> userSettings = configData.getNavActionSettings();

        if (null == userSettings) { // Invalid user settings
            return defaultSettings;
        }

        // update default setting
        for (String key: userSettings.keySet()) {
            NavActionSetting userSetting = userSettings.get(key);

            boolean found = false;
            for (int i=0; i<defaultSettings.size(); i++) {
                NavActionPackage setting = defaultSettings.get(i);

                if (key.equals(setting.getName())) { // found
                    // caption
                    if (StringUtils.isNotBlank(userSetting.getCaption())) {
                        setting.setCaption(userSetting.getCaption());
                    }

                    // icon
                    if (StringUtils.isNotBlank(userSetting.getIcon())) {
                        if (userSetting.getIcon().trim().equals("-")) { //disable icon
                            setting.setIcon(null);
                        } else { // update icon
                            setting.setIcon(userSetting.getIcon());
                        }
                    }

                    found = true;
                    break;
                }
            }

            // invalid user setting?
            if (!found) {
                throw new RuntimeException("无效的NavAction:" + key);
            }
        }

        return defaultSettings;
    }
    // 定义客户端的参数
    private static ClientParametersMobile getClientParametersMobile(UxSystemParameters sysPara) {
        ClientParametersMobile toClient = new ClientParametersMobile();

        if (StringUtils.isNotBlank(sysPara.getMobileIndividualityFormKey())) {
            toClient.setIndividualityFormKey(sysPara.getMobileIndividualityFormKey());
        }else{
            throw new UnsupportedOperationException("未正确配置 IndividualityMobileFormKey");
        }
        
        return toClient;
    }
    private List<ThemePackage> getAvaliableThemePackagesByPlatform(int platform, DefaultContext context) {
        List<ThemePackage> avaliableThemes = getAvaliableThemePackages(context);
        List<ThemePackage> newAvaliableThemes = new ArrayList<>();
        avaliableThemes.forEach(ThemePackage -> {
            Integer platformSupport = ThemePackage.getPlatformType();
            if (null==platformSupport){
                platformSupport = PlatformType.PC;  //兼容旧版本 - 默认视为仅兼容 PC
            }
            if((platformSupport.intValue() & platform) == platform){
                newAvaliableThemes.add(ThemePackage);
            }
        });

        return newAvaliableThemes;
    }
    private void mergeThemeDecorations(List<ThemePackage> themes, List<ThemeDecoration> themeDecorations, String progName){
        if ( (null==themeDecorations) || (themeDecorations.isEmpty()) ){
            return;
        }

        if ( (null!=themes) && (! themes.isEmpty()) ){
            for(ThemePackage t: themes){
                doMergeThemeDecorations(t, themeDecorations, progName);
            }
        }
    }
    private void doMergeThemeDecorations(ThemePackage theme, List<ThemeDecoration> themeDecorations, String progName){
        List<String> applyPaths = new ArrayList<>();

        String themeName = theme.getName();
        for (ThemeDecoration td: themeDecorations){
            String pattern = td.getApplyTo();

            List<ColorGroup> colorGroups = td.getColorGroups();
            if (null==colorGroups || colorGroups.isEmpty()){
                continue;
            }
            
            if (Wildcard.match(themeName, pattern)){
                theme.setColorGroups(colorGroups);
                String cgNames = colorGroups.stream().map(cg -> cg.getName()).collect(Collectors.joining(", "));
                applyPaths.add(progName+"/"+pattern+"("+cgNames+")");
            }
        }

        if ( ! applyPaths.isEmpty() ){
            String lastDecoration = applyPaths.get(applyPaths.size()-1);
            log.info("主题 '{}' 最终被修饰为 '{}', 处理路径为: {}",
                     themeName, lastDecoration, StringUtils.join(applyPaths, " -> "));
        }
    }

}
