package com.bokesoft.yigoee.tech.bootsupport.web.enhancements.resver.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.yigoee.tech.bootsupport.web.enhancements.resver.intf.IStaticResourceVersionContext;
import com.bokesoft.yigoee.tech.bootsupport.web.enhancements.resver.intf.IStaticResourceVersionContext.Traceable;

/**
 * 静态资源版本分析处理工具
 */
public class StaticUrlVersionUtils {
    private static final Logger log = LoggerFactory.getLogger(StaticUrlVersionUtils.class);

    public static String process(String rawContent, IStaticResourceVersionContext context) throws IOException {
        return processContent(rawContent, context, new Stack<String>(), new AtomicInteger(0));
    }

    /**
     * 输入原始文件内容，返回更新版本内容后的新文件内容
     */
    protected static String processContent(String rawContent, IStaticResourceVersionContext context, Stack<String> refStack, AtomicInteger traceIdBuilder) throws IOException {
        if (StringUtils.isBlank(rawContent)){
            return rawContent;
        }

        // Get references, <url, filename>
        List<String> references = getReferences(rawContent);
        if (references == null || references.size() == 0) {
            return rawContent;
        }

        // Get file version, <url, version>
        HashMap<String, String> versions = new HashMap<String, String>();
        int parentTraceId = traceIdBuilder.intValue();
        for (String url: references) {
            int childTraceId = traceIdBuilder.incrementAndGet();
            String version = processVersion(url, context, refStack, traceIdBuilder);

            if (context instanceof Traceable){
                Traceable tracer = (Traceable) context;
                tracer.appendChild(parentTraceId, childTraceId);
            }

            if (!StringUtils.isBlank(version)) {
                versions.put(url, version);
            }
        }

        // Update content
        String contentProcessed = processResult(rawContent, versions);

        return contentProcessed;
    }

    /**
     * 输入文件内容，提取所有引用文件的信息
     * 每个文件提供url+filename信息组，方便后续处理
     */
    protected static List<String> getReferences(String content) {
        final Pattern URL_PATTERN = Pattern.compile("\"[^ \\f\\n\\r\\t\\v\"'*]+\\.(js|css|less)\"");
        Matcher matcher = URL_PATTERN.matcher(content);
        
        List<String> result = new ArrayList<String>();
        while (matcher.find()) {
            String matchItem = matcher.group();
            String url = matchItem.substring(1, matchItem.length() - 1); // 去掉前后双引号
            if (!result.contains(url)) {
                result.add(url);
            }
        }

        return result;
    }

    /**
     * 计算指定资源的版本
     * @param url
     * @param context
     * @param refStack
     * @return
     */
    protected static String processVersion(String url, IStaticResourceVersionContext context, Stack<String> refStack, AtomicInteger traceIdBuilder) {
        String filename = getFilename(url);
        
        if (filename == null || filename.trim().length() == 0) {
            return null;
        }

        int traceId = traceIdBuilder.intValue();
        Traceable tracer = null;
        if (context instanceof Traceable){
            tracer = (Traceable) context;

            tracer.nodeCreated(traceId, filename);
        }

        String ver = context.getCachedVersion(filename); // read cache
        if (StringUtils.isNotBlank(ver)){
            if (null!=tracer){
                tracer.versionCreated(traceId, ver);
            }
            return ver;
        }

        // calculated?
        if (refStack.contains(filename)) {
            String dummyVer = "circular-references-" + filename; //出现循环调用
            log.warn("Circular reference found: file={}, url={} .", filename, url);

            if (null!=tracer){
                tracer.versionCreated(traceId, dummyVer);
            }

            return dummyVer;
        }

        refStack.push(filename);
        try {
            List<String> contents = context.getContents(filename);
            if (null==contents || contents.isEmpty()) { // invalid resource content
                if (null!=tracer){
                    tracer.updateInstanceCount(traceId, 0);
                }
                return null;
            }

            if (null!=tracer){
                tracer.updateInstanceCount(traceId, contents.size());
            }

            String content = StringUtils.join(contents, "\n");

            String contentProcessed = processContent(content, context, refStack, traceIdBuilder);
            ver = DigestUtils.md5Hex(contentProcessed);
            log.debug("Version of '{}': {} .", url, ver);

            if (null!=tracer){
                tracer.versionCreated(traceId, ver);
            }

            // update cache
            context.cacheVersion(filename, ver);

            return ver;
        } catch (IOException ex){
            return ExceptionUtils.rethrow(ex);
        } finally {
            refStack.pop();
        }
    }

    /**
     * 基于原始内容，识别计算出的版本结果，更新内容
     */
    protected static String processResult(String rawContent, Map<String ,String> versions) {
        String result = rawContent;
        for (String url: versions.keySet()) {
            String version = versions.get(url);
            
            String pattern = "\"" + url + "\"";
            result = result.replaceAll(pattern, "\"" + url + "?v=" + version + "\"");
        }

        return result;
    }

    private static String getFilename(String url) {
        if (StringUtils.isBlank(url)) {
            return null;
        }

        int startPos = url.lastIndexOf("/");
        if (startPos == -1) {
            return url;
        }
        if (startPos == (url.length() - 1)) {
            return null;
        }

        return url.substring(startPos + 1);
    }

}
