package com.bokesoft.yes.excel.cmd.stamp.expand.work;

import java.util.ArrayList;
import java.util.List;

import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;

import com.bokesoft.yes.excel.template.ExcelTemplateField;
import com.bokesoft.yes.excel.template.ExcelTemplateTable;
import com.bokesoft.yes.excel.template.constants.ExcelTemplateConstants;
import com.bokesoft.yes.excel.template.style.StampExcelStyleFactory;
import com.bokesoft.yes.excel.template.util.ExcelTemplateUtils;
import com.bokesoft.yes.excel.template.util.ExcelUtil;
import com.bokesoft.yigo.common.struct.IPairItem;
import com.bokesoft.yigo.meta.exceltemplate.ExcelTemplateCellType;
import com.bokesoft.yigo.struct.expand.IExpandWork;
import com.bokesoft.yigo.struct.expand.processor.DimensionItem;

/**
 * Excel中的扩展表格标题头部分处理
 * 
 * @author chenbb
 *
 */
public class ExcelTableHeadExpandWork implements IExpandWork<ExcelTemplateField, ExcelTemplateField> {
	
	private Workbook workbook = null;
	
	private StampExcelStyleFactory styleFactory = null;
	
	private ExcelTemplateTable templateTable = null;
	
	public ExcelTableHeadExpandWork(Workbook workbook, ExcelTemplateTable templateTable, StampExcelStyleFactory styleFactory) {
		this.workbook = workbook;
		this.templateTable = templateTable;
		this.styleFactory = styleFactory;
	}

	@Override
	public void doWork(List<String> fixedFieldKeys,
			List<DimensionItem<IPairItem, ExcelTemplateField, ExcelTemplateField>> colRootDimensionItems,
			List<DimensionItem<IPairItem, ExcelTemplateField, ExcelTemplateField>> rowRootDimensionItems) {
		
		int maxDeepOfDimension = 0;
		for (DimensionItem<IPairItem, ExcelTemplateField, ExcelTemplateField> rootDimenItem : colRootDimensionItems) {
			maxDeepOfDimension = Math.max(rootDimenItem.getSourceField().getDimensionCount(), maxDeepOfDimension);
			
			// 后移部分单元格,用于写入扩展列标题
			shiftExpandCells(rootDimenItem, templateTable, workbook);
			
			// 将新生成的扩展列标题写入 excel
			ArrayList<ExcelTemplateField> listExpandHeadFields = new ArrayList<>();
			int parentColIndex = rootDimenItem.getSourceField().getColIndex();
			calcChildrenExpandFields(rootDimenItem.getChildren(), listExpandHeadFields, parentColIndex);
			for (ExcelTemplateField field : listExpandHeadFields) {
				fillFieldCaption(field, workbook, styleFactory);
			}
			
			// 合并扩展根标题
			ExcelTemplateField sourceField = rootDimenItem.getSourceField();
			ExcelTemplateField detailHeadField = templateTable.getDetailHeadField(sourceField.getColIndex());
			if (detailHeadField != null) {
				detailHeadField.setColSpan(detailHeadField.getColSpan() + rootDimenItem.getLeafCount() - 1); 
			}
		}
		
		// 将未进行扩展的表格头字段，进行纵向的合并单元格处理
		int headRowSpan = templateTable.hasDetailHead() ? maxDeepOfDimension + 1 : maxDeepOfDimension;
		for (String fixedFieldKey : fixedFieldKeys) {
			ExcelTemplateField field = templateTable.getField(fixedFieldKey);
			ExcelTemplateField headField = templateTable.getDetailHeadField(field.getColIndex());
			if (headField != null && headField.getColSpan() <=1) {
				headField.setRowSpan(headRowSpan);
			}
		}
		
		// 清除表格模板中的脏列
		removeDetailHeadInTemplateTable(templateTable);
		
		// 将字段新的位置信息，记录在系统sheet中
		for (ExcelTemplateField field : templateTable.getFields()) {
			updateSytemSheet(workbook, field);
		}
		
		// 移除扩展相关的字段
		removeBaseExpandField(templateTable);
	}
	
	private void removeBaseExpandField(ExcelTemplateTable templateTable) {
		ArrayList<ExcelTemplateField> removeFieldList = new ArrayList<>();
		for (ExcelTemplateField field : templateTable.getFields()) {
			if (field.isDimensionField() || field.isExpandField()) {
				removeFieldList.add(field);
			}
		}
		for (ExcelTemplateField field : removeFieldList) {
			templateTable.remove(field);
		}
	}

	private void removeDetailHeadInTemplateTable(ExcelTemplateTable templateTable) {
		ArrayList<ExcelTemplateField> list = new ArrayList<>();
		int detailStartRow = templateTable.getStartDtlRowIndex();
		for (ExcelTemplateField field : templateTable.getFields()) {
			if (field.getRowIndex() < detailStartRow && field.getSourceType() == ExcelTemplateCellType.Const) {
				list.add(field);
			}
		}
		for (ExcelTemplateField field : list) {
			templateTable.remove(field);
		}
	}
	
	/**
	 * 后移单元格位置
	 * 
	 * @param rootDimenItem
	 * @param templateTable
	 * @param workbook
	 */
	private void shiftExpandCells(DimensionItem<IPairItem, ExcelTemplateField, ExcelTemplateField> rootDimenItem, ExcelTemplateTable templateTable, Workbook workbook) {
		int inc = rootDimenItem.getLeafCount() - 1;
		ExcelTemplateField sourceField = rootDimenItem.getSourceField();
		int startColIndex = sourceField.getColIndex();
		int firstHeadRow = templateTable.getFirstRowIndex();
		//int firstHeadRow = headField.getRowIndex();
		int lastHeadRow =  sourceField.getRowIndex();
		Sheet sheet = workbook.getSheet(sourceField.getSheetName());
		for (int rowIndex = firstHeadRow; rowIndex <= lastHeadRow; rowIndex ++) {
			int lastCellIndex = sheet.getRow(rowIndex).getLastCellNum();
			sheet.getRow(rowIndex).shiftCellsRight(startColIndex + 1, lastCellIndex, inc);
		}
	}
	
	/**
	 * 填写单元格内容
	 * 
	 * @param field
	 * @param workbook
	 * @param styleFactory
	 */
	private void fillFieldCaption(ExcelTemplateField field, Workbook workbook, StampExcelStyleFactory styleFactory) {
		Sheet sheet = ExcelUtil.getSheet(workbook, field.getSheetName());
		ExcelUtil.setCellValue(sheet, field.getRowIndex(), field.getColIndex(), field.getDefine(), field, styleFactory);
	}
	
	/**
	 * 生成列标题中的扩展信息
	 * 
	 * @param dimenItems
	 * @param listExpandHeadFields
	 */
	private void calcChildrenExpandFields(List<DimensionItem<IPairItem, ExcelTemplateField, ExcelTemplateField>> dimenItems,
			List<ExcelTemplateField> listExpandHeadFields, int parentColIndex) {
		if (dimenItems == null || dimenItems.size() == 0) {
			return;
		}
		int inc = 0;
		for (DimensionItem<IPairItem, ExcelTemplateField, ExcelTemplateField> child : dimenItems) {
			ExcelTemplateField field = child.getDimensionField();
			ExcelTemplateField targetField = ExcelTemplateField.create(child.getData().getCaption() , field.getRowIndex(), parentColIndex + inc, field.getSheetName());
			targetField.setExpandBy(field);
			listExpandHeadFields.add(targetField);
			int colSpan = child.getLeafCount();
			targetField.setColSpan(colSpan);
			calcChildrenExpandFields(child.getChildren(), listExpandHeadFields, parentColIndex + inc);
			inc += colSpan;
		}
	}
	
	/**
	 * 将字段新的位置信息，记录在系统sheet中
	 * @param workbook
	 * @param field
	 */
	private void updateSytemSheet(Workbook workbook, ExcelTemplateField field) {
		Sheet sysSheet = ExcelUtil.getSheet(workbook, ExcelTemplateConstants.YIGO_TEMPLATE);
		
		String bindCellStr = ExcelTemplateUtils.getBindCellStr(field.getSheetName(), field.getRowIndex() + 1, field.getColIndex() + 1);
		int rowIndex = findRowIndexByDefine(sysSheet, field.getDefine());
		if (rowIndex >= 0) {
			ExcelUtil.setCellFormula(sysSheet, rowIndex, ExcelTemplateConstants.BINDINGCELL_COLUMN_INDEX, bindCellStr);
			ExcelTemplateField detailHeadField =  templateTable.getDetailHeadField(field.getColIndex());
			if (detailHeadField != null) {
				bindCellStr = ExcelTemplateUtils.getBindCellStr(detailHeadField.getSheetName(), detailHeadField.getRowIndex() + 1, detailHeadField.getColIndex() + 1);
				ExcelUtil.setCellFormula(sysSheet, rowIndex, ExcelTemplateConstants.EXPAND_BINDING_CAPTION_COLUMN_INDEX, bindCellStr);
			}
		}
	}
	
	/**
	 * 查找字段在系统sheet的位置
	 * @param sysSheet
	 * @param define
	 * @return
	 */
	private int findRowIndexByDefine(Sheet sysSheet, String define) {
		if (define == null) return -1;
		Object value = null;
		for (int row = 1; row<=sysSheet.getLastRowNum(); row ++) {
			value = ExcelUtil.getCellValue(sysSheet, row, ExcelTemplateConstants.CELLSOURCE_COLUMN_INDEX);
			if (define.equals(value)) {
				return row;
			}
		}
		return -1;
	}
}
