package com.bokesoft.yes.excel.cmd.stamp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;

import org.apache.poi.ss.usermodel.Workbook;

import com.bokesoft.yes.csv.cmd.normal.IExportPostProcess;
import com.bokesoft.yes.data.service.DocumentServiceManager;
import com.bokesoft.yes.data.service.IDocumentService;
import com.bokesoft.yes.data.service.ModuleNameConstants;
import com.bokesoft.yes.excel.cmd.normal.IExport;
import com.bokesoft.yes.excel.cmd.stamp.expand.ExcelExpandMaster;
import com.bokesoft.yes.excel.cmd.stamp.exportor.DocumentExportor4Stamp;
import com.bokesoft.yes.excel.cmd.stamp.exportor.DocumentExportor4XmlTemplate;
import com.bokesoft.yes.excel.cmd.stamp.exportor.IDocumentExportor;
import com.bokesoft.yes.excel.cmd.stamp.output.strategy.BatchOutputStrategy;
import com.bokesoft.yes.excel.cmd.stamp.output.strategy.IOutputStrategy;
import com.bokesoft.yes.excel.cmd.stamp.output.strategy.MultiBatchOutputStrategy;
import com.bokesoft.yes.excel.cmd.stamp.output.strategy.SingleOutputStrategy;
import com.bokesoft.yes.excel.template.ExcelTemplate;
import com.bokesoft.yes.excel.template.style.StampExcelStyleFactory;
import com.bokesoft.yes.excel.template.util.ExcelTemplateUtils;
import com.bokesoft.yes.excel.transformer.ExcelTransformerFactory;
import com.bokesoft.yes.excel.utils.ExcelFileUtil;
import com.bokesoft.yes.struct.filedata.FileData;
import com.bokesoft.yigo.common.def.EExcelTemplateType;
import com.bokesoft.yigo.common.def.TableSourceType;
import com.bokesoft.yigo.common.ui.AbstractRuntimeUIConfig;
import com.bokesoft.yigo.meta.exceltemplate.MetaExcelWorkbook;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.struct.document.Document;
import com.bokesoft.yigo.util.ExcelUtils;

/**
 * 带印记的excel导出
 * 
 * 导出后excel文件中将存在一个隐藏的sheet用于记录导出的模板信息
 * 
 * 注: 该导出方式不支持扩展列
 * 
 * @author chenbb
 *
 */
public class ExportExcelWithStamp implements IExport {
	/** 上下文环境*/
	private DefaultContext context = null;
	/** 待导出界面的配置对象*/
	//private MetaForm metaForm4Export = null;
	/** 导出后扩展服务名*/
	private String postExportServiceName = null;
	/**自定义导出后的文件名(不包含后缀)*/
	private String exportFileName=null;

	private Document baseDocument = null;
	
	private String templateKey = null;
	
	private String curUIFormKey = null;
	
	private AbstractRuntimeUIConfig runtimeUIConfig;
	
	private MetaForm metaForm4Export = null;
	
	private EExcelTemplateType templateType = EExcelTemplateType.Single;
	
	private boolean needDownload = false;
	
	public ExportExcelWithStamp(DefaultContext context, String templateKey, EExcelTemplateType templateType, 
			String curUIFormKey, Document baseDocument, AbstractRuntimeUIConfig runtimeUIConfig, String postExportServiceName, String exportFileName,boolean needDownload) throws Throwable {
		this.context = context;
		this.templateKey = templateKey;
		this.templateType = templateType;
		this.curUIFormKey = curUIFormKey;
		this.postExportServiceName = postExportServiceName;
		this.baseDocument = baseDocument;
		this.runtimeUIConfig = runtimeUIConfig;
		this.exportFileName = exportFileName;
		this.needDownload = needDownload;
	}
	

	@Override
	public FileData exportData() throws Throwable {
		Workbook workbook = exportToWorkbook();		
		IMetaFactory metaFactory = context.getVE().getMetaFactory();
		String filePath = ExcelUtils.getExportFilePath(metaFactory, metaForm4Export.getKey());
		
		FileData fileData = new FileData();
		if (!needDownload){				
			fileData = ExcelFileUtil.writeExcel(workbook, filePath);
		}else{
			fileData = ExcelFileUtil.convertWorkbookToFileData(metaForm4Export.getKey(), exportFileName, workbook);
			fileData.setPath(filePath);
		}
		IExportPostProcess process = ExcelUtils.getPostExtProcess(context, postExportServiceName);
		if( process != null ) {
			process.process(context, fileData);
		}
		if(needDownload){
			return fileData;
		}else{
			return new FileData();
		}
	}
	
	/**
	 * 导出excel并转化成二进制格式
	 * @return
	 * @throws Throwable
	 */
	public byte[] exportToBytes() throws Throwable {
		Workbook workbook = exportToWorkbook();
		byte[] bytes = parseWorkbookToBytes(workbook);
		return bytes;
	}

	/**
	 * 导出excel, 外部调用时需要自行关闭Workbook
	 * @return
	 * @throws Throwable
	 */
	private Workbook exportToWorkbook() throws Throwable {
		Workbook workbook = null;
		IMetaFactory metaFactory = context.getVE().getMetaFactory();
		MetaForm curUIMetaForm = metaFactory.getMetaForm(curUIFormKey);
		
		if (EExcelTemplateType.SingleBatch_V0 == templateType || EExcelTemplateType.Single_V0 == templateType) {
			MetaExcelWorkbook metaWorkBook = ExcelTemplateUtils.getMetaExcelWorkbook(templateKey, templateType, curUIMetaForm, runtimeUIConfig, metaFactory);
			metaForm4Export = context.getVE().getMetaFactory().getMetaForm(metaWorkBook.getFormKey());
			workbook = export4V0(metaWorkBook);
		} else {
			ExcelTemplate excelSheetTemplate = ExcelTemplateUtils.getExcelTemplate(templateKey, templateType, curUIMetaForm, runtimeUIConfig, metaFactory);
			metaForm4Export = context.getVE().getMetaFactory().getMetaForm(excelSheetTemplate.getFormKey());
			workbook = export4Stamp(excelSheetTemplate, metaForm4Export.getProjectKey());
		}
		
		workbook.setForceFormulaRecalculation(true);
		return workbook;
	}
	
	/**
	 * 	调用 ExportExcelWithTemplate 方式进行批量导出，每个Document创建一个sheet, 导出workbook中无印记
	 * @param metaWorkBook
	 * @return
	 * @throws Throwable
	 */
	private Workbook export4V0(MetaExcelWorkbook metaWorkBook) throws Throwable {
		DocumentExportor4XmlTemplate documentExportor = new DocumentExportor4XmlTemplate(metaWorkBook, context);
		switch (templateType) {
		case SingleBatch_V0:
			// 多文档导出时候，每个Document创建一个sheet
			batchExportByMultiDocument(templateKey, metaWorkBook.getFormKey(), documentExportor, false);
			break;
		case Single_V0:
			documentExportor.export(baseDocument, "");
			break;
		default:
			break;
		}
		return documentExportor.getWorkbook();
	}
	
	/**
	 * 带印记方式导出
	 * @param excelTemplate
	 * @param projectKey
	 * @return
	 * @throws Throwable
	 */
	private Workbook export4Stamp(ExcelTemplate excelTemplate, String projectKey) throws Throwable {
		Workbook workbook = ExcelTemplateUtils.createTemplateExcelWorkbook(excelTemplate, projectKey, context.getVE().getMetaFactory());
		// 值转换工具类
		IOutputStrategy outputStrategy = null;
		IDocumentExportor documentExportor = null;
		ExcelTransformerFactory transformerFactory = new ExcelTransformerFactory();
		StampExcelStyleFactory styleFactory = new StampExcelStyleFactory(workbook, excelTemplate.isExcelFormatTemplate());
		switch (excelTemplate.getType()) {
		case Batch:
			outputStrategy = excelTemplate.getSheetCount() > 1 ? 
					  new MultiBatchOutputStrategy(workbook, styleFactory, excelTemplate) 
					: new BatchOutputStrategy(workbook, styleFactory, excelTemplate);
			documentExportor = new DocumentExportor4Stamp(outputStrategy, excelTemplate, transformerFactory, context);
			batchExport4Stamp(workbook, excelTemplate.getKey(), excelTemplate.getFormKey(), documentExportor);
			break;
		case Single:
			if (excelTemplate.hasExpandField()) {
				ExcelExpandMaster expandMaster = new ExcelExpandMaster(this.baseDocument, workbook, styleFactory, this.context);
				expandMaster.doExpand(excelTemplate);
			}
			outputStrategy = new SingleOutputStrategy(workbook, styleFactory, excelTemplate);
			documentExportor = new DocumentExportor4Stamp(outputStrategy, excelTemplate, transformerFactory, context);
			documentExportor.export(baseDocument, "");
			break;
		case MultiBatch:
			outputStrategy = new MultiBatchOutputStrategy(workbook, styleFactory, excelTemplate);
			documentExportor = new DocumentExportor4Stamp(outputStrategy, excelTemplate, transformerFactory, context);
			batchExport4Stamp(workbook, excelTemplate.getKey(), excelTemplate.getFormKey(), documentExportor);
			break;
		default:
			throw new Throwable("unsupport exprort excel template type!"); 
		}
		return workbook;
	}
	
	/**
	 * 批量导出
	 * 
	 * @param workbook
	 * @param transformerFactory
	 * @param styleFactory
	 * @throws Throwable
	 */
	private void batchExport4Stamp(Workbook workbook, String templateKey, String exportFormKey, IDocumentExportor exporter) throws Throwable {
		// 如果当前数据对象的主表为实体表,那么直接以批量格式导出该数据对象
		if (baseDocument.getMetaDataObject() != null 
				&& baseDocument.getMetaDataObject().getMainTable() != null
				&& baseDocument.getMetaDataObject().getMainTable().getSourceType() == TableSourceType.DATAOBJECT) {
			exporter.export(baseDocument, "");
		} else {
			batchExportByMultiDocument(templateKey, exportFormKey, exporter, true);
		}
	}
	
	/**
	 * 多文档批量导出
	 * @param excelSheetTemplate
	 * @param metaForm4Export
	 * @param exporter
	 * @param useCursorDataTable 设置DataTable 取数数方式，对应设置DataTable中的setCache方式，为true时 DataTable的setRowIndex,size等方法无效
	 * @throws Throwable
	 */
	private void batchExportByMultiDocument(String templateKey, String exportFormKey, IDocumentExportor exportor, boolean useCursorDataTable) throws Throwable {
		Document document = null;
		List<Long> oIDs = ExcelUtils.getOIDsFromDocument(this.baseDocument);
		IDocumentService docService = DocumentServiceManager.getInstance()
				.getDocumentService(context, ModuleNameConstants.EXCEL, "ExportExcelWithStamp", exportFormKey, templateKey);
		context.setDocument(baseDocument);
		docService.init(context, exportFormKey, oIDs, useCursorDataTable);
		while (docService.hasNext()) {
			document = docService.next();
			if (document != null) {
				context.setDocument(document);
				exportor.export(document, docService.getDocumentTitle(document));
			}
		}
		context.setDocument(baseDocument);
	}

	
	private byte[] parseWorkbookToBytes(Workbook workbook) throws IOException {
		byte[] bytes;
		ByteArrayOutputStream bos = null;
		try{
			bos = new ByteArrayOutputStream();
			workbook.write(bos);
			bytes = bos.toByteArray();
		} finally {
			if(bos != null){
				bos.close();
			}
			if(workbook != null){
				workbook.close();
			}
		}
		return bytes;
	}
}
