package com.bokesoft.yigo.meta.flatcanvas.node;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Observer;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.flatcanvas.common.AttributeItem;
import com.bokesoft.yigo.meta.flatcanvas.convertor.IContentHandler;
import com.bokesoft.yigo.meta.flatcanvas.convertor.INode;
import com.bokesoft.yigo.meta.flatcanvas.struct.FCAttrNames;
import com.bokesoft.yigo.meta.flatcanvas.struct.FCGraphType;

/**
 * 平面图中的区域
 * 
 * @author chenbinbin
 *
 */
public class MetaFCSection extends AbstractMetaFCObject {
	
	public static final String TAG_NAME = "Section";
	
	public static final List<String> ATTRS_SECTION = Arrays.asList(
			FCAttrNames.ATTR_id, FCAttrNames.ATTR_key, FCAttrNames.ATTR_caption, FCAttrNames.ATTR_visible);
	
	private MetaFCDataBind dataBindModel = new MetaFCDataBind();
	
	private MetaFCGraph graphModel = new MetaFCGraph();
	
	private MetaFCSectionCollection sectionCollection = new MetaFCSectionCollection();
	
	private MetaFCSection parent = null;
	
	public MetaFCSection() {
	}
	
	public MetaFCSection(String type) {
		setType(type);
	}
	
	/**
	 * 设置区域id属性
	 * 
	 * @param id 区域id属性
	 */
	public void setId(String id) {
		set(FCAttrNames.ATTR_id, id);
	}
	
	/**
	 * 获取区域id属性 
	 * 
	 * @return 区域id属性
	 */
	public String getId() {
		return get(FCAttrNames.ATTR_id);
	}
	
	/**
	 * 设置区域标识
	 * 
	 * @param key 区域标识
	 */
	public void setKey(String key) {
		set(FCAttrNames.ATTR_key, key);
	}
	
	/**
	 * 获取区域标识
	 * 
	 * @return 区域标识
	 */
	public String getKey() {
		return get(FCAttrNames.ATTR_key);
	}
	
	/**
	 * 设置区域名称
	 * 
	 * @param caption 区域名称
	 */
	public void setCaption(String caption) {
		set(FCAttrNames.ATTR_caption, caption);
	}
	
	/**
	 * 获取区域名称
	 * 
	 * @return 区域名称
	 */
	public String getCaption() {
		return get(FCAttrNames.ATTR_caption);
	}
	
	/**
	 * 设置边框线颜色
	 * 
	 * @param red 颜色的R值
	 * @param green 颜色的G值
	 * @param blue 颜色的B值
	 */
	public void setStroke(int red, int green, int blue) {
		graphModel.setStroke(red, green, blue);
	}
	
	/**
	 * 设置填充色
	 * 
	 * @param red 颜色的R值
	 * @param green 颜色的G值
	 * @param blue 颜色的B值
	 */
	public void setFillColor(int red, int green, int blue) {
		graphModel.setFill(red, green, blue);
	}
	
	/**
	 * 设置图形类型
	 * 
	 * @param type 图形类型
	 * 
	 * @see FCGraphType
	 */
	public void setType(String type) {
		graphModel.set(FCAttrNames.ATTR_type, type);
	}
	
	/**
	 * 获取图形类型
	 * 
	 * @return 图形类型
	 * 
	 * @see FCGraphType
	 */
	public String getType() {
		return TypeConvertor.toString(graphModel.get(FCAttrNames.ATTR_type));
	}
	
	/**
	 * 获取是否可见属性
	 * 
	 * @return 是否可见属性
	 */
	public String getVisible() {
		return TypeConvertor.toString(this.get(FCAttrNames.ATTR_visible));
	}
	
	/**
	 * 添加区域
	 * 
	 * @param metaSection 区域配置对象
	 */
	public void addSection(MetaFCSection metaSection) {
		sectionCollection.addItem(metaSection);
	}
	
	/**
	 * 移除区域
	 * 
	 * @param section 区域配置对象
	 */
	public void removeSection(MetaFCSection section) {
		sectionCollection.remove(section);
	}
	
	/**
	 * 设置父区域
	 * 
	 * @param parent 父区域配置对象
	 */
	public void setParent(MetaFCSection parent) {
		this.parent = parent;
	}
	
	/**
	 * 获取父区域
	 * 
	 * @return 父区域配置对象
	 */
	public MetaFCSection getParent() {
		return this.parent;
	}
	
	/**
	 * 获取区域图形配置信息
	 * 
	 * @return 图形配置信息
	 */
	public MetaFCGraph getGraph(){
		return this.graphModel;
	}
	
	/**
	 * 获取区域绑定的数据信息
	 * 
	 * @return 数据配置信息
	 */
	public MetaFCDataBind getDataBind() {
		return this.dataBindModel;
	}
	
	/**
	 * 获取指定子区域
	 * 
	 * @param key 标识
	 * @return 子区域
	 */
	public MetaFCSection getSection(String key) {
		return sectionCollection.getSection(key);
	}
	
	/**
	 * 通过序号获取指定子区域
	 * 
	 * @param index 序号
	 * @return 指定位置的子区域
	 */	
	public MetaFCSection getSection(int index) {
		return this.sectionCollection.get(index);
	}
	
	/**
	 * 获取子区域集合
	 * 
	 * @return 子区域集合
	 */
	public Collection<MetaFCSection> sections() {
		return this.sectionCollection.values();
	}
	
	/**
	 * 获取当前区域TableKey
	 * 
	 * @return 当前区域数据表标志
	 */
	public String getCurTableKey() {
		return this.dataBindModel.getCurTableKey();
	}
	
	/**
	 * 获取TableKey,递归根区域
	 * 
	 * @return 区域数据表标志
	 */
	public String getTableKey() {
		MetaFCSection tmp = this;
		while (tmp != null) {
			String curTableKey = tmp.getCurTableKey();
			if (!StringUtil.isBlankOrNull(curTableKey)) {
				return tmp.getCurTableKey();
			}
			tmp = tmp.getParent();
		}
		return null;
	}
	
	/**
	 * 
	 * 是否为表格区域中的子孙区域
	 * 
	 * @return 是否为表格区域中的子孙区域
	 */
	public boolean isSectionInTable() {
		String tableKey = this.getCurTableKey();
		return StringUtil.isBlankOrNull(tableKey) && !StringUtil.isBlankOrNull(this.getTableKey());
	}

	@Override
	public List<String> getPropItemKeys() {
		return ATTRS_SECTION;
	}
	
	@Override
	public String getTagName() {
		return TAG_NAME;
	}

	@Override
	protected void loadChild(Element elem) {
		NodeList childNodes = elem.getChildNodes();
		for (int i=0; i< childNodes.getLength(); i++) {
			Node node = childNodes.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE) {
				Element element = (Element) node;
				if (MetaFCGraph.TAG_NAME.equalsIgnoreCase(element.getTagName())) {
					graphModel.loadFromElement(element);
				} else if (MetaFCDataBind.TAG_NAME.equalsIgnoreCase(element.getTagName())) {
					dataBindModel.loadFromElement(element);
				} else if (MetaFCSectionCollection.TAG_NAME.equalsIgnoreCase(element.getTagName())) {
					sectionCollection.loadFromElement(element);
					for (MetaFCSection section: sectionCollection.values()) {
						section.setParent(this);
					}
				}
			}
		}
	}
	
	@Override
	protected boolean isCollection() {
		return false;
	}

	@Override
	protected boolean saveChild(IContentHandler saveHandler, INode node) {
		this.graphModel.saveTo(saveHandler, node);
		this.dataBindModel.saveTo(saveHandler, node);
		this.sectionCollection.saveTo(saveHandler, node);
		return true;
	}

	public void addAttrObserver(Observer observer) {
		super.addAttrObserver(observer);
		graphModel.addAttrObserver(observer);
	}
	
	@Override
	public List<AttributeItem> getPropItems() {
		List<AttributeItem> items = new ArrayList<AttributeItem>();
		List<AttributeItem> sectionPropItems = super.getPropItems();
		for (AttributeItem item: sectionPropItems) {
			items.add(item);
		}
		
		List<AttributeItem> graphPropItems = graphModel.getPropItems();
		for (AttributeItem item: graphPropItems) {
			items.add(item);
		}
		
		dataBindModel.setIsSectionInTable(this.isSectionInTable());
		List<AttributeItem> dataPropItems = dataBindModel.getPropItems();
		for (AttributeItem item: dataPropItems) {
			items.add(item);
		}
		return items;
	}
}
