package com.bokesoft.yigo.meta.base;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map.Entry;

import org.apache.commons.lang3.SerializationUtils;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.SerializationException;
import com.bokesoft.yes.common.struct.StringHashMap;
import com.bokesoft.yes.common.util.Callback;
import com.bokesoft.yigo.common.json.JSONSerializable;

/**
 * AbstractMetaObject表示所有元数据的基类,其主要用于提供基础的用于持久化需要的机制.
 * <p>其主要包括以下基础的机制:</p>
 * <ul>
 * <li>取得标记(TagName)
 * <li>取得所有子对象的列表
 * <li>遍历方法
 * <li>通过标记新增子对象
 * </ul>
 * @author 王元和
 * @since YES1.0
 */
public abstract class AbstractMetaObject implements JSONSerializable, Serializable {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	protected StringHashMap<String> attributeMap = null;

	public AbstractMetaObject() {
	}
	
	/**
	 * 返回所有子对象的集合(如果当前对象是Collection，Collection中的元素不包含在内)
	 * @param list 子配置对象列表
	 */
	public abstract void getChildMetaObjects(LinkedList<Object> list);
	
	/**
	 * 取得对象标识名称
	 * @return 对象标记名称 
	 */
	public abstract String getTagName();
	
	/**
	 * 遍历所有的子对象，并对子对象进行处理
	 * @param env 环境
	 * @param doc 文档
	 * @param node 节点
	 * @param flag 处理标志，一般确定为保存和载入，根据外部实现类来解决，这里仅用于传递参数
	 * @param runType 运行类型，用于判断是否处理设计器状态还是程序状态
	 * @throws MetaException 配置异常
	 */
	protected void traversalChildMetaObjects(IMetaEnv env, Object doc, Object node, Object flag, int runType) throws MetaException {
		LinkedList<Object> list = new LinkedList<Object>();
		getChildMetaObjects(list);
		Iterator<Object> itChild = list.iterator();
		Object childObject = null;
		while ( itChild.hasNext() ) {
			childObject = itChild.next();
			if ( childObject != null ) {
				if ( childObject instanceof AbstractMetaObject ) {
					AbstractMetaObject childMetaObject = (AbstractMetaObject)childObject;
					Object childNode = env.prepare(childMetaObject, node);
					childMetaObject.traversal(env, doc, childNode, flag, runType);
				} else if (childObject instanceof Collection<?>){
					Iterator<?> itItem = ((Collection<?>) childObject).iterator();
					while ( itItem.hasNext() ) {
						Object tmpItem = itItem.next();
						AbstractMetaObject childMetaObject = (AbstractMetaObject)tmpItem;
						Object childNode = env.prepare(childMetaObject, node);
						childMetaObject.traversal(env, doc, childNode, flag, runType);
					}
				}
			}
		}
	}
	
	protected void traversalCollectionObjects(IMetaEnv env, Object doc, Object node, Object flag, int runType) throws MetaException {
	}
	
	/**
	 * 遍历所有的子对象，并对自身及子对象执行操作
	 * @param env 遍历外部环境
	 * @param doc 文档
	 * @param node 抽象的节点
	 * @param flag 处理标志，一般确定为保存和载入，根据外部实现类来解决，这里仅用于传递参数
	 * @param runType 运行类型，用于判断是否处理设计器状态还是程序状态
	 * @throws MetaException 如果没有Action定义，抛出无Action定义异常
	 */
	public void traversal(IMetaEnv env, Object doc, Object node, Object flag, int runType) throws MetaException {
		IMetaAction<AbstractMetaObject> action = env.getAction(this, node);
		if ( action == null ) {
			throw new MetaException(MetaException.NO_ACTION_DEFINED, "No action defined for " + getTagName());
		}
		action.process(env, doc, node, this, flag, runType);
		traversalCollectionObjects(env, doc, node, flag, runType);
		traversalChildMetaObjects(env, doc, node, flag, runType);
	}

	public void preProcessMetaObject(IMetaEnv env, Object doc, Object node, Object flag, int runType) 
			throws MetaException {
		if ( needPreLoad() ) {
			IMetaAction<AbstractMetaObject> action = env.getAction(this, node);
			if ( action == null ) {
				throw new MetaException(MetaException.NO_ACTION_DEFINED, "No action defined for " + this.getTagName());
			}
			action.process(env, doc, node, this, flag, runType);
		}
	}
	
	/**
	 * 是否需要检查继承,子类覆盖
	 * @return 是否需要检查继承
	 */
	public boolean needCheckExtend(){
		return false;
	}
	
	/**
	 * 是否需要预加载，由于需要Key来建立映射，所以属性需要预加载
	 * @return 是否需要预加载，由具体实现类覆盖，默认不需要预加载
	 */
	public boolean needPreLoad() {
		return false;
	}
	
	/**
	 * 是否有独立节点，用于确定节点是否寄生于别的节点内
	 * @return 是否有独立节点，由具体实现类覆盖，默认需要独立节点
	 */
	public boolean isStandaloneNode() {
		return true;
	}
	
	/**
	 * 是否系统自动生成的元数据
	 * @return 标志，默认值为false，少数地方覆盖
	 */
	public boolean isAutoGen() {
		return false;
	}
	
	public abstract AbstractMetaObject createChildMetaObject(IMetaEnv env, Object doc, Object node, String tagName, Object flag, int runType) throws MetaException;
	
	/**
	 * 复制一个元数据对象
	 * @return 新的元数据对象
	 */
	@Override
	public abstract AbstractMetaObject clone();

	/**
	 * 深度复制一个元数据对象
	 * @return 新的元数据对象
	 */

	public  AbstractMetaObject depthClone() {
		return SerializationUtils.clone(this);
	}

	/**
	 * 新建一个空的元数据对象
	 * @return 新的空元数据对象
	 */
	public abstract AbstractMetaObject newInstance();
	
	/**
	 * 后期处理，用于在元数据创建之后，允许有机会对元数据进行一些扩展处理，比如生成一些汇总类信息。
	 * @param flag 处理标志，用于父对象向子传对象传递一些特定的标志
	 * @param callback 回调对象
	 * @throws MetaException 配置结构错误相关的异常
	 */
	public void doPostProcess(int flag, Callback<AbstractMetaObject,Boolean> callback) throws MetaException {
	}
	
	/**
	 * 向一个列表中加入所有数组中的项目的辅助函数，内部使用
	 * @param list 列表
	 * @param items 需要加入的项目数组
	 */
	public void addAll(LinkedList<Object> list, Object[] items) {
		int length = items.length;
		for ( int i = 0; i<length; ++i ) {
			if ( items[i] != null ) {
				list.add(items[i]);
			}
		}
	}
	
	/**
	 * 向一个列表中加入所有数组中项目的辅助函数，内部使用
	 * @param list 列表
	 * @param items 需要加入的项目数组
	 */
	public void addAll2(LinkedList<Object> list, Object... items) {
		int length = items.length;
		for ( int i = 0; i<length; ++i ) {
			if ( items[i] != null ) {
				list.add(items[i]);
			}
		}
	}
	
	/**
	 * 设计器运行时,将节点元素的原始属性及其对应值存入
	 * @param key 标识
	 * @param value 值
	 */
	public void addAttribute(String key,String value) {
		if( attributeMap == null ) {
			attributeMap = new StringHashMap<String>();
		}
		attributeMap.put(key, value);
	}
	
	/**
	 * 获取单个属性值
	 * @param key 标识
	 * @return 值
	 */
	public String getAttribute(String key) {
		if( attributeMap == null ) {
			return null;
		}
		return attributeMap.get(key);
	}
	
	/**
	 * 获取保存属性映射集合的迭代器
	 * @return 属性集合迭代器
	 */
	public Iterator<Entry<String, String>> attIterator() {
		if( attributeMap == null ) {
			attributeMap = new StringHashMap<String>();
		}
		return attributeMap.entrySet().iterator();
	}
	
	@Override
	public JSONObject toJSON() throws SerializationException {
		return null;
	}
	
	@Override
	public void fromJSON(JSONObject json) throws SerializationException {
		
	}
	
	
	public void traversalSingle(IMetaEnv env, Object doc, Object node, Object flag, int runType) throws MetaException {
		IMetaAction<AbstractMetaObject> action = env.getAction(this, node);
		if ( action == null ) {
			throw new MetaException(MetaException.NO_ACTION_DEFINED, "No action defined for " + getTagName());
		}
		action.process(env, doc, node, this, flag, runType);
	}
}
