package com.bokesoft.yigo.meta.flatcanvas.node;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.flatcanvas.common.AttributeItem;
import com.bokesoft.yigo.meta.flatcanvas.convertor.IContentHandler;
import com.bokesoft.yigo.meta.flatcanvas.convertor.INode;
import com.bokesoft.yigo.meta.flatcanvas.struct.FCAttrNames;
import com.bokesoft.yigo.meta.flatcanvas.struct.FCGraphType;
import com.bokesoft.yigo.meta.flatcanvas.util.FCUtil;

public class MetaFCGraph extends AbstractMetaFCNode {
	
	public static final String TAG_NAME = "Graph";
	
	private static final List<String> ATTRS_COMMON = Arrays.asList(
			FCAttrNames.ATTR_type,
			FCAttrNames.ATTR_fill, FCAttrNames.ATTR_stroke, FCAttrNames.ATTR_strokeWidth,
			FCAttrNames.ATTR_strokeOpacity, FCAttrNames.ATTR_ondblclick, FCAttrNames.ATTR_showcaption, FCAttrNames.ATTR_statufill);
	
	private static HashMap<String, List<String>> ATTRS_GRAPH_MAP = new HashMap<String, List<String>>();
	static {
		ATTRS_GRAPH_MAP.put(FCGraphType.RECT, Arrays.asList(FCAttrNames.ATTR_x, FCAttrNames.ATTR_y, FCAttrNames.ATTR_width, FCAttrNames.ATTR_height));
		ATTRS_GRAPH_MAP.put(FCGraphType.CIRCLE, Arrays.asList(FCAttrNames.ATTR_x, FCAttrNames.ATTR_y, FCAttrNames.ATTR_r));
		ATTRS_GRAPH_MAP.put(FCGraphType.TEXT, Arrays.asList(FCAttrNames.ATTR_x, FCAttrNames.ATTR_y, FCAttrNames.ATTR_text));
		ATTRS_GRAPH_MAP.put(FCGraphType.POLYGON, Arrays.asList(FCAttrNames.ATTR_points));
		ATTRS_GRAPH_MAP.put(FCGraphType.POLYLINE, Arrays.asList(FCAttrNames.ATTR_from, FCAttrNames.ATTR_to, FCAttrNames.ATTR_points));
	}
	
	private MetaFCTextCollection textCollection = new MetaFCTextCollection();
	
	/**
	 * 获取图形相关的点坐标
	 * 
	 * @return 相关的点坐标
	 */
	public String getPoints() {
		return get(FCAttrNames.ATTR_points);
	}
	
	/**
	 * 设置图形相关的点坐标
	 * 
	 * @param points 相关的点坐标
	 */
	public void setPoints(String points) {
		set(FCAttrNames.ATTR_points, points);
	}
	
	/**
	 * 设置起始节点标识
	 * 
	 * @param start 起始节点标识
	 */
	public void setFrom(String start) {
		set(FCAttrNames.ATTR_from, start);
	}
	
	/**
	 * 获取起始节点标识
	 * 
	 * @return 起始节点标识
	 */
	public String getFrom() {
		return get(FCAttrNames.ATTR_from);
	}
	
	/**
	 * 设置结束节点标识
	 * 
	 * @param end 结束节点标识
	 */
	public void setTo(String end) {
		set(FCAttrNames.ATTR_to, end);
	}
	
	/**
	 * 获取结束节点标识
	 * 
	 * @return 结束节点标识
	 */
	public String getTo() {
		return get(FCAttrNames.ATTR_to);
	}
	
	/**
	 * 获取图形半径
	 * 
	 * @return 图形半径
	 */
	public Integer getR() {
		return TypeConvertor.toInteger(get(FCAttrNames.ATTR_r));
	}
	
	/**
	 * 设置图形半径
	 * 
	 * @param r 图形半径
	 */
	public void setR(int r) {
		set(FCAttrNames.ATTR_r, r);
	}
	
	/**
	 * 获取图形宽度
	 * 
	 * @return 图形宽度
	 */
	public Integer getWidth() {
		return TypeConvertor.toInteger(get(FCAttrNames.ATTR_width));
	}
	
	/**
	 * 设置图形宽度
	 * 
	 * @param width 图形宽度
	 * 
	 */
	public void setWidth(int width) {
		set(FCAttrNames.ATTR_width, width);
	}
	
	/**
	 * 获取图形高度
	 * 
	 * @return 图形高度
	 */
	public Integer getHeight() {
		return TypeConvertor.toInteger(get(FCAttrNames.ATTR_height));
	}
	
	/**
	 * 设置图形高度
	 * 
	 * @param height 图形高度
	 */
	public void setHeight(int height) {
		set(FCAttrNames.ATTR_height, height);
	}
	
	/**
	 * 获取图形类型
	 * 
	 * @return 图形类型
	 * 
	 * @see FCGraphType
	 */
	public String getType() {
		return this.get(FCAttrNames.ATTR_type);
	}
	
	/**
	 * 设置图形类型
	 * 
	 * @param type 图形类型
	 * 
	 * @see FCGraphType
	 */
	public void setType(String type) {
		set(FCAttrNames.ATTR_type, type);
	}
	
	/**
	 * 获取文本信息
	 * 
	 * @return 文本信息
	 */
	public String getText() {
		return this.get(FCAttrNames.ATTR_text);
	}
	
	/**
	 * 设置文本信息
	 * 
	 * @param text 文本信息
	 */
	public void setText(String text) {
		set(FCAttrNames.ATTR_text, text);
	}
	
	/**
	 * 设置图形边框线颜色
	 * 
	 * @param stroke 边框线颜色
	 */
	public void setStroke(double red, double green, double blue) {
		set(FCAttrNames.ATTR_stroke, FCUtil.colorValueToHex(red, green, blue));
	}
	
	/**
	 * 获取图形边框线颜色
	 * 
	 * @return 边框线颜色
	 */
	public Object getStroke() {
		return get(FCAttrNames.ATTR_stroke);
	}
	
	/**
	 * 设置图形线宽
	 * 
	 * @param width 图形线宽
	 */
	public void setStrokeWidth(double width) {
		set(FCAttrNames.ATTR_strokeWidth, width);
	}
	
	/**
	 * 获取图形线宽
	 * 
	 * @return 图形线宽
	 */
	public double getStrokeWidth() {
		return TypeConvertor.toDouble(get(FCAttrNames.ATTR_strokeWidth));
	}
	
	/**
	 * 获取图形边线透明度
	 * 
	 * @return 形边线透明度(0~1)
	 */
	public double getStrokeOpacity() {
		return TypeConvertor.toDouble(get(FCAttrNames.ATTR_strokeOpacity));
	}
	
	/**
	 * 设置图形边线透明度
	 * 
	 * @param opacity 形边线透明度(0~1)
	 */
	public void setStrokeOpacity(double opacity) {
		set(FCAttrNames.ATTR_strokeOpacity, opacity);
	}
	
	/**
	 * 设置图形填充
	 * 
	 * @param red 图形填充颜色的R值
	 * @param green 图形填充颜色的G值
	 * @param blue 图形填充颜色的B值
	 */
	public void setFill(double red, double green, double blue) {
		set(FCAttrNames.ATTR_fill, FCUtil.colorValueToHex(red, green, blue));
	}
	
	/**
	 * 获取图形填充 
	 * 
	 * @return 图形填充
	 */
	public Object getFill() {
		return get(FCAttrNames.ATTR_fill);
	}
	
	/**
	 * 获取状态填充
	 * @return 状态填充
	 */
	public String getStatusFill() {
		return get(FCAttrNames.ATTR_statufill);
	}
	
	/**
	 * 设置图形双击事件
	 * 
	 * @param sFormula 双击事件
	 */
	public void setOnDblClick(String sFormula) {
		set(FCAttrNames.ATTR_ondblclick, sFormula);
	}
	
	/**
	 * 获取图形双击事件
	 * 
	 * @return 双击事件
	 */
	public String getOnDblClick() {
		return get(FCAttrNames.ATTR_ondblclick);
	}
	
	@Override
	public List<String> getPropItemKeys() {
		String type = this.getType();
		List<String> listGraphAttr = ATTRS_GRAPH_MAP.get(type);
		if (listGraphAttr == null) {
			listGraphAttr = new ArrayList<String>();
		}
		List<String> list = FCUtil.concatList(listGraphAttr, ATTRS_COMMON);
		return list;
	}
	
	@Override
	public List<AttributeItem> getPropItems() {
		List<AttributeItem> items = new ArrayList<AttributeItem>();
		List<String> list = getPropItemKeys();
		for (String attrName : list) {
			AttributeItem item = getAttrItem(attrName);
			if (item == null) {
				set(attrName, "");
			}
			items.add(getAttrItem(attrName));
		}
		return items;
	}


	@Override
	public String getTagName() {
		return TAG_NAME;
	}

	@Override
	protected void loadChild(Element elem) {
		NodeList childNodes = elem.getChildNodes();
		for (int i=0; i< childNodes.getLength(); i++) {
			Node node = childNodes.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE) {
				Element element = (Element) node;
				if (MetaFCTextCollection.TAG_NAME.equalsIgnoreCase(element.getTagName())) {
					textCollection.loadFromElement(element);
				}
			}
		}
	}

	@Override
	protected boolean isCollection() {
		return false;
	}

	@Override
	protected boolean saveChild(IContentHandler saveHandler, INode node) {
		return this.textCollection.saveTo(saveHandler, node);
	}

	public Collection<MetaFCText> texts() {
		return textCollection.values();
	}
	
	public void addText(MetaFCText text) {
		textCollection.addItem(text);
	}
	
	public void removeText(MetaFCText text) {
		textCollection.remove(text);
	}
}
