package com.bokesoft.yigo.meta.base;

import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Consumer;

import com.bokesoft.yes.common.struct.StringLinkedHashMap;
/**
 * GenericKeyCollection定义[key, value]关联的集合类对象，其存储的子对象必须从KeyPairMetaObject派生。
 * @author wangyh
 *
 * @param <T> KeyPairMetaObject的派生类
 */
public abstract class GenericKeyCollection<T extends KeyPairMetaObject> extends
		AbstractMetaObject implements Iterable<T> {

	/**
	 *
	 */
	private static final long serialVersionUID = 1L;

	protected StringLinkedHashMap<T> elementMap = null;

	public GenericKeyCollection() {
		elementMap = new StringLinkedHashMap<T>();
	}

	/**
	 * 增加元素
	 * @param element 子元素
	 * @return 新增的元素
	 */
	public T add(T element) {
		synchronized(elementMap) {
			elementMap.put(element.getKey(), element);
		}
		return element;
	}

	/**
	 * 批量添加元素
	 * @param items 元素集合
	 */
	public void addAll(List<T> items) {
		for( T t : items ) {
			this.add(t);
		}
	}

	/**
	 * 指定key删除元素
	 * @param key 元素的键值
	 */
	public void remove(String key) {
		synchronized(elementMap) {
			elementMap.remove(key);
		}
	}

	/**
	 * 根据key取得相应的元素
	 * @param key 元素的键值
	 * @return 如果存在key指定的元素则返回，否则返回null
	 */
	public T get(String key) {
		synchronized(elementMap) {
			return elementMap.get(key);
		}
	}

	/**
	 * 判断是否存在key指定的元素
	 * @param key 元素的键值
	 * @return 如果存在key指定的元素则返回true，否则返回false
	 */
	public boolean containsKey(String key) {
		synchronized(elementMap) {
			return elementMap.containsKey(key);
		}
	}

	@Override
	public Iterator<T> iterator() {
		synchronized(elementMap) {
			return elementMap.values().iterator();
		}
	}

	/**
	 * 返回键值对条目的迭代器
	 * @return 键值对条目的迭代器
	 */
	public Iterator<Entry<String, T>>  entryIterator() {
		synchronized(elementMap) {
			return elementMap.entrySet().iterator();
		}
	}

	/**
	 * 返回包含元素的数量
	 * @return 集合中元素的数理
	 */
	public int size() {
		synchronized(elementMap) {
			return elementMap.size();
		}
	}

	/**
	 * 判断集合是否为空
	 * @return 如果集合为空返回true，否则返回false
	 */
	public boolean isEmpty() {
		synchronized(elementMap) {
			return elementMap.isEmpty();
		}
	}

	/**
	 * 根据序号返回指定位置的元素
	 * @param index 序号
	 * @return index所指位置的元素
	 */
	@SuppressWarnings("unchecked")
	public T get(int index) {
		return (T)elementMap.values().toArray()[index];
	}

	/**
	 * 返回第一个元素
	 * @return 第一个元素
	 */
	public T first() {
		synchronized(elementMap) {
			return this.elementMap.entrySet().iterator().next().getValue();
		}
	}

	/**
	 * 清除所有元素
	 */
	public void clear() {
		synchronized(elementMap) {
			elementMap.clear();
		}
	}

	/**
	 * 获取所有Key
	 * @return 所有Key
	 */
	public Set<String> keySet() {
		synchronized(elementMap) {
			return elementMap.keySet();
		}
	}

	@Override
	public void traversalCollectionObjects(IMetaEnv env, Object doc, Object node,
										   Object flag, int runType) throws MetaException {
		Iterator<Entry<String, T>> itEntry = elementMap.entrySet().iterator();
		Entry<String, T> entry = null;
		while (itEntry.hasNext()) {
			entry = itEntry.next();
			T element = entry.getValue();
			if ( !element.isAutoGen() ) {
				Object childNode = env.prepare(element, node);
				element.traversal(env, doc, childNode, flag, runType);
			}
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public AbstractMetaObject clone() {
		GenericKeyCollection<T> newObject = (GenericKeyCollection<T>)newInstance();
		Iterator<Entry<String, T>> it = elementMap.entrySet().iterator();
		Entry<String, T> entry = null;
		while ( it.hasNext() ) {
			entry = it.next();
			newObject.add((T)entry.getValue().clone());
		}
		return newObject;
	}

	public void forEach(Consumer<? super T> action) {
		synchronized(elementMap) {
			elementMap.values().forEach(action);
		}
	}
}
