package com.bokesoft.yes.meta.persist.dom.solution;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.base.IMetaResourceFilter;
import com.bokesoft.yigo.meta.solution.MetaProject;

/**
 * BaseMetaScanLoad类用于扫描工程下面的所有配置中的元数据的基础信息
 */
public abstract class BaseMetaScanLoad {

	/** 元文件读写代理类 */
	protected IMetaResolver resolver = null;

	/** 工程元文件对象 */
	protected MetaProject metaProject = null;

	/** 扫描根目录的相对路径(相对于工程Project的目录) */
	protected String resourcePath = null;
	
	protected String fileFilter = ".xml";
	
	protected Object context = null;
	
	protected IMetaResourceFilter filter = null;

	public BaseMetaScanLoad(IMetaResolver resolver,
			MetaProject metaProject, String resourcePath, Object context) {
		this.resolver = resolver;
		this.metaProject = metaProject;
		this.resourcePath = resourcePath;
		this.context = context;
	}
	
	public void setFilter(IMetaResourceFilter filter) {
		this.filter = filter;
	}
	
	public IMetaResourceFilter getFilter() {
		return this.filter;
	}

	protected Object getRoot() {
		return null;
	}

	protected ExecutorService service = null;
	protected List<Future<?>> futureTaskList = null;
	
	public void load() throws Throwable {   
		try {
			preLoad();
			loadImpl();
			postLoad();
		}finally {
			if(service != null) {
				service.shutdown();
			}
		}

	}

	protected void preLoad() throws Throwable{
		service = Executors.newFixedThreadPool(5);
		futureTaskList = new ArrayList<>();
	}
	
	protected void loadImpl() throws Throwable{
		scan(getRoot(), resourcePath);
	}
	
	protected void postLoad() throws Throwable {
		if(futureTaskList == null) {
			return;
		}
		
		for (Future<?> task : futureTaskList) {
			task.get();
		}
	}
	
	/**
	 * 载入目录信息
	 * @param parent 上级对象
	 * @param path 上级目录
	 * @param filePath 当前目录
	 * @return 目录信息
	 * @throws Throwable 如果出错抛出的异常，一般为文件异常或XML格式异常
	 */
	protected Object loadFolder(Object parent, String path, String filePath) throws Throwable {
		return null;
	}
	
	/**
	 * 扫描指定目录以及所有子目录
	 * @param parent 上级对象
	 * @param filePath 相对路径(相对于工程Project的目录)
	 * @throws Throwable 如果出错抛出的异常，一般为文件异常或XML格式异常
	 */
	protected void scan(Object parent, String filePath) throws Throwable {
		List<String> resourceList = new ArrayList<String>();
		List<String> nameList = new ArrayList<String>();
		List<Boolean> flagList = new ArrayList<Boolean>();
		resolver.listResource(filePath, fileFilter, resourceList, nameList, flagList);
		
		if (resourceList.size() > 0) {
			Iterator<String> itPath = resourceList.iterator();
			Iterator<String> itName = nameList.iterator();
			Iterator<Boolean> itFlag = flagList.iterator();

			Future<?> futureTask;
			while (itPath.hasNext()) {
				final String path = itPath.next();
				final String name = itName.next();
				final Boolean flag = itFlag.next();
				if (flag) {
					String foldPath = filePath + resolver.getSeparator() + name;
					Object foldObject = loadFolder(parent, name, foldPath);
					scan(foldObject, foldPath);
				} else {
					String tmpPath = path.toLowerCase();
					if (tmpPath.endsWith(getFileFilter())) {
						if (filter == null || filter.isAccepted(name)) {
							String fp = resolver.isListAbsolutePath() ? path : filePath + resolver.getSeparator() + path;
				
							if(service != null) {
								futureTask = service.submit(() -> {
									try {
										loadMetaData(parent, filePath, fp, name);
									} catch (Throwable e) {
										LogSvr.getInstance().error("read file error : " + name, e);
										throw new RuntimeException(e);
									}
								});
			
								futureTaskList.add(futureTask);
							}else {
								loadMetaData(parent, filePath, fp, name);
							}
						}
					}
				}
			}
		}
	}
	
	protected String getFileFilter() {
		return this.fileFilter;
	}

	/**
	 * 载入元文件
	 * 
	 * @param parent 父对象
	 * @param parentPath 当前文件所在目录
	 * @param filePath 相对路径(相对于工程Project的目录)
	 * @param fileName 文件的名称
	 * @throws Exception 如果出错抛出的异常，一般为文件异常或XML格式异常
	 */
	public abstract void loadMetaData(Object parent, String parentPath, String filePath, String fileName) throws Exception;

}
