package com.bokesoft.yigo.struct.i18n;

import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yigo.meta.base.AbstractMetaObject;
import com.bokesoft.yigo.meta.commondef.MetaOperation;
import com.bokesoft.yigo.meta.commondef.MetaOperationCollection;
import com.bokesoft.yigo.meta.entry.MetaEntry;
import com.bokesoft.yigo.meta.entry.MetaEntryItem;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.form.MetaFormProfile;
import com.bokesoft.yigo.meta.form.component.MetaComponent;
import com.bokesoft.yigo.meta.form.component.grid.MetaGridColumn;
import com.bokesoft.yigo.meta.solution.MetaLang;
import com.bokesoft.yigo.meta.solution.MetaProject;
import com.bokesoft.yigo.meta.solution.MetaSolution;
import com.bokesoft.yigo.meta.util.MetaUtil;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;

/**
 * 从数据库中的表Sys_i18n_info中初始化形成的对象，以提供后续翻译调用。<br/>
 * 1、该对象中存在数据行的Map集合，可以方便查找对应的数据。<br/>
 * 2、数据行的字段为：SolutionKey、ProjectKey、FormKey、TranslationKey、TranslationType、Lang、Caption、CaptionFull、ManualCaption、ManualCaptionFull<br/>
 * 3、数据行中，要求SolutionKey、ProjectKey、FormKey为空时，必须填入*字符进行表示
 */
public class I18NInfo {
    /**
     * 基类语种-中文
     */
    public static final String BaseLang_ZH = "zh-CN";
    /**
     * 基类语种-英文
     */
    public static final String BaseLang_EN = "en-US";

    /**
     * 词条类型-通用
     */
    public static final String TranslationType_Common = "Common";
    /**
     * 词条类型-组件
     */
    public static final String TranslationType_Component = "Component";
    /**
     * 词条类型-操作
     */
    public static final String TranslationType_Operation = "Operation";
    /**
     * 词条类型-菜单入口
     */
    public static final String TranslationType_Entry = "Entry";
    /**
     * 词条类型-工作流
     */
    public static final String TranslationType_BPM = "BPM";
    /**
     * 词条类型-数据元素/域
     */
    public static final String TranslationType_DataElement = "DataElement";

    /**
     * i18nInfo的系统数据库表名称
     */
    public static final String i18nInfoTableKey = "YES_Core_I18N";
    /**
     * 单例实例对象
     */
    private static I18NInfo instance;
    /**
     * 从数据表中获取的数据形成的Map集合，通过TranslationType值进行分类
     */
    protected HashMap<String, I18NInfoMap> dataMap = new HashMap<>();
    /**
     * I18NInfo的相关信息缓存对象
     */
    protected I18NInfoCacheInterface cache;
    /**
     * 版本号，初始化结束后从缓存中获取版本号，保存更新对象后，版本号加1，并更新到缓存
     */
    protected int version = -1;
    /**
     * 最后修改的数据行的修改时间，格式为：年月日时分秒毫秒，比如20250408102553129
     */
    protected long lastModified = -1;
    /**
     * 最后校验时间
     */
    protected long lastCheckTime = -1;

    /**
     * 获取单例对象
     *
     * @return 单例对象
     */
    public static I18NInfo getInstance() {
        if (instance == null) {
            instance = new I18NInfo();
        }
        return instance;
    }

    public void setCache(I18NInfoCacheInterface cache) {
        this.cache = cache;
    }

    /**
     * 从缓存中获取当前的版本号和最后的修改时间。
     */
    public void resetFromCache() {
        version = cache.getVersion();
        lastModified = cache.getLastModified();
        lastCheckTime = new Date().getTime();
    }

    /**
     * 每过5分钟检查本地版本号和最后修改时间是否同缓存中一致，不一致则需要同步
     *
     * @return 一致返回false，不一致返回true
     */
    public boolean checkNeedUpdate() {
        long nowTime = new Date().getTime();
        boolean needUpdate = false;
        try {
            if (lastCheckTime > 0 && nowTime - lastCheckTime > 1000 * 60 * 5) {//间隔没超过5分钟，则不需要同步
                int cache_version = cache.getVersion();
                long cache_lastModified = cache.getLastModified();
                needUpdate = version != cache_version || lastModified < cache_lastModified;
                lastCheckTime = nowTime;
            }
        } catch (Exception e) {
            lastCheckTime = nowTime;
            LogSvr.getInstance().error("checkNeedUpdate Error", e);
        }
        return needUpdate;
    }

    /**
     * 保存刷新对象后，更新当前版本号，并同步到缓存
     */
    public void updateVersion() {
        version++;
        cache.setVersion(version);
    }

    /**
     * 保存刷新对象后，更新最后的修改时间，并同步到缓存
     *
     * @param newLastModified 最后的修改时间
     */
    public void updateLastModified(long newLastModified) {
        if (lastModified < newLastModified) {
            lastModified = newLastModified;
            cache.setLastModified(lastModified);
        }
    }

    public int getVersion() {
        return version;
    }

    public long getLastModified() {
        return lastModified;
    }

    public void clearMap() {
        dataMap.clear();
    }

    /**
     * 获取当前对象中，已经存在的语种类型
     * @return 已经存在的语种类型集合
     */
    public ArrayList<String> getLanguages() {
        ArrayList<String> list = new ArrayList<>();
        if(dataMap.isEmpty()) return list;
        I18NInfoMap map = dataMap.values().iterator().next();
        list.addAll(map.getChildMap().keySet());
        return list;
    }

    public I18NInfoMap getParentInfoMap(String translationType, String solutionKey, String projectKey, String formKey, String lang) {
        I18NInfoMap translationTypeMap = dataMap.get(translationType);
        I18NInfoMap langMap = translationTypeMap != null ? translationTypeMap.getChildByKey(lang) : null;
        I18NInfoMap solutionMap = langMap != null ? langMap.getChildByKey(solutionKey) : null;
        I18NInfoMap projectMap = solutionMap != null ? solutionMap.getChildByKey(projectKey) : null;
        return projectMap != null ? projectMap.getChildByKey(formKey) : null;
    }

    /**
     * 删除数据行对象
     *
     * @param item 数据行对象
     */
    public void deleteItem(I18NInfoItem item) {
        I18NInfoMap parentMap = getParentInfoMap(item.getTranslationType(), item.getSolutionKey(), item.getProjectKey(), item.getFormKey(), item.getLang());
        if (parentMap != null) {
            parentMap.removeItem(item.getTranslationKey());
        }
    }

    /**
     * 添加新的数据行对象
     *
     * @param item 新的数据行对象
     */
    public void addNewItem(I18NInfoItem item) {
        I18NInfoMap translationTypeMap = dataMap.get(item.getTranslationType());
        if (translationTypeMap == null) {
            translationTypeMap = new I18NInfoMap();
            translationTypeMap.setMapType(I18NInfoMap.MapType_TranslationType);
            translationTypeMap.setKey(item.getTranslationType());
            dataMap.put(translationTypeMap.getKey(), translationTypeMap);
        }
        I18NInfoMap langMap = translationTypeMap.getChildByKey(item.getLang());
        if (langMap == null) {
            langMap = new I18NInfoMap();
            langMap.setMapType(I18NInfoMap.MapType_Lang);
            langMap.setKey(item.getLang());
            translationTypeMap.putChildByKey(langMap.getKey(), langMap);
        }
        I18NInfoMap solutionMap = langMap.getChildByKey(item.getSolutionKey());
        if (solutionMap == null) {
            solutionMap = new I18NInfoMap();
            solutionMap.setMapType(I18NInfoMap.MapType_Solution);
            solutionMap.setKey(item.getSolutionKey());
            langMap.putChildByKey(solutionMap.getKey(), solutionMap);
        }
        I18NInfoMap projectMap = solutionMap.getChildByKey(item.getProjectKey());
        if (projectMap == null) {
            projectMap = new I18NInfoMap();
            projectMap.setMapType(I18NInfoMap.MapType_Project);
            projectMap.setKey(item.getProjectKey());
            solutionMap.putChildByKey(projectMap.getKey(), projectMap);
        }
        I18NInfoMap formMap = projectMap.getChildByKey(item.getFormKey());
        if (formMap == null) {
            formMap = new I18NInfoMap();
            formMap.setMapType(I18NInfoMap.MapType_Form);
            formMap.setKey(item.getFormKey());
            projectMap.putChildByKey(formMap.getKey(), formMap);
        }
        formMap.putItemByKey(item.getTranslationKey(), item);
    }

    /**
     * 获取I18NInfo的数据行对象
     *
     * @param translationType 翻译词条类型
     * @param translationKey  词条翻译标识
     * @param solutionKey     解决方案标识
     * @param projectKey      工程标识
     * @param formKey         表单标识
     * @param lang            语种类型
     * @return 返回数据行对象
     */
    public I18NInfoItem getItem(String translationType, String translationKey, String solutionKey, String projectKey, String formKey, String lang) {
        try {
            if (solutionKey == null || solutionKey.isEmpty() || solutionKey.equals("*")) solutionKey = "*";
            if (projectKey == null || projectKey.isEmpty() || projectKey.equals("*")) projectKey = "*";
            if (formKey == null || formKey.isEmpty() || formKey.equals("*")) formKey = "*";
            I18NInfoItem item = null;
            I18NInfoMap parentMap = getParentInfoMap(translationType, solutionKey, projectKey, formKey, lang);
            if (parentMap != null) {
                item = parentMap.getItemByKey(translationKey);
            }
            if (item == null) {
                parentMap = getParentInfoMap(translationType, solutionKey, projectKey, "*", lang);
                if (parentMap != null) {
                    item = parentMap.getItemByKey(translationKey);
                }
            }
            if (item == null) {
                parentMap = getParentInfoMap(translationType, solutionKey, "*", "*", lang);
                if (parentMap != null) {
                    item = parentMap.getItemByKey(translationKey);
                }
            }
            if (item == null) {
                parentMap = getParentInfoMap(translationType, "*", "*", "*", lang);
                if (parentMap != null) {
                    item = parentMap.getItemByKey(translationKey);
                }
            }
            return item;
        } catch (Throwable e) {
            LogSvr.getInstance().error("I18NInfo getItem", e);
        }
        return null;
    }

    /**
     * 获取I18NInfo的数据行对象
     *
     * @param translationType 翻译词条类型
     * @param translationKey  词条翻译标识
     * @param formKey         表单标识
     * @param lang            语种类型
     * @return 返回数据行对象
     */
    public I18NInfoItem getItem(String translationType, String translationKey, String formKey, String lang) {
        try {
            String solutionKey = "*";
            String projectKey = "*";
            if (formKey == null || formKey.isEmpty() || formKey.equals("*")) {
                formKey = "*";
            } else {
                MetaFormProfile metaFormProfile = MetaFactory.getGlobalInstance().getMetaFormList().get(formKey);
                projectKey = metaFormProfile.getProject().getKey();
                solutionKey = ((MetaSolution) ((MetaProject) metaFormProfile.getProject()).getSolution()).getKey();
            }
            return getItem(translationType, translationKey, solutionKey, projectKey, formKey, lang);
        } catch (Throwable e) {
            LogSvr.getInstance().error("I18NInfo getItem", e);
        }
        return null;
    }

    /**
     * 根据配置对象，获取对应的翻译类型
     *
     * @param metaObject 配置对象
     * @return 翻译类型
     */
    private static String getTranslationType(AbstractMetaObject metaObject) {
        String translationType = TranslationType_Common;
        if (metaObject == null) return translationType;
        if (metaObject instanceof MetaOperation || metaObject instanceof MetaOperationCollection) {
            translationType = TranslationType_Operation;
        } else if (metaObject instanceof MetaComponent || metaObject instanceof MetaGridColumn) {
            translationType = TranslationType_Component;
        } else if (metaObject instanceof MetaEntry || metaObject instanceof MetaEntryItem) {
            translationType = TranslationType_Entry;
        }
        return translationType;
    }

    public String getLocalCaptionString(IMetaFactory metaFactory, MetaForm metaForm, String locale, String group, String word, String org,
                                        String s, MetaLang metaLang, AbstractMetaObject metaObject, boolean useFull) throws Throwable {
        String result = "";
        String lang = metaLang == null ? locale : metaLang.getKey();
        String translationType = getTranslationType(metaObject);
        String formKey = metaForm == null ? null : metaForm.getKey();

        String baseLang = metaLang == null ? BaseLang_ZH : metaLang.getBaseLang();
        if (baseLang.equals(BaseLang_EN)) {
            I18NInfoItem enItem = getInstance().getItem(translationType, org, formKey, BaseLang_EN);
            if (enItem == null) {
                enItem = getInstance().getItem(TranslationType_Common, org, formKey, BaseLang_EN);
            }
            if (enItem != null) {
                org = enItem.getCaption();
                if (enItem.getManualCaption() != null && !enItem.getManualCaption().isEmpty()) {
                    org = enItem.getManualCaption();
                }
            }
        }
        I18NInfoItem i18NInfoItem = getInstance().getItem(translationType, org, formKey, lang);
        if (i18NInfoItem == null) {
            i18NInfoItem = getInstance().getItem(TranslationType_Common, org, formKey, lang);
        }
        if (i18NInfoItem != null) {
            result = i18NInfoItem.getCaption();
            if (i18NInfoItem.getManualCaption() != null && !i18NInfoItem.getManualCaption().isEmpty()) {
                result = i18NInfoItem.getManualCaption();
            }
            if (useFull) {
                result = i18NInfoItem.getCaptionFull();
                if (i18NInfoItem.getManualCaptionFull() != null && !i18NInfoItem.getManualCaptionFull().isEmpty()) {
                    result = i18NInfoItem.getManualCaptionFull();
                }
            }
        }
        if (result.isEmpty()) {
            result = MetaUtil.getLocalString(metaFactory, metaForm, locale, group, word, org, s);
        }
        return result;
    }

    public String getCaptionString(IMetaFactory metaFactory, String locale, String projectKey, String group, String word,
                                   String org, MetaLang metaLang, AbstractMetaObject metaObject) throws Throwable {
        String result = "";
        String lang = metaLang == null ? locale : metaLang.getKey();
        String translationType = getTranslationType(metaObject);
        String solutionKey = null;
        if (projectKey != null && !projectKey.isEmpty()) {
            MetaProject metaProject = metaFactory.getMetaProject(projectKey);
            if (metaProject != null) {
                MetaSolution metaSolution = (MetaSolution) metaProject.getSolution();
                solutionKey = metaSolution.getKey();
            }
        }

        String baseLang = metaLang == null ? BaseLang_ZH : metaLang.getBaseLang();
        if (baseLang.equals(BaseLang_EN)) {
            I18NInfoItem enItem = getInstance().getItem(translationType, org, solutionKey, projectKey, null, BaseLang_EN);
            if (enItem == null) {
                enItem = getInstance().getItem(TranslationType_Common, org, solutionKey, projectKey, null, BaseLang_EN);
            }
            if (enItem != null) {
                org = enItem.getCaption();
                if (enItem.getManualCaption() != null && !enItem.getManualCaption().isEmpty()) {
                    org = enItem.getManualCaption();
                }
            }
        }
        I18NInfoItem i18NInfoItem = getInstance().getItem(translationType, org, solutionKey, projectKey, null, lang);
        if (i18NInfoItem == null) {
            i18NInfoItem = getInstance().getItem(TranslationType_Common, org, solutionKey, projectKey, null, lang);
        }
        if (i18NInfoItem != null) {
            result = i18NInfoItem.getCaption();
            if (i18NInfoItem.getManualCaption() != null && !i18NInfoItem.getManualCaption().isEmpty()) {
                result = i18NInfoItem.getManualCaption();
            }
        }

        if (result.isEmpty()) {
            result = MetaUtil.getString(metaFactory, lang, projectKey, group, word, org);
        }
        return result;
    }
}