package com.bokesoft.yigo.struct.datatable;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.lang3.ArrayUtils;

import com.bokesoft.yes.common.util.LockUtil;
import com.bokesoft.yes.struct.datatable.Row;
import com.bokesoft.yigo.common.def.DataType;
import com.bokesoft.yigo.common.util.TypeConvertor;

/**
 * 基于DataTable的索引
 */
public class Index {
	/**
	 * 字段数组
	 */
	final String[] fieldArray;
	/**
	 * 字段在DataTable中的列位置
	 */
	private int[] fieldIndexes;
	/** 索引字段类型 */
	private int[] fieldDataTypes;
	/**
	 * DataTable对象,即索引的主体
	 */
	final private DataTable rst;
	
	/** 索引过的缓冲  */
	private Map<IndexValue, List<IndexRow>> indexRows;

	private Map<Integer, IndexValue> bookmarkToIndexValues;

	public Index(DataTable rst, String[] fieldArray) {
		this.rst = rst;
		this.fieldArray = fieldArray;
		initFieldIndexes();
	}

	/**
	 * 初始化字段
	 */
	private void initFieldIndexes() {
		int count = fieldArray.length;
		fieldIndexes = new int[count];
		fieldDataTypes = new int[count];
		DataTableMetaData metaData = rst.getMetaData();
		for (int i = 0; i < count; i++) {
			ColumnInfo columnInfo = metaData.getColumnInfo(fieldArray[i]);
			fieldIndexes[i] = columnInfo.getColumnIndex();
			fieldDataTypes[i] = columnInfo.getDataType();
		}
	}

	/**
	 * 索引是否符合查询字段
	 * @param filterColumns
	 * @return
	 */
	public boolean isSatisfy(String[] filterColumns) {
		int length = filterColumns.length;
		if (fieldArray.length != length) {
			return false;
		}
		for (int i = 0; i < length; i++) {
			if (!fieldArray[i].equalsIgnoreCase(filterColumns[i])) {
				return false;
			}
		}
		return true;
	}

	private IndexValue newIndexValue(int bookmark, Object[] indexValues, int[] fieldDataTypes) {
		IndexValue result = new IndexValue(indexValues, fieldDataTypes);
		if (bookmarkToIndexValues == null) {
			bookmarkToIndexValues = new HashMap<Integer, IndexValue>();
		}
		bookmarkToIndexValues.put(bookmark, result);
		return result;
	}

	/**
	 * 初始化排序过的缓冲
	 * 
	 * @throws Throwable
	 */
	private void initCache() throws Throwable {
		if (indexRows == null) {
			int size = rst.size();
			Map<IndexValue, List<IndexRow>> data = new ConcurrentHashMap<>(size);
			for (int rowIndex = 0; rowIndex < size; rowIndex++) {
				int bookmark = rst.getBookmark(rowIndex);
				Object[] indexValues = getIndexValues(rowIndex);
				IndexValue key = newIndexValue(bookmark, indexValues, fieldDataTypes);
				IndexRow indexRow = new IndexRow(rowIndex, bookmark);
				List<IndexRow> rows = data.get(key);
				if (rows == null) {
					rows = new CopyOnWriteArrayList<>();
					data.put(key, rows);
				}
				if( !rows.contains(indexRow)) {
					rows.add(indexRow);
				}
			}
			indexRows = data;
		} else if (updateBookmarks != null || insertBookmarks != null || deleteRows != null) {
			updateIndexRows();
			updateBookmarks = null;
			insertBookmarks = null;
			deleteRows = null;
		}
		//System.out.println(rst.hashCode() + "\t" + this.fields + "\tinitCache");
	}
	
	private Object[] getIndexValues(int rowIndex) {
		@SuppressWarnings("deprecation")
		Object[] dataList = rst.getRowByIndex(rowIndex).getDataList();
		int fieldCount = fieldIndexes.length;
		Object[] result = new Object[fieldCount];
		for (int i = 0; i < fieldCount; i++) {
			Object value = dataList[fieldIndexes[i]];
			result[i] = convertType(value, fieldDataTypes[i]);
		}
		return result;
	}
	
	public static Object convertType(Object value, int fieldDataType) {
		switch (fieldDataType) {
		case DataType.LONG:
			return TypeConvertor.toLong(value);
		case DataType.INT:
		case DataType.BOOLEAN:
			return TypeConvertor.toInteger(value);
		case DataType.NUMERIC:
			return TypeConvertor.toBigDecimal(value);
		case DataType.STRING:
		case DataType.FIXED_STRING:
			return TypeConvertor.toString(value);
		default:
			throw new RuntimeException("DataTable索引不支持类型：" + fieldDataType + "，请联系开发人员。");
		}
	}

	/**
	 * 清除缓冲,目前不支持DataTable结构变化
	 */
	public void clear() {
		if (indexRows != null) {
			indexRows.clear();
		}
		indexRows = null;
		//System.out.println(rst.hashCode() + "\t" + this.fields + "\tclear");
	}

	/**
	 * 使用多个条件进行过滤
	 * 
	 * @param filters
	 * @return
	 * @throws Throwable
	 */
	public int[] filter(Object[] filterValues) throws Throwable {
		LockUtil.execute("Index.InitCache." + this.hashCode(), () -> {
			initCache();
			return null;
		});
		return filter_impl(filterValues);
	}
	
	/**
	 * 索引值中带数组的过滤实现，通过递归调用实现多个值中数组<br>
	 * 因实际调用中多个值中有数组的情况较少，所以容忍这种实现的相对低效率
	 * @param filterValues
	 * @return
	 * @throws Throwable
	 */
	private int[] filter_impl(Object[] filterValues) throws Throwable {
		for (int i = 0, columnSize = filterValues.length; i < columnSize; i++) {
			Object filterValue = filterValues[i];
			if (filterValue.getClass().isArray()) {
				int valueCount = Array.getLength(filterValue);
				Object[][] newFilterValues = new Object[valueCount][columnSize];
				for (int j = 0; j < valueCount; j++) {
					for (int k = 0; k < columnSize; k++) {
						newFilterValues[j][k] = k == i ? Array.get(filterValue, j) : filterValues[k];
					}
				}
				IntList intList = new IntList(Integer.min(rst.size(), 64));
				for (int j = 0; j < valueCount; j++) {
					intList.add(filter_impl(newFilterValues[j]));
				}
				return intList.toArray();
			}
		}
		return filter_implNoArray(filterValues);
	}
	
	/**
	 * 索引值中没有数组的过滤实现
	 * @param filterValues
	 * @return
	 * @throws Throwable
	 */
	private int[] filter_implNoArray(Object[] filterValues) throws Throwable {
		IndexValue key = new IndexValue(filterValues, fieldDataTypes);
		List<IndexRow> resultRows = indexRows.get(key);

		if (resultRows == null) {
			return ArrayUtils.EMPTY_INT_ARRAY;
		}
		int resultSize = resultRows.size();
		int[] result = new int[resultSize];
		int rstSize = rst.size();
		for (int i = 0; i < resultSize; i++) {
			IndexRow indexRow = resultRows.get(i);
			int rowIndexTmp = indexRow.rowIndex;
			if (rowIndexTmp >= rstSize || rst.getBookmark(rowIndexTmp) != indexRow.bookmark) { // 根据Bookmark更新可能变化了的RowIndex
				rowIndexTmp = rst.getRowIndexByBookmark(indexRow.bookmark);
				indexRow.rowIndex = rowIndexTmp;
			}
			result[i] = rowIndexTmp;
		}
		return result;
	}
	
	/**
	 * 取索引的不同值
	 * @return
	 * @throws Throwable 
	 */
	public Object[][] getDistinctKeyValues() throws Throwable {
		LockUtil.execute("Index.InitCache." + this.hashCode(), () -> {
			initCache();
			return null;
		});
		int fieldCount = fieldArray.length;
		int size = indexRows.size();
		Object[][] result = new Object[size][fieldCount];
		int index = 0;
		for (IndexValue indexValue : indexRows.keySet()) {
			System.arraycopy(indexValue.indexValues, 0, result[index], 0, fieldCount);
			index++;
		}
		return result;
	}

//	/**
//	 * 单个条件,单个字段进行In过滤
//	 * 
//	 * @param rows
//	 * @param filter
//	 * @param indexFieldIndex
//	 * @return
//	 */
//	private List<IndexRow> filterIn(List<IndexRow> rows, int[] inValues,
//			int indexFieldIndex) {
//		List<IndexRow> result = null;
//		int rowCount = rows.size();
//		int rowIndex = 0;
//		int filterInCount = inValues.length;
//		int filterInIndex = 0;
//		if (rowCount == 0 || filterInCount == 0) {
//			return null;
//		}
//		IndexRow row = rows.get(rowIndex);
//		IntColumnData columnData = (IntColumnData) data.getColumnData(indexFieldIndex);
//		int indexValue = columnData.getIntData(row.rowIndex);
//		int filterInValue = inValues[filterInIndex];
//		while (true) {
//			if (indexValue == filterInValue) {
//				if (result == null) {
//					result = new ArrayList<IndexRow>();
//				}
//				result.add(row);
//				rowIndex++;
//				if (rowIndex >= rowCount) {
//					break;
//				}
//				row = rows.get(rowIndex);
//				indexValue = columnData.getIntData(row.rowIndex);
//			} else if (indexValue < filterInValue) {
//				rowIndex++;
//				if (rowIndex >= rowCount) {
//					break;
//				}
//				row = rows.get(rowIndex);
//				indexValue = columnData.getIntData(row.rowIndex);
//			} else if (indexValue > filterInValue) {
//				filterInIndex++;
//				if (filterInIndex >= filterInCount) {
//					break;
//				}
//				filterInValue = inValues[filterInIndex];
//			}
//		}
//		return result;
//	}

	/** 索引中需要更新的Bookmark */
	private HashSet<Integer> updateBookmarks;

	/** 索引中需要插入的Bookmark */
	private List<Integer> insertBookmarks;

	/** 索引中需要更新的行 */
	private List<Row> deleteRows;
	
	/**
	 * 更新索引数据，只是将变化传递过来
	 */
	public void updateIndex(HashSet<Integer> updateBookmarks, List<Integer> insertBookmarks, List<Row> deleteRows) {
		if (indexRows == null) {
			return;
		}
		if (updateBookmarks != null) {
			if (this.updateBookmarks == null) {
				this.updateBookmarks = new HashSet<Integer>(updateBookmarks);
			} else {
				this.updateBookmarks.addAll(updateBookmarks);
			}
		}
		if (insertBookmarks != null) {
			if (this.insertBookmarks == null) {
				this.insertBookmarks = new CopyOnWriteArrayList<>(insertBookmarks);
			} else {
				this.insertBookmarks.addAll(insertBookmarks);
			}
		}
		if (deleteRows != null) {
			if (this.deleteRows == null) {
				this.deleteRows = new CopyOnWriteArrayList<>(deleteRows);
			} else {
				this.deleteRows.addAll(deleteRows);
			}
		}
	}
	
	/**
	 * 更新索引数据
	 */
	private void updateIndexRows() {
		// 需要处理一些特殊情况：
		// 1.插入后删除，在本方法中处理
		// 2.删除后恢复，这一步可反复做，在Indexes.recoverBookmark()中处理
		// 3.修改后删除，在updateRow()方法中处理
		if (deleteRows != null) {
			for (Row row : deleteRows) {
				if (insertBookmarks != null) {
					int index = insertBookmarks.indexOf(row.getBookmark()); // 处理新增的删除行，如果这一行有性能问题，可排序后再比较
					if (index >= 0) {
						insertBookmarks.remove(index);
						continue;
					}
				}
				deleteRow(row);
			}
		}
		if (insertBookmarks != null) {
			for (int bookmark : insertBookmarks) {
				insertRow(bookmark);
			}
		}
		if (updateBookmarks != null) {
			for (Integer bookmark : updateBookmarks) {
				updateRow(bookmark);
			}
		}
	}
	
	private void deleteRow(Row row) {
		int deleteBookmark = row.getBookmark();
		deleteRowByBookmark(deleteBookmark);
	}
	
	private void deleteRowByBookmark(int deleteBookmark) {
		IndexValue key = bookmarkToIndexValues.get(deleteBookmark);
		List<IndexRow> rows = indexRows.get(key);
		int length = rows.size();
		if (length == 1) {
			indexRows.remove(key);
		} else {
			for (int i = length - 1; i >= 0; i--) {
				IndexRow indexRow = rows.get(i);
				if (indexRow.bookmark == deleteBookmark) {
					rows.remove(i);
					break;
				}
			}
		}
	}
	
	private void insertRow(int bookmark) {
		int rowIndex = rst.getRowIndexByBookmark(bookmark);
		Object[] indexValues = getIndexValues(rowIndex);
		IndexValue key = newIndexValue(bookmark, indexValues, fieldDataTypes);
		IndexRow indexRow = new IndexRow(rowIndex, bookmark);
		List<IndexRow> rows = indexRows.get(key);
		if (rows == null) {
			rows = new CopyOnWriteArrayList<IndexRow>();
			indexRows.put(key, rows);
		}
		if (!rows.contains(indexRow)) {
			rows.add(indexRow);
		}
	}
	
	private void updateRow(int bookmark) {
		int rowIndex = rst.getRowIndexByBookmark(bookmark);
		if (rowIndex < 0) { // 表示这一行被删除了
			return;
		}
		Object[] indexValues = getIndexValues(rowIndex);
		IndexValue oldIndexValue = bookmarkToIndexValues.get(bookmark);
		if (!IndexValue.isSame(indexValues, oldIndexValue.indexValues)) {
			deleteRowByBookmark(bookmark);
			
			IndexValue key = newIndexValue(bookmark, indexValues, fieldDataTypes);
			IndexRow indexRow = new IndexRow(rowIndex, bookmark);
			List<IndexRow> rows = indexRows.get(key);
			if (rows == null) {
				rows = new CopyOnWriteArrayList<IndexRow>();
				indexRows.put(key, rows);
			}
			if (!rows.contains(indexRow)) {
				rows.add(indexRow);
			}
		}
	}
}

/**
 * 索引的值
 */
class IndexValue {
	IndexValue(Object[] indexValues, int[] fieldDataTypes) {
		this.indexValues = indexValues;
		for (int i = 0, size = indexValues.length; i < size; i++) {
			indexValues[i] = Index.convertType(indexValues[i], fieldDataTypes[i]);
		}
	}
	
	final Object[] indexValues;
	
	private int hashCode = 0;

	@Override
	public int hashCode() {
		if (hashCode == 0) {
			for (Object obj : indexValues) {
				hashCode = hashCode * 31 + obj.hashCode();
			}
		}
		return hashCode;
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == null || !(obj instanceof IndexValue)) {
			return false;
		}
		return isSame(this.indexValues, ((IndexValue) obj).indexValues);
	}
	
	/**
	 * 对两个数据进行比较
	 * @param values1
	 * @param values2
	 * @return
	 */
	public static boolean isSame(Object[] values1, Object[] values2) {
		if (values1 == values2) {
			return true;
		}
		for (int i = values1.length - 1; i >=0; i--) {
			if (!isSameObj(values1[i], values2[i])) {
				return false;
			}
		}
		return true;
	}

	private static boolean isSameObj(Object obj1, Object obj2) {
		if (obj1 == obj2) {
			return true;
		}
		if (obj1 == null || obj2 == null) {
			return false;
		}
		if (obj1 instanceof Integer) {
			return obj2 instanceof Integer && obj1.equals(obj2);
		} else if (obj1 instanceof Long) {
			return obj2 instanceof Long && obj1.equals(obj2);
		} else if (obj1 instanceof String) {
			return obj2 instanceof String && obj1.equals(obj2);
		} else if (obj1 instanceof BigDecimal) {
			return obj2 instanceof BigDecimal && obj1.equals(obj2);
		}
		throw new RuntimeException("不支持的类型比较，" + obj1.getClass().getName());
	}
}

/**
 * 索引行
 */
class IndexRow {
	/** 记录行号，以减少根据bookmark进行hash查找 */
	int rowIndex;
	final int bookmark;

	public IndexRow(int rowIndex, int bookmark) {
		this.rowIndex = rowIndex;
		this.bookmark = bookmark;
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == null) {
			return false;
		}
		IndexRow o2 = (IndexRow) obj;
		return this.bookmark == o2.bookmark;
	}
}

/**
 * 拷贝自org.apache.poi.util.IntList
 */
class IntList
{
    private int[]            _array;
    private int              _limit;
    private int              fillval = 0;
    private static final int _default_size = 128;

    /**
     * create an IntList of default size
     */

    public IntList()
    {
        this(_default_size);
    }

    public IntList(final int initialCapacity)
    {
        this(initialCapacity,0);
    }


    /**
     * create a copy of an existing IntList
     *
     * @param list the existing IntList
     */

    public IntList(final IntList list)
    {
        this(list._array.length);
        System.arraycopy(list._array, 0, _array, 0, _array.length);
        _limit = list._limit;
    }

    /**
     * create an IntList with a predefined initial size
     *
     * @param initialCapacity the size for the internal array
     */

    public IntList(final int initialCapacity, int fillvalue)
    {
        _array = new int[ initialCapacity ];
        if (fillval != 0) {
            fillval = fillvalue;
            fillArray(fillval, _array, 0);
        }
        _limit = 0;
    }

    private void fillArray(int val, int[] array, int index) {
      for (int k = index; k < array.length; k++) {
        array[k] = val;
      }
    }

    /**
     * add the specfied value at the specified index
     *
     * @param index the index where the new value is to be added
     * @param value the new value
     *
     * @exception IndexOutOfBoundsException if the index is out of
     *            range (index < 0 || index > size()).
     */

    public void add(final int index, final int value)
    {
        if (index > _limit)
        {
            throw new IndexOutOfBoundsException();
        }
        else if (index == _limit)
        {
            add(value);
        }
        else
        {

            // index < limit -- insert into the middle
            if (_limit == _array.length)
            {
                growArray(_limit * 2);
            }
            System.arraycopy(_array, index, _array, index + 1,
                             _limit - index);
            _array[ index ] = value;
            _limit++;
        }
    }

    /**
     * Appends the specified element to the end of this list
     *
     * @param value element to be appended to this list.
     *
     * @return true (as per the general contract of the Collection.add
     *         method).
     */

    public boolean add(final int value)
    {
        if (_limit == _array.length)
        {
            growArray(_limit * 2);
        }
        _array[ _limit++ ] = value;
        return true;
    }
    
    public boolean add(final int[] values)
    {
        if (values == null) {
            return true;
        }
        int valuesSize = values.length;
        if (_limit + valuesSize > _array.length) {
            growArray(Integer.max(_limit * 2, _limit + valuesSize));
        }
        System.arraycopy(values, 0, _array, _limit, valuesSize);
        _limit += valuesSize;
        return true;
    }

    /**
     * Appends all of the elements in the specified collection to the
     * end of this list, in the order that they are returned by the
     * specified collection's iterator.  The behavior of this
     * operation is unspecified if the specified collection is
     * modified while the operation is in progress.  (Note that this
     * will occur if the specified collection is this list, and it's
     * nonempty.)
     *
     * @param c collection whose elements are to be added to this
     *          list.
     *
     * @return true if this list changed as a result of the call.
     */

    public boolean addAll(final IntList c)
    {
        if (c._limit != 0)
        {
            if ((_limit + c._limit) > _array.length)
            {
                growArray(_limit + c._limit);
            }
            System.arraycopy(c._array, 0, _array, _limit, c._limit);
            _limit += c._limit;
        }
        return true;
    }

    /**
     * Inserts all of the elements in the specified collection into
     * this list at the specified position.  Shifts the element
     * currently at that position (if any) and any subsequent elements
     * to the right (increases their indices).  The new elements will
     * appear in this list in the order that they are returned by the
     * specified collection's iterator.  The behavior of this
     * operation is unspecified if the specified collection is
     * modified while the operation is in progress.  (Note that this
     * will occur if the specified collection is this list, and it's
     * nonempty.)
     *
     * @param index index at which to insert first element from the
     *              specified collection.
     * @param c elements to be inserted into this list.
     *
     * @return true if this list changed as a result of the call.
     *
     * @exception IndexOutOfBoundsException if the index is out of
     *            range (index < 0 || index > size())
     */

    public boolean addAll(final int index, final IntList c)
    {
        if (index > _limit)
        {
            throw new IndexOutOfBoundsException();
        }
        if (c._limit != 0)
        {
            if ((_limit + c._limit) > _array.length)
            {
                growArray(_limit + c._limit);
            }

            // make a hole
            System.arraycopy(_array, index, _array, index + c._limit,
                             _limit - index);

            // fill it in
            System.arraycopy(c._array, 0, _array, index, c._limit);
            _limit += c._limit;
        }
        return true;
    }

    /**
     * Removes all of the elements from this list.  This list will be
     * empty after this call returns (unless it throws an exception).
     */

    public void clear()
    {
        _limit = 0;
    }

    /**
     * Returns true if this list contains the specified element.  More
     * formally, returns true if and only if this list contains at
     * least one element e such that o == e
     *
     * @param o element whose presence in this list is to be tested.
     *
     * @return true if this list contains the specified element.
     */

    public boolean contains(final int o)
    {
        boolean rval = false;

        for (int j = 0; !rval && (j < _limit); j++)
        {
            if (_array[ j ] == o)
            {
                rval = true;
            }
        }
        return rval;
    }

    /**
     * Returns true if this list contains all of the elements of the
     * specified collection.
     *
     * @param c collection to be checked for containment in this list.
     *
     * @return true if this list contains all of the elements of the
     *         specified collection.
     */

    public boolean containsAll(final IntList c)
    {
        boolean rval = true;

        if (this != c)
        {
            for (int j = 0; rval && (j < c._limit); j++)
            {
                if (!contains(c._array[ j ]))
                {
                    rval = false;
                }
            }
        }
        return rval;
    }

    /**
     * Compares the specified object with this list for equality.
     * Returns true if and only if the specified object is also a
     * list, both lists have the same size, and all corresponding
     * pairs of elements in the two lists are equal.  (Two elements e1
     * and e2 are equal if e1 == e2.)  In other words, two lists are
     * defined to be equal if they contain the same elements in the
     * same order.  This definition ensures that the equals method
     * works properly across different implementations of the List
     * interface.
     *
     * @param o the object to be compared for equality with this list.
     *
     * @return true if the specified object is equal to this list.
     */

    public boolean equals(final Object o)
    {
        boolean rval = this == o;

        if (!rval && (o != null) && (o.getClass() == this.getClass()))
        {
            IntList other = ( IntList ) o;

            if (other._limit == _limit)
            {

                // assume match
                rval = true;
                for (int j = 0; rval && (j < _limit); j++)
                {
                    rval = _array[ j ] == other._array[ j ];
                }
            }
        }
        return rval;
    }

    /**
     * Returns the element at the specified position in this list.
     *
     * @param index index of element to return.
     *
     * @return the element at the specified position in this list.
     *
     * @exception IndexOutOfBoundsException if the index is out of
     *            range (index < 0 || index >= size()).
     */

    public int get(final int index)
    {
        if (index >= _limit)
        {
            throw new IndexOutOfBoundsException(
                  index + " not accessible in a list of length " + _limit
            );
        }
        return _array[ index ];
    }

    /**
     * Returns the hash code value for this list.  The hash code of a
     * list is defined to be the result of the following calculation:
     *
     * <code>
     * hashCode = 1;
     * Iterator i = list.iterator();
     * while (i.hasNext()) {
     *      Object obj = i.next();
     *      hashCode = 31*hashCode + (obj==null ? 0 : obj.hashCode());
     * }
     * </code>
     *
     * This ensures that list1.equals(list2) implies that
     * list1.hashCode()==list2.hashCode() for any two lists, list1 and
     * list2, as required by the general contract of Object.hashCode.
     *
     * @return the hash code value for this list.
     */

    public int hashCode()
    {
        int hash = 0;

        for (int j = 0; j < _limit; j++)
        {
            hash = (31 * hash) + _array[ j ];
        }
        return hash;
    }

    /**
     * Returns the index in this list of the first occurrence of the
     * specified element, or -1 if this list does not contain this
     * element.  More formally, returns the lowest index i such that
     * (o == get(i)), or -1 if there is no such index.
     *
     * @param o element to search for.
     *
     * @return the index in this list of the first occurrence of the
     *         specified element, or -1 if this list does not contain
     *         this element.
     */

    public int indexOf(final int o)
    {
        int rval = 0;

        for (; rval < _limit; rval++)
        {
            if (o == _array[ rval ])
            {
                break;
            }
        }
        if (rval == _limit)
        {
            rval = -1;   // didn't find it
        }
        return rval;
    }

    /**
     * Returns true if this list contains no elements.
     *
     * @return true if this list contains no elements.
     */

    public boolean isEmpty()
    {
        return _limit == 0;
    }

    /**
     * Returns the index in this list of the last occurrence of the
     * specified element, or -1 if this list does not contain this
     * element.  More formally, returns the highest index i such that
     * (o == get(i)), or -1 if there is no such index.
     *
     * @param o element to search for.
     *
     * @return the index in this list of the last occurrence of the
     *         specified element, or -1 if this list does not contain
     *         this element.
     */

    public int lastIndexOf(final int o)
    {
        int rval = _limit - 1;

        for (; rval >= 0; rval--)
        {
            if (o == _array[ rval ])
            {
                break;
            }
        }
        return rval;
    }

    /**
     * Removes the element at the specified position in this list.
     * Shifts any subsequent elements to the left (subtracts one from
     * their indices).  Returns the element that was removed from the
     * list.
     *
     * @param index the index of the element to removed.
     *
     * @return the element previously at the specified position.
     *
     * @exception IndexOutOfBoundsException if the index is out of
     *            range (index < 0 || index >= size()).
     */

    public int remove(final int index)
    {
        if (index >= _limit)
        {
            throw new IndexOutOfBoundsException();
        }
        int rval = _array[ index ];

        System.arraycopy(_array, index + 1, _array, index, _limit - index);
        _limit--;
        return rval;
    }

    /**
     * Removes the first occurrence in this list of the specified
     * element (optional operation).  If this list does not contain
     * the element, it is unchanged.  More formally, removes the
     * element with the lowest index i such that (o.equals(get(i)))
     * (if such an element exists).
     *
     * @param o element to be removed from this list, if present.
     *
     * @return true if this list contained the specified element.
     */

    public boolean removeValue(final int o)
    {
        boolean rval = false;

        for (int j = 0; !rval && (j < _limit); j++)
        {
            if (o == _array[ j ])
            {
                if (j+1 < _limit) {
                    System.arraycopy(_array, j + 1, _array, j, _limit - j);
                }
                _limit--;
                rval = true;
            }
        }
        return rval;
    }

    /**
     * Removes from this list all the elements that are contained in
     * the specified collection
     *
     * @param c collection that defines which elements will be removed
     *          from this list.
     *
     * @return true if this list changed as a result of the call.
     */

    public boolean removeAll(final IntList c)
    {
        boolean rval = false;

        for (int j = 0; j < c._limit; j++)
        {
            if (removeValue(c._array[ j ]))
            {
                rval = true;
            }
        }
        return rval;
    }

    /**
     * Retains only the elements in this list that are contained in
     * the specified collection.  In other words, removes from this
     * list all the elements that are not contained in the specified
     * collection.
     *
     * @param c collection that defines which elements this set will
     *          retain.
     *
     * @return true if this list changed as a result of the call.
     */

    public boolean retainAll(final IntList c)
    {
        boolean rval = false;

        for (int j = 0; j < _limit; )
        {
            if (!c.contains(_array[ j ]))
            {
                remove(j);
                rval = true;
            }
            else
            {
                j++;
            }
        }
        return rval;
    }

    /**
     * Replaces the element at the specified position in this list
     * with the specified element
     *
     * @param index index of element to replace.
     * @param element element to be stored at the specified position.
     *
     * @return the element previously at the specified position.
     *
     * @exception IndexOutOfBoundsException if the index is out of
     *            range (index < 0 || index >= size()).
     */

    public int set(final int index, final int element)
    {
        if (index >= _limit)
        {
            throw new IndexOutOfBoundsException();
        }
        int rval = _array[ index ];

        _array[ index ] = element;
        return rval;
    }

    /**
     * Returns the number of elements in this list. If this list
     * contains more than Integer.MAX_VALUE elements, returns
     * Integer.MAX_VALUE.
     *
     * @return the number of elements in this IntList
     */

    public int size()
    {
        return _limit;
    }

    /**
     * Returns an array containing all of the elements in this list in
     * proper sequence.  Obeys the general contract of the
     * Collection.toArray method.
     *
     * @return an array containing all of the elements in this list in
     *         proper sequence.
     */

    public int [] toArray()
    {
        int[] rval = new int[ _limit ];

        System.arraycopy(_array, 0, rval, 0, _limit);
        return rval;
    }

    /**
     * Returns an array containing all of the elements in this list in
     * proper sequence.  Obeys the general contract of the
     * Collection.toArray(Object[]) method.
     *
     * @param a the array into which the elements of this list are to
     *          be stored, if it is big enough; otherwise, a new array
     *          is allocated for this purpose.
     *
     * @return an array containing the elements of this list.
     */

    public int [] toArray(final int [] a)
    {
        int[] rval;

        if (a.length == _limit)
        {
            System.arraycopy(_array, 0, a, 0, _limit);
            rval = a;
        }
        else
        {
            rval = toArray();
        }
        return rval;
    }

    private void growArray(final int new_size)
    {
        int   size      = (new_size == _array.length) ? new_size + 1
                                                      : new_size;
        int[] new_array = new int[ size ];

        if (fillval != 0) {
          fillArray(fillval, new_array, _array.length);
        }

        System.arraycopy(_array, 0, new_array, 0, _limit);
        _array = new_array;
    }
}   // end public class IntList