package com.bokesoft.yigo.struct.document;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map.Entry;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.JSONHelper;
import com.bokesoft.yes.common.json.SerializationException;
import com.bokesoft.yes.common.struct.StringHashMap;
import com.bokesoft.yes.common.util.Callback;
import com.bokesoft.yes.struct.document.TableFilterJSONConstants;
import com.bokesoft.yigo.common.json.JSONSerializable;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.dataobject.MetaTableSource;

/**
 * 表格查询的各类条件
 * 
 * @author 刘翔翔
 */
public class TableFilterDetail implements JSONSerializable {
	/** 表的KEY */
	private String tableKey = "";
	/** 载入明细表数据时某些字段的条件值 */
	private StringHashMap<Object> conditionValueMap = null;
	/** 过滤条件的条件值 */
	private ArrayList<Object> paraValues = null;
	/** 查询的起始行 */
	private int startRow = 0;
	/** 查询的最大行数 */
	private int maxRows = 0;
	/** 数据行的OID */
	private long OID = -1;
	/** 表的来源标识，决定用哪个来源为表载入数据 */
	private String sourceKey = "";
	/** 向后查多少行(后台分页),默认只查一页*/
	private int pageIndicatorCount = 1;
	/** 设置是否需要加载 用于部分操作时单据数据不需要全部加载*/
	private boolean ignoreLoad = false;
	
	/** 表的来源 */
	private MetaTableSource source = null;

	/** 全文检索需要用到的数据结构 */
	private SolrMap solrMap = null;

	/** 排序条件 */
	private ArrayList<String> sortList = null;
	/** 排序类型 */
	private ArrayList<Boolean> sortTypeList = null;
	/** 属否查询总行数*/
	private Boolean queryTotalRowCount = null;
	
	public TableFilterDetail() {
	}

	public TableFilterDetail(String tableKey) {
		this.tableKey = tableKey;
	}

	/**
	 * 设置表来源
	 * @param source 来源
	 */
	public void setSource(MetaTableSource source) {
		this.source = source;
	}
	
	/**
	 * 取得表来源
	 * @return 表来源
	 */
	public MetaTableSource getSource() {
		return this.source;
	}
	
	/**
	 * 设置来源标识
	 * @param sourceKey 来源标识
	 */
	public void setSourceKey(String sourceKey) {
		this.sourceKey = sourceKey;
	}
	
	/**
	 * 取得来源标识
	 * @return 来源标识
	 */
	public String getSourceKey() {
		return this.sourceKey;
	}

	public void addFieldValueCondition(String fieldKey, Object value) {
		if (conditionValueMap == null)
			conditionValueMap = new StringHashMap<Object>();
		conditionValueMap.put(fieldKey, value);
	}

	public void appendParaValue(Object value) {
		if (paraValues == null)
			paraValues = new ArrayList<Object>();

		paraValues.add(value);
	}

	public Object getParaValue(int index) {
		return paraValues.get(index);
	}

	public boolean containFieldValue(String fieldKey) {
		return conditionValueMap == null ? false : conditionValueMap.containsKey(fieldKey);
	}

	public Object getFieldValue(String fieldKey) {
		return conditionValueMap.get(fieldKey);
	}

	public void removeFieldValueCondition(String fieldKey) {
		if (conditionValueMap == null)
			return;
		conditionValueMap.remove(fieldKey);
	}

	public void clearFieldValueCondition() {
		if (conditionValueMap == null)
			return;
		conditionValueMap.clear();
	}

	/**
	 * 设置查询的起始行
	 * 
	 * @param startRow
	 *            行序号
	 */
	public void setStartRow(int startRow) {
		this.startRow = startRow;
	}

	/**
	 * 取得查询的起始行
	 * 
	 * @return 行序号
	 */
	public int getStartRow() {
		return startRow;
	}

	/**
	 * 取得查询的结束行，根据起始行和最大行数计算
	 * 后台分页需要往后多查几行
	 * 
	 * @return 查询的结束行
	 */
	public int getEndRow() {
		return this.startRow + this.maxRows * this.pageIndicatorCount;
	}

	/**
	 * 取得查询的最大行数
	 * 
	 * @return 行数
	 */
	public int getMaxRows() {
		return this.maxRows;
	}

	/**
	 * 设置查询的最大行数
	 * 
	 * @param maxRows
	 *            行数
	 */
	public void setMaxRows(int maxRows) {
		this.maxRows = maxRows;
	}

	/**
	 * 设置数据对象标识
	 * 
	 * @param OID
	 *            数据对象标识
	 */
	public void setOID(long OID) {
		this.OID = OID;
	}

	/**
	 * 取得数据对象标识
	 * 
	 * @return 数据对象标识
	 */
	public long getOID() {
		return this.OID;
	}
	
	
	/**
	 * 获取全文检索的辅助数据
	 * @return 全文检索辅助数据
	 */
	public SolrMap getSolrMap() {
		return solrMap;
	}

	/**
	 * 设置全文检索的辅助数据
	 * @param solrMap 全文检索辅助数据
	 */
	public void setSolrMap(SolrMap solrMap) {
		this.solrMap = solrMap;
	}
	
	
	
	public ArrayList<String> getSortList() {
		return sortList;
	}

	public ArrayList<Boolean> getSortTypeList() {
		return sortTypeList;
	}

	public void addOrderField(String fieldKey, Boolean value) {
		if (sortList == null) {
			sortList = new ArrayList<String>();
			sortTypeList = new ArrayList<Boolean>();
		}
		
		int index = sortList.indexOf(fieldKey);
		if (index > -1) {
			sortList.remove(index);
			sortTypeList.remove(index);
		}

		sortList.add(fieldKey);
		sortTypeList.add(value);
	}
	
	public void clearOrderField() {
		if (sortList == null) {
			sortList = new ArrayList<String>();
			sortTypeList = new ArrayList<Boolean>();
		}
		
		sortList.clear();
		sortTypeList.clear();
	}
	
	public String getSortStr(Callback<String,String> callBack) throws Throwable {		
		StringBuilder SQL = new StringBuilder(" ");
		int size = sortList.size();
		for (int i = 0; i < size; ++i) {
			SQL.append(callBack.call(sortList.get(i)));
			if (!sortTypeList.get(i))
				SQL.append(" DESC");
			SQL.append(",");
		}
		SQL.delete(SQL.length() - 1, SQL.length());
		SQL.append(" ");
		return SQL.toString();
	}

	public boolean existSort(){
		return sortList != null && !sortList.isEmpty();
	}
	
	
	public JSONObject toJSON() throws SerializationException, JSONException {
		JSONObject tableFilterObj = new JSONObject();
		tableFilterObj.put(TableFilterJSONConstants.TABLEKEY, tableKey);
		tableFilterObj.put(TableFilterJSONConstants.STARTROW, startRow);
		tableFilterObj.put(TableFilterJSONConstants.MAXROWS, maxRows);
		tableFilterObj.put(TableFilterJSONConstants.OID, OID);
		tableFilterObj.put(TableFilterJSONConstants.SOURCEKEY, sourceKey);
		tableFilterObj.put(TableFilterJSONConstants.INDICATORCOUNT, pageIndicatorCount);
		/*处理solrMap*/
		if(solrMap != null){
			try{
				JSONObject jsonObj = solrMap.toJSON();
				tableFilterObj.put(TableFilterJSONConstants.SOLRMAP, jsonObj);
			}
			catch(Throwable e){
				
			}
			
		}
		
			
		JSONObject fieldValue = new JSONObject();

		if (conditionValueMap != null) {
			String fieldKey;
			Object v;
	
			for (Entry<String, Object> e : conditionValueMap.entrySet()) {
				fieldKey = e.getKey();
				v = e.getValue();
				fieldValue.put(fieldKey, v);
			}
			
		}
		tableFilterObj.put(TableFilterJSONConstants.FIELDVALUES, fieldValue);

		JSONArray sortArray = new JSONArray();
		if (sortList != null) {
			for (String value : sortList) {				
				sortArray.put(value);
			}
		}		
		tableFilterObj.put(TableFilterJSONConstants.ORDERFIELD, sortArray);
		
		JSONArray sortTypeArray = new JSONArray();
		if (sortTypeList != null) {
			for (Boolean value : sortTypeList) {				
				sortTypeArray.put(value);
			}
		}		
		tableFilterObj.put(TableFilterJSONConstants.ORDERFIELDTYPE, sortTypeArray);

		return tableFilterObj;
	}

	public void fromJSON(JSONObject jsonObj) throws SerializationException, JSONException {
		this.tableKey = JSONHelper.readFromJSON(jsonObj, TableFilterJSONConstants.TABLEKEY, "");
		this.startRow = JSONHelper.readFromJSON(jsonObj, TableFilterJSONConstants.STARTROW, 0);
		this.maxRows = JSONHelper.readFromJSON(jsonObj, TableFilterJSONConstants.MAXROWS, 0);
		this.OID = JSONHelper.readFromJSON(jsonObj, TableFilterJSONConstants.OID, -1);
		this.sourceKey = JSONHelper.readFromJSON(jsonObj, TableFilterJSONConstants.SOURCEKEY, "");
		this.pageIndicatorCount = JSONHelper.readFromJSON(jsonObj, TableFilterJSONConstants.INDICATORCOUNT, 1);
		/*处理solrmap*/
		JSONObject mapObject = jsonObj.optJSONObject(TableFilterJSONConstants.SOLRMAP);
		if(mapObject != null){
			try{
				this.solrMap = new SolrMap();
				this.solrMap.fromJSON(mapObject);
			}
			catch(Throwable e){
				this.solrMap = null;
			}
		}
		
		conditionValueMap = new StringHashMap<Object>();

		JSONObject fieldValues = jsonObj.optJSONObject(TableFilterJSONConstants.FIELDVALUES);
		if (fieldValues != null && fieldValues.length() > 0) {

			Iterator<?> it = fieldValues.keys();
			String key;
			Object value;
			while (it.hasNext()) {
				key = TypeConvertor.toString(it.next());
				value = fieldValues.get(key);

				conditionValueMap.put(key, value);
			}
		}

		sortList = new ArrayList<String>();
		JSONArray sortArray = jsonObj.getJSONArray(TableFilterJSONConstants.ORDERFIELD);
		for (int i = 0; i < sortArray.length(); i++) {
			sortList.add(sortArray.getString(i));
		}
		
		sortTypeList = new ArrayList<Boolean>();
		JSONArray sortTypeArray = jsonObj.getJSONArray(TableFilterJSONConstants.ORDERFIELDTYPE);
		for (int i = 0; i < sortTypeArray.length(); i++) {
			sortTypeList.add(sortTypeArray.getBoolean(i));
		}
		
	}

	public String getTableKey() {
		return tableKey;
	}

	public boolean isLimit() {
		return this.maxRows > 0;
	}
	
	public int getPageIndicatorCount() {
		return pageIndicatorCount;
	}

	public void setPageIndicatorCount(int pageIndicatorCount) {
		this.pageIndicatorCount = pageIndicatorCount;
	}

	public ArrayList<Object> getParaValues() {
		return paraValues;
	}

	public void setParaValues(ArrayList<Object> paraValues) {
		this.paraValues = paraValues;
	}
	
	public void setIgnoreLoad(boolean b){
		this.ignoreLoad = b;
	}
	
	public boolean isIgnoreLoad(){
		return this.ignoreLoad;
	}
	
	/**
	 * 设置是否查询总行数
	 * @param b
	 */
	public void setQueryTotalRowCount(Boolean b) {
		this.queryTotalRowCount = b;
	}
	
	/**
	 * 是否查询总行数
	 * @return 是否查询总行数
	 */
	public Boolean needQueryTotalRowCount() {
		return this.queryTotalRowCount;
	}
	
	@SuppressWarnings("unchecked")
	protected Object clone() {
		TableFilterDetail newObject = new TableFilterDetail();
		newObject.tableKey = tableKey;
		newObject.conditionValueMap = conditionValueMap == null ? null : (StringHashMap<Object>)conditionValueMap.clone();
		newObject.paraValues = paraValues == null ? null : (ArrayList<Object>)paraValues.clone();
		newObject.startRow = startRow;
		newObject.maxRows = maxRows;
		newObject.OID = OID;
		newObject.sourceKey = sourceKey;
		newObject.pageIndicatorCount = pageIndicatorCount;
		newObject.ignoreLoad = ignoreLoad;
		newObject.source = source == null ? null : (MetaTableSource)source.clone();
		newObject.solrMap = solrMap == null ? null : (SolrMap)solrMap.clone();
		newObject.sortList = sortList == null ? null : (ArrayList<String>)sortList.clone();
		newObject.queryTotalRowCount = queryTotalRowCount;
		newObject.sortTypeList = sortTypeList == null ? null : (ArrayList<Boolean>)sortTypeList.clone();
		return newObject;
	}
}
