package com.bokesoft.yigo.struct.datatable;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.lang3.ArrayUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.SerializationException;
import com.bokesoft.yes.struct.abstractdatatable.IDataTable;
import com.bokesoft.yes.struct.abstractdatatable.RowState;
import com.bokesoft.yes.struct.abstractdatatable.SortCriteria;
import com.bokesoft.yes.struct.datatable.Row;
import com.bokesoft.yes.struct.datatable.filter.FilterEval;
import com.bokesoft.yes.struct.datatable.filter.FormulaFilter;
import com.bokesoft.yes.struct.datatable.json.AllDataRowsImpl;
import com.bokesoft.yes.struct.datatype.DataTypeAction;
import com.bokesoft.yes.struct.document.DocumentJSONConstants;
import com.bokesoft.yes.struct.document.IJSONHandler;
import com.bokesoft.yigo.common.def.SystemField;
import com.bokesoft.yigo.common.json.JSONSerializable;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.struct.exception.StructException;

/**
 * <code>DataTable</code>
 * 是用于表示规则二维表数据的模型。DataTable的由行列组成，列的数量是有限的，有序且拥有唯一标识（即为列的KEY
 * ），有指定的数据类型；而行的数量没有限制的，但是是有序的 。所以表格中的每个单元都是可以用两个整数来引用（行、列均从0开始按顺标记）。
 * <p>
 * 列：
 * <p>
 * 创建一个DataTable的新对象时，DataTable为0行0列，要得到想要的表格，先必须在表格中按照顺序添加列。例如想要创建一个表格，
 * 第一列的KEY为AA，数据类型为 INT，第二列的KEY为BB，数据类型为String：
 * 
 * <pre>
 * DataTable table = new DataTable();
 * table.addColumn(new ColumnInfo(&quot;AA&quot;, DataType.INT));
 * table.addColumn(new ColumnInfo(&quot;BB&quot;, DataType.STRING));
 * </pre>
 * 
 * <p>
 * 行：
 * <p>
 * 当列的定义完成后，就可以在表格中插入行，插入的行为空白行，其中的单元格的值均为空，数据类型由单元格所在的列决定。同时表格中的行是支持游标操作，所以
 * 关于行的操作均支持当前行、指定行两种方式（插入行支持在表中最末插入一行）。例如，要在表格末添加一个空白行，然后在第一行之前插入一行：
 * 
 * <pre>
 * table.appendRow();
 * table.insertRow(0);
 * </pre>
 * <p>
 * 单元格：
 * <p>
 * 表格中有数据后，然后就可以对单元格进行设值、取值操作
 * 。单元格的定位是由行、列的定位决定的，其中行的定位支持两种形式：当前行、行的序号；列的定位支持两种形式
 * ：列的键值、列的序号。例如，将m行n列的单元格的值改为o
 * 
 * <pre>
 * table.setObject(m, n, o);
 * </pre>
 * 
 * @author 刘翔翔
 * @see ColumnInfo
 * @see DataTableMetaData
 */
public class DataTable implements IDataTable<Row>, Externalizable, Comparator<Row>, JSONSerializable, IJSONHandler {

	/** 当前可见行的视图 */
	protected ArrayList<Row> viewDataList = new ArrayList<Row>();

	/** 所有的行数据 */
	private ArrayList<Row> allDataList = new ArrayList<Row>();

	/** 行对象的哈希映射表，是行的BookMark和Index的Map关系 */
	private HashMap<Integer, Integer> bookmarkMap = new HashMap<Integer, Integer>();

	/** 表格的元数据 */
	protected DataTableMetaData metaData = null;

	/** 表格的过滤条件 */
	private FilterEval filterEval = null;

	/** 是否将已经删除的行显示出来 */
	private boolean showDeleted = false;

	/** 游标 */
	private int pos = -1;

	/** 排序条件 */
	private SortCriteria[] vCriteria = null;

	/** 行的唯一编号的计数器 */
	private int bookmarkSeed = 0;

	/** 表的标识 */
	private String key = "";

	/** 缓存提供者，主要用于游标 */
	private ICacheProvider cache = null;

	/**
	 * DataTable的无参构造器
	 */
	public DataTable() {
		metaData = new DataTableMetaData();
	}
	
	/**
	 * DataTable的有参构造器
	 * @param metaData 表格的元数据
	 */
	public DataTable(DataTableMetaData metaData) {
		this.metaData = metaData;
	}

	/**
	 * 在最后一列之后再添加一列
	 * 
	 * @param column
	 *            列的信息
	 * @return 添加列的序号
	 * @throws StructException
	 *             表格结构异常
	 */
	public int addColumn(ColumnInfo column) throws StructException {
		metaData.addColumn(column);
		int index = metaData.getColumnCount() - 1;
		doColumnDataChange(index, true);
		if (SystemField.SEQUENCE_SYS_KEY.equals(column.getColumnKey())) {
			this.updateSequenceNumbers(0);
		}
		return index;
	}

	/**
	 * 在指定的位置插入一列，0～（columnIndex-1）列位置不变，columnIndex的位置放入新加列，其它列顺序向后位移一列
	 * 
	 * @param columnIndex
	 *            位置的序号，范围[0,N],N为列的数量
	 * @param column
	 *            列的信息
	 * @return 添加列的序号
	 * @throws StructException
	 *             表格结构异常
	 */
	public int addColumn(int columnIndex, ColumnInfo column) throws StructException {
		metaData.addColumn(columnIndex, column);
		doColumnDataChange(columnIndex, true);
		if (SystemField.SEQUENCE_SYS_KEY.equals(column.getColumnKey())) {
			this.updateSequenceNumbers(0);
		}
		return columnIndex;
	}

	/**
	 * 根据指定的键值删除对应的列
	 * 
	 * @param columnKey
	 *            列的键值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void removeColumn(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey);
		metaData.removeColumn(columnKey);
		doColumnDataChange(col, false);
	}

	/**
	 * 执行列变更引发的行数据变动
	 * 
	 * @param index
	 *            列的序号
	 * @param add
	 *            是否为添加列
	 */
	private final void doColumnDataChange(int index, boolean add) {

	}

	/**
	 * 返回表格的元数据
	 * 
	 * @return 表格的元数据
	 * @see DataTableMetaData
	 */
	public DataTableMetaData getMetaData() {
		return metaData;
	}

	/**
	 * 设置表格过滤条件公式
	 * 
	 * @param sFilter
	 *            过滤条件
	 * @throws Throwable 处理异常
	 */
	public void setFilter(String sFilter) throws Throwable {
		if (filterEval instanceof FormulaFilter)
			((FormulaFilter) filterEval).setFilter(sFilter);
		else
			filterEval = new FormulaFilter(sFilter);
	}

	/**
	 * 设置表格过滤条件对象
	 * 
	 * @param filterEval
	 *            表格过滤条件对象
	 */
	public void setFilterEval(FilterEval filterEval) {
		this.filterEval = filterEval;
	}

	/**
	 * 获取表格当前的过滤条件公式（若当前过滤条件不是公式形式的则返回空指针）
	 * 
	 * @return 表格当前的过滤条件公式
	 */
	public String getFilter() {
		if (filterEval instanceof FormulaFilter) {
			return ((FormulaFilter) filterEval).getFilter();
		}
		return null;
	}

	/**
	 * 执行过滤，游标移动到-1的位置。过滤条件为null或者空字符串时，显示全部数据（已删除的数据按照属性showDeleted来决定是否显示）
	 * 
	 * @throws Throwable
	 *             过滤异常，比如格式错误
	 * 
	 * @see DataTable#showDeleted
	 */
	public void filter() throws Throwable {
		if (filterEval == null || !filterEval.needCheck()) {
			viewDataList.clear();
			bookmarkMap.clear();
			for (Row r : allDataList) {
				if (!showDeleted && r.isDeleted())
					continue;
				viewDataList.add(r);
				bookmarkMap.put(r.getBookmark(), viewDataList.size() - 1);
			}
		} else {
			filterEval.init(this);
			viewDataList.clear();
			for (Row r : allDataList) {
				if (!showDeleted && r.isDeleted())
					continue;
				viewDataList.add(r);
			}

			ArrayList<Integer> removeList = new ArrayList<Integer>();

			beforeFirst();
			while (next()) {
				if (!filterEval.filterCheck(getRowByIndex(pos)))
					removeList.add(0, pos);
			}

			for (int index : removeList)
				viewDataList.remove(index);

			bookmarkMap.clear();
			beforeFirst();
			while (next())
				bookmarkMap.put(getBookmark(), pos);
			beforeFirst();
			// TODO: 这里重新考虑Index的情况
		}
		updateSequenceNumbers(0);
		pos = -1;
	}

	/**
	 * 设置表格过滤条件公式，并返回过滤结果（但不真正执行过滤）
	 * 
	 * @param sFilter 过滤
	 * @return 过滤结果的BookMark的数组
	 * @throws Throwable 处理异常
	 * 
	 */
	public ArrayList<Integer> filter(String sFilter) throws Throwable {
		if (filterEval instanceof FormulaFilter)
			((FormulaFilter) filterEval).setFilter(sFilter);
		else
			filterEval = new FormulaFilter(sFilter);
		return getfilterRet();
	}

	/**
	 * 设置表格过滤条件公式，并返回过滤结果（但不真正执行过滤）
	 * 
	 * @param filterEval 过滤
	 * @return 过滤结果的BookMark的数组
	 * @throws Throwable 处理异常
	 */
	public ArrayList<Integer> filter(FilterEval filterEval) throws Throwable {
		this.filterEval = filterEval;
		return getfilterRet();
	}

	/**
	 * 执行过滤规则，得出行的BookMark的数组，但不对表格执行过滤
	 * 
	 * @return 过滤结果的BookMark的数组
	 * @throws Throwable 处理异常
	 */
	private ArrayList<Integer> getfilterRet() throws Throwable {
		ArrayList<Integer> bkList = new ArrayList<>();
		if (filterEval == null || !filterEval.needCheck()) {
			// 无过滤规则是
			for (Row r : allDataList) {
				if (!showDeleted && r.isDeleted())
					continue;
				bkList.add(r.getBookmark());
			}
		} else {
			// 避免副作用
			filterEval.init(this);
			for (Row r : allDataList) {
				if (!showDeleted && r.isDeleted())
					continue;
				if (filterEval.filterCheck(r))
					bkList.add(r.getBookmark());
			}
		}
		return bkList;
	}

	/**
	 * 定义排序的方式，使用某一列进行升序排序的便捷方法，此方法仅进行参数设置，不执行实际排序
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param ascending
	 *            是否为升序
	 * @throws StructException
	 *             列不存在或者列的数据类型不支持排序
	 */
	public void setSort(String columnKey, boolean ascending) throws StructException {
		vCriteria = new SortCriteria[] { new SortCriteria(columnKey, ascending) };
		metaData.getColumnInfo(columnKey).getDataTypeAction().sortCheck();
	}

	/**
	 * 定义排序的方式，按照给定的若个列以及每列的升序、降序参数进行排序，此方法仅进行参数设置，不执行实际排序
	 * 
	 * @param vCriteria
	 *            排序的参数的数组
	 * @throws StructException
	 *             列不存在或者列的数据类型不支持排序
	 * @see DataTable#vCriteria
	 */
	public void setSort(SortCriteria[] vCriteria) throws StructException {
		this.vCriteria = vCriteria;
		if (vCriteria == null)
			return;
		for (SortCriteria s : vCriteria) {
			String key = s.getColumnKey();
			metaData.getColumnInfo(key).getDataTypeAction().sortCheck();
		}
	}

	/**
	 * 获取当前的排序条件
	 * 
	 * @return 当前的排序条件
	 */
	public SortCriteria[] getSortCriteria() {
		return vCriteria;
	}
	
	/**
	 * 清除排序条件
	 */
	public void clearCriteria() {
		this.vCriteria = null;
	}

	
	public void sort() throws Throwable {
		this.sort(this);
	}
	
	/**
	 * 进行排序，游标移动到-1的位置。若排序规则为空，则恢复原始顺序。
	 * 
	 * @throws Throwable
	 *             排序异常，包括列不存在
	 * 
	 */
	public void sort(Comparator<? super Row> comparator) throws Throwable {
		if (vCriteria == null || vCriteria.length == 0) {
			viewDataList.clear();
			viewDataList.addAll(allDataList);
			filter();
		} else {
			Collections.sort(allDataList, comparator);// 如只排序viewDataList,toJSON有问题!
			viewDataList.clear();
			bookmarkMap.clear();
			for (Row r : allDataList) {
				if (!showDeleted && r.isDeleted())
					continue;
				viewDataList.add(r);
				bookmarkMap.put(r.getBookmark(), viewDataList.size() - 1);
			}
		}
		updateSequenceNumbers(0);
		pos = -1;
	}
	

	/**
	 * 设置是否显示已经删除的行,调用此方法即完成过滤
	 * 
	 * @param showDeleted
	 *            是否显示已经删除的行
	 * @throws Throwable
	 *             表格结构异常
	 */
	public void setShowDeleted(boolean showDeleted) throws Throwable {
		if (showDeleted != this.showDeleted) {
			this.showDeleted = showDeleted;
			if (!(showDeleted && this.viewDataList.size() == this.allDataList.size())) { // 如果显示删除，可视行和所有行相同
				filter();
			}
		}
	}
	
	/**
	 * 设置是否显示已经删除的行的状态，但不完成过滤，内部函数，请勿在二次开发中使用
	 * @param showDeleted 是否显示已经删除的行的状态
	 */
	@Deprecated
	public void setShowDeletedFlag(boolean showDeleted) {
		this.showDeleted = showDeleted;
	}

	/**
	 * 清除当前的过滤条件,并使用为空的条件做一次过滤
	 * 
	 * @throws Throwable
	 *             表格结构异常
	 */
	public void clearFilter() throws Throwable {
		setFilterEval(null);
		filter();
	}

	/**
	 * 对象序列化为二进制，范围包括dataTableMetaData、viewData
	 */
	public void writeExternal(ObjectOutput out) throws IOException {
		metaData.writeExternal(out);
		int size = viewDataList.size();
		out.writeInt(size);
		for (Row r : viewDataList)
			r.writeExternal(out);
	}

	/**
	 * 二进制序列化为对象，范围包括dataTableMetaData、allData、viewData
	 */
	public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
		metaData = new DataTableMetaData();
		metaData.readExternal(in);
		int size = in.readInt();
		for (int i = 1; i <= size; i++) {
			Row r = new Row(in);
			allDataList.add(r);
			viewDataList.add(r);
		}
	}

	/**
	 * 返回DataTable按值复制的对象，复制的对象元数据与当前相同，
	 * 复制的对象allDataList、viewDataList的值相同（从当前对象的viewDataList中复制）
	 * ，不包含过滤条件、排序，且游标为-1。
	 * 
	 * @return DataTable按值复制的对象
	 */
	public DataTable deepClone() {
		DataTable clone = new DataTable();
		clone.setKey(key);
		clone.bookmarkSeed = this.bookmarkSeed;
		clone.metaData = metaData.deepClone();
		for (Row r : viewDataList) {
			Row rc = r.deepClone(clone);
			clone.allDataList.add(rc);
			clone.viewDataList.add(rc);
			clone.bookmarkMap.put(rc.getBookmark(), clone.viewDataList.size() - 1);
		}
		return clone;
	}

	// 游标的查询与操作

	/**
	 * 设置当前游标的位置，有效参数范围为[-1,N]，N为行的数量
	 * 
	 * @param pos
	 *            游标的位置
	 * @throws StructException
	 *             游标越界
	 */
	public void setPos(int pos) throws StructException {
		if (pos < -1 || pos > viewDataList.size())
			throw new StructException(StructException.OUT_OF_BOUNDS, "position: " + pos + ", Size: " + viewDataList.size());
		this.pos = pos;
	}

	/**
	 * 查询当前游标的位置
	 * 
	 * @return 游标的位置
	 */
	public int getPos() {
		return pos;
	}

	/**
	 * 游标移至第一行之前
	 */
	public void beforeFirst() {
		pos = -1;
	}

	/**
	 * 游标移至最后一行之后
	 */
	public void afterLast() {
		pos = viewDataList.size();
	}

	/**
	 * 游标下移一行（若游标已经到达最后一行之后，游标不移动）
	 * 
	 * @return 如果新的当前行有效，则返回 <code>true</code>；如果不存在下一行，则返回 <code>false</code>
	 */
	public boolean next() {
		if (isAfterLast())
			return false;
		pos++;
		return !isAfterLast();
	}

	/**
	 * 游标下移一行(带缓冲方式)
	 * 
	 * @deprecated 只在内部数据处理计算中使用，外部程序请使用next()
	 * @param fetch 语法标志，无用
	 * @return 如果新的游标有效，舞台true，否则返回false
	 */
	public boolean next(boolean fetch) {
		if (this.cache != null) {
			return this.cache.next(this);
		} else {
			return this.next();
		}
	}

	/**
	 * 游标移到上一行（若游标已经到达第一行之前，游标不移动）
	 * 
	 * @return 如果新的当前行有效，则返回 <code>true</code>；如果不存在上一行，则返回 <code>false</code>
	 */
	public boolean previous() {
		if (isBeforeFirst())
			return false;
		pos--;
		return !isBeforeFirst();
	}

	/**
	 * 游标移到第一行，0行的情况返回false
	 * 
	 * @return 是否操作成功
	 */
	public boolean first() {
		if (viewDataList.size() == 0)
			return false;
		pos = 0;
		return true;
	}

	/**
	 * 游标移到最后一行，0行的情况返回false
	 * 
	 * @return 是否操作成功
	 */
	public boolean last() {
		if (viewDataList.size() == 0)
			return false;
		pos = viewDataList.size() - 1;
		return true;
	}

	/**
	 * 获取当前可见行的数量
	 * 
	 * @return 当前可见行的数量
	 */
	public int size() {
		return viewDataList.size();
	}

	/**
	 * 判断当前表是否为空，即没有行数据
	 * 
	 * @return 如果当前表为空则返回true，否则返回false
	 */
	public boolean isEmpty() {
		return this.viewDataList.isEmpty();
	}

	/**
	 * 查询游标是否在第一行之前
	 * 
	 * @return 游标是否在第一行之前
	 */
	public boolean isBeforeFirst() {
		return pos == -1;
	}

	/**
	 * 查询游标是否在最后一行之后
	 * 
	 * @return 游标是否在最后一行之后
	 */
	public boolean isAfterLast() {
		return pos == viewDataList.size();
	}

	/**
	 * 查询游标是否在第一行
	 * 
	 * @return 游标是否在第一行
	 */
	public boolean isFirst() {
		return !viewDataList.isEmpty() && pos == 0;
	}

	/**
	 * 查询游标是否在最后一行
	 * 
	 * @return 游标是否在最后一行
	 */
	public boolean isLast() {
		return !viewDataList.isEmpty() && pos == viewDataList.size() - 1;
	}
	
	/**
	 * 判断是否为最后一行数据(带缓冲方式)
	 * @deprecated 只在内部数据处理计算中使用，外部程序请使用isLast()
	 * @param fetch 语法标志，无用
	 * @return
	 */
	public boolean isLast(boolean fetch) {
		if (this.cache != null) {
			return this.cache.isLast();
		} else {
			return this.isLast();
		}
	}

	/**
	 * 判断当前浮标位置是否有效
	 * 
	 * @return 如果总行数不为空，浮标大于等于0并且小于总行数，返回true，否则返回false
	 */
	public boolean isValid() {
		return !viewDataList.isEmpty() && pos >= 0 && pos < viewDataList.size();
	}

	// OriginalObject的get方法

	/**
	 * 查询指定单元格上一次初始化重置时的值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格上一次初始化重置时的值
	 */
	public Object getOriginalObject(int rowIndex, int columnIndex) {
		return viewDataList.get(rowIndex).getOriginalObject(columnIndex);
	}

	/**
	 * 查询指定单元格上一次初始化重置时的值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格上一次初始化重置时的值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Object getOriginalObject(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getOriginalObject(rowIndex, col);
	}

	/**
	 * 查询当前行指定列上一次初始化重置时的值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格上一次初始化重置时的值
	 */
	public Object getOriginalObject(int columnIndex) {
		return getOriginalObject(pos, columnIndex);
	}

	/**
	 * 查询当前行指定列上一次初始化重置时的值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格上一次初始化重置时的值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Object getOriginalObject(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getOriginalObject(pos, col);
	}

	/**
	 * 查询指定单元格的Object值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Object值
	 */
	public Object getObject(int rowIndex, int columnIndex) {
		return viewDataList.get(rowIndex).getObject(columnIndex);
	}

	/**
	 * 设置指定单元格的Object值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Object值
	 */
	public void setObject(int rowIndex, int columnIndex, Object value) {
		checkLength(columnIndex, value);
		viewDataList.get(rowIndex).setObject(columnIndex, value);
	}

	/**
	 * 设置指定单元格为无穷值
	 *
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 */
	public void setInfinity(int rowIndex, int columnIndex) {
		viewDataList.get(rowIndex).setObject(columnIndex, TypeConvertor.INFINITY);
	}

	/**
	 * 设置指定单元格的Object值,控制是否改变行状态
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的值
	 * @param changeState
	 *            是否改变行的状态
	 */
	public void setObject(int rowIndex, int columnIndex, Object value, boolean changeState) {
		checkLength(columnIndex, value);
		viewDataList.get(rowIndex).setObject(columnIndex, value, changeState);
	}
	
	public void setCheckLength(boolean checkLength) {
		this.checkLength = checkLength;
	}

	private boolean checkLength = false;
	
	private void checkLength(int columnIndex, Object value) {
		if (!checkLength) {
			return;
		}
		if (key != null && key.endsWith("_NODB")) {
			return;
		}
		ColumnInfo columnInfo = metaData.getColumnInfo(columnIndex);
		if (columnInfo.getDataType() == com.bokesoft.yigo.common.def.DataType.STRING && value != null) {
			if(!columnInfo.getCheckLength()){
				return;
			}
			String colKey = columnInfo.getColumnKey();
			if (colKey.endsWith("_NODB") || colKey.endsWith("_NODBItemKey")) {
				return;
			}
			long l1 = columnInfo.getLength();
			long l2 = value.toString().length();
			if (l1 < l2) {
				throw new StructException(StructException.COLUMN_LENGTH_ERROR, StructException.formatMessage(null,
						StructException.COLUMN_LENGTH_ERROR, this.getKey(), columnInfo.getColumnKey(), l1, value, l2));
			}
		}
	}

	/**
	 * 查询指定单元格的Object值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Object值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Object getObject(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
//		if (col == -1){
//			throw new StructException(StructException.UNFINDED_COLUMN, 
//					StructException.formatMessage(null, StructException.UNFINDED_COLUMN,columnKey));
//		}
		return getObject(rowIndex, col);
	}

	/**
	 * 查询指定单元格的Object值，内部实现，二次开发中请勿使用
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Object值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Object impl_getObject(int rowIndex, String columnKey) throws StructException {
		int col = metaData.impl_findColumnIndexByKey(columnKey, true);
		return getObject(rowIndex, col);
	}

	/**
	 * 设置指定单元格的Object值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Object值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setObject(int rowIndex, String columnKey, Object value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setObject(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的Object值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Object值
	 */
	public Object getObject(int columnIndex) {
		return getObject(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的Object值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Object值
	 */
	public void setObject(int columnIndex, Object value) {
		setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的Object值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Object值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Object getObject(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
//		if (col == -1){
//			throw new StructException(StructException.UNFINDED_COLUMN, 
//					StructException.formatMessage(null, StructException.UNFINDED_COLUMN,columnKey));
//		}
		return getObject(pos, col);
	}

	/**
	 * 查询当前行指定列的Object值，内部实现，二次开发中请勿使用
	 * 
	 * @param columnKey
	 *            列的键值，必须为全小写
	 * @return 单元格的Object值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Object impl_getObject(String columnKey) throws StructException {
		int col = metaData.impl_findColumnIndexByKey(columnKey, true);
		return getObject(pos, col);
	}

	/**
	 * 设置当前行指定列的Object值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Object值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setObject(String columnKey, Object value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setObject(pos, col, value);
	}

	/**
	 * 设置当前行指定列的Object值,控制是否改变行状态
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Object值
	 * @param changeState
	 *            是否改变行状态
	 * 
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setObject(String columnKey, Object value, boolean changeState) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setObject(pos, col, value, changeState);
	}

	// Int的get/set方法

	/**
	 * 查询指定单元格的Int值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Int值
	 */
	public Integer getInt(int rowIndex, int columnIndex) {
		Object o = getObject(rowIndex, columnIndex);
		return TypeConvertor.toInteger(o);
	}

	/**
	 * 设置指定单元格的Int值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Int值
	 */
	public void setInt(int rowIndex, int columnIndex, Integer value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的Int值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Int值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Integer getInt(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getInt(rowIndex, col);
	}

	/**
	 * 设置指定单元格的Int值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Int值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setInt(int rowIndex, String columnKey, Integer value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setInt(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的Int值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Int值
	 */
	public Integer getInt(int columnIndex) {
		return getInt(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的Int值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Int值
	 */
	public void setInt(int columnIndex, Integer value) {
		this.setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的Int值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Int值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Integer getInt(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
//		if (col == -1)
//			return -1;
		return getInt(pos, col);
	}

	/**
	 * 设置当前行指定列的Int值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Int值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setInt(String columnKey, Integer value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setInt(pos, col, value);
	}

	// String的get/set方法

	/**
	 * 查询指定单元格的String值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的String值
	 */
	public String getString(int rowIndex, int columnIndex) {
		return (String) getObject(rowIndex, columnIndex);
	}

	/**
	 * 设置指定单元格的String值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的String值
	 */
	public void setString(int rowIndex, int columnIndex, String value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的String值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的String值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public String getString(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getString(rowIndex, col);
	}

	/**
	 * 设置指定单元格的String值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的String值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setString(int rowIndex, String columnKey, String value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setString(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的String值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的String值
	 */
	public String getString(int columnIndex) {
		return getString(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的String值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的String值
	 */
	public void setString(int columnIndex, String value) {
		setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的String值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的String值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public String getString(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getString(pos, col);
	}

	/**
	 * 设置当前行指定列的String值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的String值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setString(String columnKey, String value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setString(pos, col, value);
	}

	// DateTime的get/set方法

	/**
	 * 查询指定单元格的DateTime值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的DateTime值
	 */
	public Date getDateTime(int rowIndex, int columnIndex) {
		return (Date) getObject(rowIndex, columnIndex);
	}

	/**
	 * 设置指定单元格的DateTime值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的DateTime值
	 */
	public void setDateTime(int rowIndex, int columnIndex, Date value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的DateTime值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的DateTime值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Date getDateTime(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getDateTime(rowIndex, col);
	}

	/**
	 * 设置指定单元格的DateTime值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的DateTime值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setDateTime(int rowIndex, String columnKey, Date value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setDateTime(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的DateTime值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的DateTime值
	 */
	public Date getDateTime(int columnIndex) {
		return getDateTime(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的DateTime值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的DateTime值
	 */
	public void setDateTime(int columnIndex, Date value) {
		setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的DateTime值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的DateTime值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Date getDateTime(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getDateTime(pos, col);
	}

	/**
	 * 设置当前行指定列的DateTime值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的DateTime值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setDateTime(String columnKey, Date value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setDateTime(pos, col, value);
	}

	// Numeric的get/set方法

	/**
	 * 查询指定单元格的Numeric值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Numeric值
	 */
	public BigDecimal getNumeric(int rowIndex, int columnIndex) {
		return TypeConvertor.toBigDecimal(getObject(rowIndex, columnIndex));
	}

	/**
	 * 设置指定单元格的Numeric值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Numeric值
	 */
	public void setNumeric(int rowIndex, int columnIndex, BigDecimal value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的Numeric值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Numeric值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public BigDecimal getNumeric(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getNumeric(rowIndex, col);
	}

	/**
	 * 设置指定单元格的Numeric值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Numeric值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setNumeric(int rowIndex, String columnKey, BigDecimal value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setNumeric(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的Numeric值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Numeric值
	 */
	public BigDecimal getNumeric(int columnIndex) {
		return getNumeric(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的Numeric值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Numeric值
	 */
	public void setNumeric(int columnIndex, BigDecimal value) {
		setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的Numeric值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Numeric值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public BigDecimal getNumeric(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getNumeric(pos, col);
	}

	/**
	 * 设置当前行指定列的Numeric值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Numeric值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setNumeric(String columnKey, BigDecimal value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setNumeric(pos, col, value);
	}

	// Binary的get/set方法

	/**
	 * 查询指定单元格的Binary值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Binary值
	 */
	public byte[] getBinary(int rowIndex, int columnIndex) {
		return (byte[]) getObject(rowIndex, columnIndex);
	}

	/**
	 * 设置指定单元格的Binary值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Binary值
	 */
	public void setBinary(int rowIndex, int columnIndex, byte[] value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的Binary值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Binary值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public byte[] getBinary(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getBinary(rowIndex, col);
	}

	/**
	 * 设置指定单元格的Binary值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Binary值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setBinary(int rowIndex, String columnKey, byte[] value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setBinary(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的Binary值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Binary值
	 */
	public byte[] getBinary(int columnIndex) {
		return getBinary(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的Binary值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Binary值
	 */
	public void setBinary(int columnIndex, byte[] value) {
		setBinary(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的Binary值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Binary值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public byte[] getBinary(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getBinary(pos, col);
	}

	/**
	 * 设置当前行指定列的Binary值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Binary值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setBinary(String columnKey, byte[] value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setBinary(pos, col, value);
	}

	// Long的get/set方法

	/**
	 * 查询指定单元格的Long值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Long值
	 */
	public Long getLong(int rowIndex, int columnIndex) {
		Object o = getObject(rowIndex, columnIndex);
		return TypeConvertor.toLong(o);
	}

	/**
	 * 设置指定单元格的Long值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Long值
	 */
	public void setLong(int rowIndex, int columnIndex, Long value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的Long值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Long值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Long getLong(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getLong(rowIndex, col);
	}

	/**
	 * 设置指定单元格的Long值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Long值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setLong(int rowIndex, String columnKey, Long value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setLong(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的Numeric值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Long值
	 */
	public Long getLong(int columnIndex) {
		return getLong(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的Long值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Long值
	 */
	public void setLong(int columnIndex, Long value) {
		setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的Long值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Long值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Long getLong(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getLong(pos, col);
	}

	/**
	 * 设置当前行指定列的Long值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Long值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setLong(String columnKey, Long value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setLong(pos, col, value);
	}

	// Boolean的get/set方法

	/**
	 * 查询指定单元格的Boolean值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Boolean值
	 */
	public Boolean getBoolean(int rowIndex, int columnIndex) {
		return TypeConvertor.toBoolean(getObject(rowIndex, columnIndex));
	}

	/**
	 * 设置指定单元格的Boolean值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Boolean值
	 */
	public void setBoolean(int rowIndex, int columnIndex, Boolean value) {
		setObject(rowIndex, columnIndex, value);
	}

	/**
	 * 查询指定单元格的Boolean值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Boolean值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Boolean getBoolean(int rowIndex, String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getBoolean(rowIndex, col);
	}

	/**
	 * 设置指定单元格的Long值
	 * 
	 * @param rowIndex
	 *            行的序号
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Boolean值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setBoolean(int rowIndex, String columnKey, Boolean value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setBoolean(rowIndex, col, value);
	}

	/**
	 * 查询当前行指定列的Numeric值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @return 单元格的Boolean值
	 */
	public Boolean getBoolean(int columnIndex) {
		return getBoolean(pos, columnIndex);
	}

	/**
	 * 设置当前行指定列的Boolean值
	 * 
	 * @param columnIndex
	 *            列的序号
	 * @param value
	 *            单元格的Boolean值
	 */
	public void setBoolean(int columnIndex, Boolean value) {
		setObject(pos, columnIndex, value);
	}

	/**
	 * 查询当前行指定列的Long值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 单元格的Boolean值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public Boolean getBoolean(String columnKey) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		return getBoolean(pos, col);
	}

	/**
	 * 设置当前行指定列的Long值
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param value
	 *            单元格的Boolean值
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public void setBoolean(String columnKey, Boolean value) throws StructException {
		int col = metaData.findColumnIndexByKey(columnKey, true);
		setBoolean(pos, col, value);
	}

	// 行操作

	/**
	 * 在指定的位置插入一个空白行，插入后游标指向插入的行。（如果游标越界，向前越界会插入在0行，向后越界则插入在末尾行）
	 * 
	 * @param rowIndex
	 *            行插入的位置
	 * @return 插入后空白行的位置序号
	 */
	public int insert(int rowIndex) {
		if (rowIndex < 0)
			rowIndex = 0;
		else if (rowIndex > viewDataList.size())
			rowIndex = viewDataList.size();

		pos = rowIndex;

		Row r = new Row(this);
		allDataList.add(r);
		viewDataList.add(rowIndex, r);
		this.updateSequenceNumbers(rowIndex);
		bookmarkMap.put(r.getBookmark(), rowIndex);

		if (rowIndex < viewDataList.size() - 1) {
			for (int i = rowIndex + 1; i < viewDataList.size(); i++) {
				bookmarkMap.put(viewDataList.get(i).getBookmark(), i);
			}
		}
		if (indexes != null) {
			indexes.insertBookmark(r.getBookmark());
		}
		return rowIndex;
	}

	/**
	 * 在游标位置插入一个空白行，插入后游标指向插入的行
	 * 
	 * @return 插入后空白行的位置序号
	 */
	public int insert() {
		return insert(pos);
	}

	/**
	 * 在数据表末尾插入一个空白行，插入后游标移动至最后一行（即为新插入的空白行）
	 * 
	 * @return 游标位置
	 */
	public int append() {
		pos = viewDataList.size();
		return insert();
	}

	/**
	 * 删除指定行。如果指定行是新增行，行数据会被彻底删除。如果是，游标会保持仍然指向原先行
	 * 
	 * @param rowIndex
	 *            行的序号
	 */
	public void delete(int rowIndex) {
		Row r = viewDataList.get(rowIndex);

		int oldSize = viewDataList.size();

		if (r.isNew()) {
			allDataList.remove(r);
			bookmarkMap.remove(r.getBookmark());
			viewDataList.remove(rowIndex);
			if (pos > rowIndex)
				pos--;
		} else {
			r.setDeleted();
			if (!showDeleted) {
				viewDataList.remove(rowIndex);
				bookmarkMap.remove(r.getBookmark());
				if (pos > rowIndex)
					pos--;
			}
		}

		this.updateSequenceNumbers(rowIndex);
		int newSize = viewDataList.size();
		if (newSize != oldSize && rowIndex < newSize) {
			for (int i = rowIndex; i < newSize; i++) {
				bookmarkMap.put(viewDataList.get(i).getBookmark(), i);
			}
		}
		if (indexes != null) {
			indexes.deleteBookmark(r);
		}
	}

	/**
	 * 强制移除行，不仅仅是对行打DELETED标识
	 * @param rowIndex
	 */
	public void forceRemove(int rowIndex) {
		Row r = viewDataList.get(rowIndex);
		boolean isDeleted = r.isDeleted();

		int oldSize = viewDataList.size();

		allDataList.remove(r);
		bookmarkMap.remove(r.getBookmark());
		viewDataList.remove(rowIndex);
		if (pos > rowIndex)
			pos--;

		int newSize = viewDataList.size();
		if (newSize != oldSize && rowIndex < newSize) {
			for (int i = rowIndex; i < newSize; i++) {
				bookmarkMap.put(viewDataList.get(i).getBookmark(), i);
			}
		}
		if (!isDeleted && indexes != null) {
			indexes.deleteBookmark(r);
		}
	}
	
	/**
	 * 删除所有行数据
	 */
	public void deleteAll() {
		pos = -1;
		allDataList.removeAll(viewDataList);
		for (Row row : viewDataList) {
			if (!row.isNew()) {
				row.setDeleted();
				allDataList.add(row);
			}
		}

		viewDataList.clear();
		bookmarkMap.clear();
		if (showDeleted) {
			viewDataList.addAll(allDataList);
			for (int i = 0; i < viewDataList.size(); i++) {
				bookmarkMap.put(viewDataList.get(i).getBookmark(), i);
			}
		}
		if (indexes != null) {
			indexes.clear();
		}
	}
	
	/**
	 * 恢复已删除行，同时去掉DELETED标识
	 * @param row
	 * @throws Throwable
	 */
	public void recoverRow(Row row) throws Throwable {
		if (!row.isDeleted()) {
			throw new RuntimeException("DataTable.recoverRow()调用错误，请联系开发人员。");
		}
		row.setState(RowState.MODIFIED);
		viewDataList.add(row);
		int rowIndex = viewDataList.size() - 1;
		bookmarkMap.put(row.getBookmark(), rowIndex);
		if (indexes != null) {
			indexes.recoverBookmark(row);
		}
	}
	
	/**
	 * 批量修改行状态
	 */
	public void batchUpdate() {
		batchUpdate(false);
	}
	
	/**
	 * 清除表格中所有行的老数据，内部函数，请勿在二次开发中使用
	 */
	@Deprecated
	public void clearOriginalData() {
		for (Row r : allDataList) {
			r.resetOriginalData();
		}
	}

	/**
	 * 批量修改行状态，内部函数，请勿在二次开发中使用
	 * @param reserveOrgData 是否保留行的原值数据
	 */
	@Deprecated
	public void batchUpdate(boolean reserveOrgData) {
		ArrayList<Row> removeList = null;
		for (Row r : allDataList) {
			if (r.isDeleted()) {
				if (removeList == null) {
					removeList = new ArrayList<Row>();
				}
				removeList.add(r);
			} else {
				if ( reserveOrgData ) {
					r.setState(RowState.NORMAL);
				} else {
					r.stateUpdate();
				}
			}
		}
		
		if (removeList != null) {
			allDataList.removeAll(removeList);
			viewDataList.removeAll(removeList);
			bookmarkMap.clear();
			for (int i = 0; i < viewDataList.size(); i++) {
				bookmarkMap.put(viewDataList.get(i).getBookmark(), i);
			}
			removeList.clear();
		}

		// 如果有数据游标定位第一行
		if (size() > 0) {
			pos = 0;
		} else {
			pos = -1;
		}
	}

	/**
	 * 删除当前行，删除后游标不变
	 */
	public void delete() {
		delete(pos);
	}

	/**
	 * 获取当前行的状态
	 * 
	 * @return 当前行的状态
	 */
	public int getState() {
		return getState(pos);
	}

	/**
	 * 修改当前行的状态
	 * @param state 状态
	 */
	public void setState(int state) {
		viewDataList.get(pos).setState(state);
	}

	/**
	 * 获取指定行的行状态
	 * 
	 * @param rowIndex
	 *            行的序号，范围[0,N-1]，N为行数量
	 * @return 当前行的状态
	 */
	public int getState(int rowIndex) {
		return viewDataList.get(rowIndex).getState();
	}

	/**
	 * 多列并带有升降式参数的行比较方法
	 */
	public int compare(Row o1, Row o2) {
		for (SortCriteria s : vCriteria) {
			int index = 0;
			try {
				index = metaData.findColumnIndexByKey(s.getColumnKey());
			} catch (StructException e) {
				// 排序规则设置时，已经经过相关的检查，这里不处理结果错误的异常，直接忽略这一列
				continue;
			}
			int ret = metaData.getColumnInfo(index).getDataTypeAction().compare(o1.getObject(index), o2.getObject(index));
			ret = s.isAscending() ? ret : -ret;
			if (ret != 0)
				return ret;
		}
		return 0;
	}

	/**
	 * 分配一个新的行的唯一编号
	 * 
	 * @return 新的唯一编号
	 */
	public int applyNewBookMark() {
		return bookmarkSeed++;
	}

	/**
	 * 获取当前行的唯一编号
	 * 
	 * @return 当前行唯一编号
	 */
	public int getBookmark() {
		return viewDataList.get(pos).getBookmark();
	}

	/**
	 * 获取当前行的上层数据行的唯一编号
	 * 
	 * @return 取得当前行的父行的唯一编号
	 */
	public int getParentBookmark() {
		return viewDataList.get(pos).getParentBookmark();
	}

	/**
	 * 设置当前行的上层数据行的唯一编号
	 * 
	 * @param bookmark
	 *            新的上层数据行的唯一编号
	 */
	public void setParentBookmark(int bookmark) {
		viewDataList.get(pos).setParentBookmark(bookmark);
	}

	/**
	 * 获取指定行的上层数据行的唯一编号
	 * 
	 * @param rowIndex
	 *            行的序号，范围[0,N-1]，N为行数量
	 * @return 上层数据行的唯一标识
	 */
	public int getParentBookmark(int rowIndex) {
		return viewDataList.get(rowIndex).getParentBookmark();
	}

	/**
	 * 设置指定行的上层数据行的唯一编号
	 * 
	 * @param rowIndex
	 *            行的序号，范围[0,N-1]，N为行数量
	 * @param bookmark
	 *            新的上层数据行的唯一编号
	 */
	public void setParentBookmark(int rowIndex, int bookmark) {
		viewDataList.get(rowIndex).setParentBookmark(bookmark);
	}

	/**
	 * 获取行的唯一编号
	 * 
	 * @param rowIndex
	 *            行号
	 * @return 唯一编号
	 */
	public int getBookmark(int rowIndex) {
		return viewDataList.get(rowIndex).getBookmark();
	}
	
	/**
	 * 根据bkmk获取RowIndex
	 * @param bkmk
	 * @return
	 */
	public int getRowIndexByBookmark(int bkmk) {
		Integer index = bookmarkMap.get(bkmk);
		if (index == null) {
			return -1;
		}
		return index;
	}

	/**
	 * 通过唯一编号修改游标位置
	 * 
	 * @param bookmark 唯一编号
	 * @throws StructException 结构异常
	 */
	public void setBookmark(int bookmark) throws StructException {
		Integer index = bookmarkMap.get(bookmark);
		if (index != null) {
			pos = index;
			return;
		}

		throw new StructException(StructException.OUT_OF_BOUNDS, "bookmark: " + bookmark + ", Size: " + viewDataList.size());
	}

	/**
	 * 某唯一编号是否存在
	 * 
	 * @param bookmark 唯一编号
	 * @return 如果bookmark存在则返回true，否则返回false
	 */
	public boolean isBookmarkExist(int bookmark) {
		return bookmarkMap.containsKey(bookmark);
	}

	/**
	 * 取得标识
	 * @return 标识
	 */
	public String getKey() {
		return key;
	}

	/**
	 * 设置标识
	 * @param key 标识
	 */
	public void setKey(String key) {
		this.key = key.intern();
	}

	/**
	 * 取得DataTable的JSON表示的结构信息，无数据
	 * @return JSON对象
	 * @throws SerializationException 序列化异常
	 * @throws JSONException JSON异常
	 */
	public JSONObject toJSONWithNoData() throws SerializationException, JSONException {
		JSONObject jsonObj = new JSONObject();
		jsonObj.put(DocumentJSONConstants.DATATABLE_BOOKMARK_SEED, bookmarkSeed);
		jsonObj.put(DocumentJSONConstants.DATATABLE_TABLEKEY, key);
		jsonObj.put(DocumentJSONConstants.DATATABLE_CHECKLENGTH, checkLength);
		JSONArray columnArray = new JSONArray();
		for (int i = 0, size = metaData.getColumnCount(); i < size; i++) {
			ColumnInfo column = metaData.getColumnInfo(i);
			columnArray.put(column.toJSON());
		}
		jsonObj.put(DocumentJSONConstants.DATATABLE_COLUMNS, columnArray);
		return jsonObj;
	}

	/**
	 * 转换为json对象
	 * 
	 * @return json对象
	 * @throws Throwable 
	 * @throws StructException 结构异常
	 */
	public JSONObject toJSON() throws Throwable {
		JSONObject jsonObj = toJSONWithNoData();
		JSONArray allDataRowArray = new JSONArray();
		
		int sequenceColumnIndex = this.getMetaData().findColumnIndexByKey(SystemField.SEQUENCE_SYS_KEY);
		if (filterEval == null || !filterEval.needCheck()) {
			// 没有过滤,先放入viewDataList,再放入AllDataList中删除的行
			for (int i = 0; i < viewDataList.size(); i++) {
				Row row = viewDataList.get(i);
				if (sequenceColumnIndex >= 0) {
					row.setObject(sequenceColumnIndex, i + 1, false);
				}
				allDataRowArray.put(row.toJSON());
			}

			// 不显示删除行时,加入删除行
			if (!showDeleted) {
				for (int i = 0; i < allDataList.size(); i++) {
					Row row = allDataList.get(i);
					if (row.isDeleted()) {
						allDataRowArray.put(row.toJSON());
					}
				}
			}
		} else {
			// 有过滤,做过filter()后ViewDataList顺序和AllDataList一致。放入AllDataList
			for (int i = 0; i < allDataList.size(); i++) {
				Row row = allDataList.get(i);
				if (sequenceColumnIndex >= 0) {
					row.setObject(sequenceColumnIndex, i + 1, false);
				}
				allDataRowArray.put(row.toJSON());
			}
		}
		jsonObj.put(DocumentJSONConstants.DATATABLE_ALLDATAROWS, allDataRowArray);

		return jsonObj;
	}

	/**
	 * 从JSON对象获得对应的属性
	 * 
	 * @param tableObj
	 *            json对象
	 * @throws JSONException
	 *             JSON异常
	 * @throws StructException
	 *             表格结构异常
	 */
	public void fromJSON(JSONObject tableObj) throws SerializationException, JSONException {
		filterEval = null;
		pos = -1;
		showDeleted = false;
		vCriteria = null;
		metaData.clear();
		allDataList.clear();
		viewDataList.clear();
		bookmarkMap.clear();
		bookmarkSeed = tableObj.getInt(DocumentJSONConstants.DATATABLE_BOOKMARK_SEED);
		key = tableObj.getString(DocumentJSONConstants.DATATABLE_TABLEKEY);
		checkLength = tableObj.optBoolean(DocumentJSONConstants.DATATABLE_CHECKLENGTH, false);
		JSONArray columnArray = tableObj.getJSONArray(DocumentJSONConstants.DATATABLE_COLUMNS);
		for (int i = 0; i < columnArray.length(); i++) {
			JSONObject columnObj = columnArray.getJSONObject(i);
			ColumnInfo columnInfo = new ColumnInfo();
			columnInfo.fromJSON(columnObj);
			metaData.addColumn(columnInfo);
		}

		if (!tableObj.isNull(DocumentJSONConstants.DATATABLE_ALLDATAROWS)) {
			JSONArray allDataRowArray = tableObj.getJSONArray(DocumentJSONConstants.DATATABLE_ALLDATAROWS);
			for (int rowIndex = 0; rowIndex < allDataRowArray.length(); rowIndex++) {
				JSONObject rowObj = allDataRowArray.getJSONObject(rowIndex);
				Row row = new Row(this, rowObj);
				allDataList.add(row);
				if (row.getState() != RowState.DELETED) {
					viewDataList.add(row);
					bookmarkMap.put(row.getBookmark(), viewDataList.size() - 1);
				}
			}
		}
		
		//如果是从前台新增的行,客户端现在是根据Sequence字段排序的。这里需要再排序一下。
		if (metaData.constains(SystemField.SEQUENCE_SYS_KEY)) {
			Collections.sort(allDataList, new Comparator<Row>() {
				@Override
				public int compare(Row o1, Row o2) {
					Integer sequence1 = TypeConvertor.toInteger(o1.getObject(SystemField.SEQUENCE_SYS_KEY));
					Integer sequence2 = TypeConvertor.toInteger(o2.getObject(SystemField.SEQUENCE_SYS_KEY));
					int result = sequence1.compareTo(sequence2);
					return result;
				}
			});
			
			Collections.sort(viewDataList, new Comparator<Row>() {
				@Override
				public int compare(Row o1, Row o2) {
					Integer sequence1 = TypeConvertor.toInteger(o1.getObject(SystemField.SEQUENCE_SYS_KEY));
					Integer sequence2 = TypeConvertor.toInteger(o2.getObject(SystemField.SEQUENCE_SYS_KEY));
					int result = sequence1.compareTo(sequence2);
					return result;
				}
			});	
			
			// 排序后重建bookmarkMap
			bookmarkMap.clear();
			for (int index = 0; index < viewDataList.size(); index++) {
				bookmarkMap.put(viewDataList.get(index).getBookmark(), index);
			}
		}

		// 定位到第一行，方便处理
		this.first();
	}

	/**
	 * 返回是否显示删除行标志
	 * 
	 * @return 是否显示删除行标志
	 */
	public boolean isShowDeleted() {
		return showDeleted;
	}

	/**
	 * 根据列名和列的值查找满足条件的第一行，如果找到则设置为当前行并返回其行号
	 * 
	 * @param columnKey
	 *            列标识
	 * @param value
	 *            列的值
	 * @return 如果找到则返回行号，否则返回-1
	 */
	public int findRow(String columnKey, Object value) {
		int columnIndex = this.metaData.findColumnIndexByKey(columnKey, true);
		int index = -1;
		for (int i = 0, size = this.viewDataList.size(); i < size; ++i) {
			Object tmpValue = this.viewDataList.get(i).getObject(columnIndex);
			if (tmpValue.equals(value)) {
				index = i;
				this.setPos(index);
				break;
			}
		}
		return index;
	}

	/**
	 * 根据列序号和列的值查找满足条件的第一行，如果找到则设置为当前行并返回其行号
	 * 
	 * @param columnIndex
	 *            列序号
	 * @param value
	 *            列的值
	 * @return 如果找到则返回行号，否则返回-1
	 */
	public int findRow(int columnIndex, Object value) {
		int index = -1;
		for (int i = 0, size = this.viewDataList.size(); i < size; ++i) {
			Object tmpValue = this.viewDataList.get(i).getObject(columnIndex);
			if (tmpValue.equals(value)) {
				index = i;
				this.setPos(index);
				break;
			}
		}
		return index;
	}

	public void clear() {
		this.allDataList.clear();
		this.viewDataList.clear();
		this.bookmarkMap.clear();
		this.filterEval = null;
		this.vCriteria = null;
		this.pos = -1;
		if (indexes != null) {
			indexes.clear();
		}
		indexes = null;
	}

	/**
	 * 不建议使用这个方法，会破坏DataTable数据的正确性。
	 * 
	 */
	@Override
	@Deprecated
	public Row getRowByIndex(int rowIndex) {
		return viewDataList.get(rowIndex);
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		appendString(sb, false);
		return sb.toString();
	}

	public void appendString(StringBuilder sb, boolean enter) {
		if (enter)
			sb.append("\t");
		sb.append("<Table key=\"");
		sb.append(key);
		sb.append("\" showDeleted=\"");
		sb.append(showDeleted);
		sb.append("\" filter=\"");
		sb.append(filterEval);
		sb.append("\" curRowIndex=\"");
		sb.append(pos);
		sb.append("\" vCriteria=\"");
		sb.append(Arrays.toString(vCriteria));
		sb.append("\" bookmarkSeed=\"");
		sb.append(bookmarkSeed);
		sb.append("\">\n");

		for (Row r : viewDataList) {
			sb.append(enter ? "\t\t" : "\t");
			r.appendString(sb);
		}
		if (enter)
			sb.append("\t");
		sb.append("</Table>\n");
	}

	/**
	 * 复制一个新的空表，只包含表结构
	 * 
	 * @return 新的空表
	 * @throws StructException 结构异常
	 */
	public DataTable cloneEmpty() throws StructException {
		DataTable newObject = new DataTable();
		newObject.setKey(key);
		int columnCount = this.metaData.getColumnCount();
		for (int i = 0; i < columnCount; ++i) {
			newObject.addColumn(this.metaData.getColumnInfo(i));
		}

		return newObject;
	}

	/**
	 * 设置当前行为新增状态
	 */
	public void setNew() {
		viewDataList.clear();
		bookmarkMap.clear();
		filterEval = null;
		vCriteria = null;
		pos = -1;
		viewDataList.addAll(allDataList);

		for (int i = 0; i < allDataList.size(); i++) {
			Row row = allDataList.get(i);
			bookmarkMap.put(row.getBookmark(), i);
			row.setState(RowState.NEW);
			row.removeOriginalData();
		}

	}

	/**
	 * 取得当前行数据数组，内部实现
	 * 
	 * @return 行数据数组
	 */
	public Object[] impl_getRow() {
		return this.viewDataList.get(this.pos).getDataList();
	}

	@Override
	public IJSONHandler newHandler(String token) {
		if ("columns".equals(token)) {
			return this.metaData;
		} else if ("all_data_rows".equals(token)) {
			return new AllDataRowsImpl(this, this.viewDataList, this.allDataList, this.bookmarkMap);
		}
		return null;
	}

	@Override
	public void putAttr(String name, String value) {
		if ("bookmark_seed".equals(name)) {
			this.bookmarkSeed = Integer.parseInt(value);
		} else if ("key".equals(name)) {
			this.key = value;
		}
	}

	public void impl_SetBookmarkSeed(int value) {
		this.bookmarkSeed = value;
	}

	@Override
	public void endHandler() {
		// TODO Auto-generated method stub

	}

	@Override
	public void endChildHandler(IJSONHandler handler) {
		// TODO Auto-generated method stub

	}

	/**
	 * 设置缓存
	 * 
	 * @param cache
	 *            缓存
	 * @deprecated 内部使用
	 */
	public void setCache(ICacheProvider cache) {
		this.cache = cache;
	}

	/**
	 * 取得缓
	 * 
	 * @deprecated 内部使用
	 * @return 缓存
	 */
	public ICacheProvider getCache() {
		return this.cache;
	}

	/**
	 * 关闭，用于关闭缓存
	 */
	public void close() {
		if (this.cache != null) {
			this.cache.close();
		}
	}

	/** 索引对象 */
	private Indexes indexes;
	
	/**
	 * 根据Bookmark，更新索引行
	 * @param bookmark
	 */
	public void updateIndexRowByBookmark(int bookmark) {
		if (indexes != null) {
			indexes.updateBookmark(bookmark);
		}
	}
	
	/**
	 * 根据多个字段值，快速查找第一行
	 * @param filterColumns
	 * @param filterValues
	 * @return
	 * @throws Throwable
	 */
	public int fastFindFirst(String[] filterColumns, Object[] filterValues) throws Throwable {
		int[] fastFilterResult = fastFilter(filterColumns, filterValues);
		return fastFilterResult == null || fastFilterResult.length == 0 ? -1 : fastFilterResult[0];
	}
	
	/**
	 * 根据多个字段值，快速过滤
	 * @param filterColumns
	 * @param filterValues
	 * @return
	 * @throws Throwable
	 */
	public int[] fastFilter(String[] filterColumns, Object[] filterValues) throws Throwable {
		if (indexes == null) {
			indexes = new Indexes(this);
		}
		int[] result = indexes.getFilterRowIndexes(filterColumns, filterValues);
		// 注释掉的是一段检查代码
		//int columnSize = filterColumns.length;
		//int[] filterIndexes = new int[columnSize];
		//DataTypeAction[] dataTypeActions = new DataTypeAction[columnSize];
		//for (int i = 0; i < columnSize; i++) {
		//	int index = metaData.findColumnIndexByKey(filterColumns[i], true);
		//	filterIndexes[i] = index;
		//	dataTypeActions[i] = metaData.getColumnInfo(index).getDataTypeAction();
		//}
		//List<Integer> resultTmp = new ArrayList<Integer>();
		//for (int i = 0, size = viewDataList.size(); i < size; i++) {
		//	Row row = viewDataList.get(i);
		//	Object[] dataList = row.getDataList();
		//	boolean found = true;
		//	for (int j = 0; j < columnSize; j++) {
		//		Object value = dataList[filterIndexes[j]];
		//		if (dataTypeActions[j].compare(filterValues[j], value) != 0) {
		//			found = false;
		//			break;
		//		}
		//	}
		//	if (found) {
		//		resultTmp.add(i);
		//	}
		//}
		//int[] tmp = ArrayUtils.toPrimitive(resultTmp.toArray(new Integer[resultTmp.size()]));
		//if (tmp.length != result.length) {
		//	throw new RuntimeException("程序错误");
		//}
		//for (int i = 0, length = tmp.length; i < length; i++) {
		//	if (tmp[i] != result[i]) {
		//		throw new RuntimeException("程序错误");
		//	}
		//}
		return result;
	}
	
	/**
	 * 根据单个字段值，快速过滤
	 * @param filterColumn
	 * @param filterValue
	 * @return
	 * @throws Throwable
	 */
	public int[] fastFilter(String filterColumn, Object filterValue) throws Throwable {
		String[] filterColumns = new String[]{filterColumn};
		Object[] filterValues = new Object[]{filterValue};
		return fastFilter(filterColumns, filterValues);
	}
	
	/**
	 * 通过索引取不同的值
	 * @param filterColumns
	 * @return
	 * @throws Throwable 
	 */
	public Object[][] fastDistinctValues(String[] filterColumns) throws Throwable {
		if (indexes == null) {
			indexes = new Indexes(this);
		}
		Object[][] result = indexes.getDistinctValues(filterColumns);
		return result;
	}
	
	/**
	 * 快速排序
	 * @param rowIndexes
	 * @param vCriteria
	 * @return
	 * @throws Throwable
	 */
	public int[] fastSort(int[] rowIndexes, SortCriteria[] vCriteria)  throws Throwable {
		int sortLength = vCriteria.length;
		int[] sortIndexes = new int[sortLength];
		DataTypeAction[] dataTypeActions = new DataTypeAction[sortLength];
		for (int i = 0; i < sortLength; i++) {
			int index = metaData.findColumnIndexByKey(vCriteria[i].getColumnKey(), true);
			sortIndexes[i] = index;
			dataTypeActions[i] = metaData.getColumnInfo(index).getDataTypeAction();
		}
		List<Integer> rowIndexArray = Arrays.asList(ArrayUtils.toObject(rowIndexes));
		Collections.sort(rowIndexArray, new Comparator<Integer>() {
			@Override
			public int compare(Integer o1, Integer o2) {
				Object[] dataList1 = viewDataList.get(o1).getDataList();
				Object[] dataList2 = viewDataList.get(o2).getDataList();
				for (int i = 0; i < sortLength; i++) {
					SortCriteria s = vCriteria[i];
					int index = sortIndexes[i];
					int ret = dataTypeActions[i].compare(dataList1[index], dataList2[index]);
					ret = s.isAscending() ? ret : -ret;
					if (ret != 0)
						return ret;
				}
				return 0;
			}
			
		});
		return ArrayUtils.toPrimitive(rowIndexArray.toArray(ArrayUtils.EMPTY_INTEGER_OBJECT_ARRAY));
	}
	
	/**
	 * 批量删除行
	 * @param rowIndexes
	 */
	public void deleteRows(int[] rowIndexes) {
		int deleteRowIndexSize = rowIndexes == null ? 0 :rowIndexes.length;
		if (deleteRowIndexSize == 0) {
			return;
		}
		int viewSize = viewDataList.size();
		if (viewSize == deleteRowIndexSize) {
			this.deleteAll();
			return;
		}
		Arrays.sort(rowIndexes);
		for (int i = deleteRowIndexSize - 1; i >= 0; i--) {
			int rowIndex = rowIndexes[i];
			Row r = viewDataList.get(rowIndex);
			if (r.isNew()) {
				allDataList.remove(r);
				bookmarkMap.remove(r.getBookmark());
				viewDataList.remove(rowIndex);
			} else {
				r.setDeleted();
				if (!showDeleted) {
					viewDataList.remove(rowIndex);
					bookmarkMap.remove(r.getBookmark());
				}
			}
			if (indexes != null) {
				indexes.deleteBookmark(r);
			}
		}
		int minDeleteRowIndex = rowIndexes[0];
		if (pos > minDeleteRowIndex) {
			pos = minDeleteRowIndex;
		}

		int newSize = viewDataList.size();
		for (int i = minDeleteRowIndex; i < newSize; i++) {
			bookmarkMap.put(viewDataList.get(i).getBookmark(), i);
		}
	}
	
	public void setMaintSequence(boolean maintSequence) {
		this.maintSequence = maintSequence;
		this.updateSequenceNumbers(0);
	}

	public boolean isMaintSequence() {
		return maintSequence;
	}

	/** 是否维护sequence */
	private boolean maintSequence = false;
	
	/**
	 * 更新维护viewDataList中的sequence
	 * 
	 * @param rowIndex 行索引,从0开始
	 */
	public void updateSequenceNumbers(int rowIndex) {
		if (!this.maintSequence) {
			return;
		}
		
		Row row = null;
		int seqNum = 0;
		int colIndex = metaData.findColumnIndexByKey(SystemField.SEQUENCE_SYS_KEY, false);
		
		if (colIndex > 0) {
			for (int i = rowIndex - 1; i >= 0; i --) {
				row = this.viewDataList.get(i);
				if (row.getState() != RowState.DELETED) {
					seqNum = TypeConvertor.toInteger(row.getObject(colIndex));
					break;
				}
			}
			
			for (int i = rowIndex; i < this.viewDataList.size(); i ++) {
				row = this.viewDataList.get(i);
				if (row.getState() != RowState.DELETED) {
					row.setObject(colIndex, ++ seqNum);
				}
			}
		}
	}
}
