package com.bokesoft.yigo.bpm;

import com.bokesoft.yes.bpm.meta.transform.BPMOperationType;
import com.bokesoft.yes.bpm.meta.transform.WorkitemInfo;
import com.bokesoft.yes.bpm.service.util.CommitWorkitemUtil;
import com.bokesoft.yigo.bpm.common.BPMContext;
import com.bokesoft.yigo.mid.base.DefaultContext;

/**
 * 提交工作项
 * 
 * @author 刘翔翔
 *
 */
public class CommitWorkitem {

	/** 独立事务提交，默认值为false */
	private boolean isolatedTransaction = false;
	
	/** 传输工作项信息*/
	private WorkitemInfo info = new WorkitemInfo();

	/**
	 * 无参构造器，工作项将不会以独立事务提交（使用当前事务）
	 */
	public CommitWorkitem() {
	}

	/**
	 * 构造器
	 * 
	 * @param isolatedTransaction
	 *            是否 独立事务提交
	 */
	public CommitWorkitem(boolean isolatedTransaction) {
		this.isolatedTransaction = isolatedTransaction;
	}
	/**
	 * 直送操作构造器
	 * @param isolatedTransaction 是否独立事务提交
	 * @param keepParts 是否保持原操作员
	 */
	public CommitWorkitem(boolean isolatedTransaction, boolean keepParts) {
		this.isolatedTransaction = isolatedTransaction;
		info.setKeepParts(keepParts);
		info.setOperationType(BPMOperationType.OPT_TRANSIT);
	}
	/**
	 * 动态驳回操作构造器
	 * @param isolatedTransaction 是否独立事务提交
	 * @param backSite 驳回的目标节点位置
	 * @param backSiteOpt 驳回的目标节点的操作员，仅驳回到会签节点需指定
	 * @param allowTransit 驳回后是否允许直送
	 */
	public CommitWorkitem(boolean isolatedTransaction, String backSite, long backSiteOpt, boolean allowTransit) {
		this.isolatedTransaction = isolatedTransaction;
		info.setBackSite(backSite);
		info.setBackSiteOpt(backSiteOpt);
		info.setAllowTransit(allowTransit);
		info.setOperationType(BPMOperationType.OPT_RETURN);
	}

	/**
	 * 执行提交工作项，在此之前请确认：
	 * <ul>
	 * <li> 若是驳回操作：请使用驳回构造器或者设置驳回类型{@link #setOperationType(int)}为驳回,同时设置必要参数
	 * <li> 若是直送操作：请使用直送构造器或者设置驳回类型为直送{@link #setOperationType(int)}为直送，同时设置必要参数
	 * </ul>
	 * @param context
	 *            上下文
	 * @param workitemID
	 *            工作项编号
	 * @param auditResult
	 *            审批结果
	 * @param userInfo
	 *            审批意见
	 * @throws Throwable
	 */
	public void commitWorkitem(DefaultContext context, Long workitemID, int auditResult, String userInfo) throws Throwable {
		info.setWorkitemID(workitemID);
		info.setAuditResult(auditResult);
		info.setUserInfo(userInfo);
		try {
			BPMContext bpmContext = new BPMContext(context);
			CommitWorkitemUtil.commitWorkitem(bpmContext, info);

			if (isolatedTransaction)
				context.commit();
		} catch (Throwable e) {
			if (isolatedTransaction && context != null) {
				context.rollback();
			}
			throw e;
		} finally {
			if (isolatedTransaction && context != null) {
				context.close();
			}
		}
	}

	/**
	 * 执行提交工作项,在此之前请确认：
	 * <ul>
	 * <li> 若是驳回操作：请使用驳回构造器或者设置驳回类型{@link #setOperationType(int)}为驳回,同时设置必要参数
	 * <li> 若是直送操作：请使用直送构造器或者设置驳回类型为直送{@link #setOperationType(int)}为直送，同时设置必要参数
	 * </ul>
	 * 
	 * @param context
	 *            上下文
	 * @param instanceID
	 *            实例编号
	 * @param workitemID
	 *            工作项编号
	 * @param auditResult
	 *            审批结果
	 * @param userInfo
	 *            审批意见
	 * @throws Throwable
	 */
	public void commitWorkitem(DefaultContext context, Long instanceID, Long workitemID, int auditResult, String userInfo) throws Throwable {
		info.setInstanceID(instanceID);
		info.setWorkitemID(workitemID);
		info.setAuditResult(auditResult);
		info.setUserInfo(userInfo);
		try {
			BPMContext bpmContext = BPMContext.getBPMContext(context, instanceID);
			CommitWorkitemUtil.commitWorkitem(bpmContext, info);
			if (isolatedTransaction)
				context.commit();
		} catch (Throwable e) {
			if (isolatedTransaction && context != null) {
				context.rollback();
			}
			throw e;
		} finally {
			if (isolatedTransaction && context != null) {
				context.close();
			}
		}
	}
	
	/**
	 * 设置驳回位置，大于0表示该提交操作是驳回操作
	 * @param backSite 驳回位置，大于0表示该提交操作是驳回操作
	 */
	public void setBackSite(String backSite) {
		info.setBackSite(backSite);
	}
	/***
	 * 设置驳回位置的操作员
	 * @param backSiteOpt 驳回目标位置的操作员
	 */
	public void setBackSiteOpt(long backSiteOpt) {
		info.setBackSiteOpt(backSiteOpt);
	}
	/**
	 * 设置驳回后是否允许直送，默认值为true
	 * @param allowTransit 驳回后是否允许直送
	 */
	public void setAllowTransit(boolean allowTransit) {
		info.setAllowTransit(allowTransit);
	}
	/**
	 * 设置是否保持原操作员
	 * @param keepParts 是否保持原操作员
	 */
	public void setKeepParts(boolean keepParts) {
		info.setKeepParts(keepParts);
	}
	/**
	 * 设置操作类型，目前支持3种类型：
	 * <ul>
	 * <li> 0: 提交操作
	 * <li> 1: 驳回操作
	 * <li> 2: 直送操作
	 * </ul>
	 * @param operationType 操作类型
	 */
	public void setOperationType(int operationType) {
		switch (operationType) {
		case 1:		
			info.setOperationType(BPMOperationType.OPT_RETURN);
			break;
		case 2:
			info.setOperationType(BPMOperationType.OPT_TRANSIT);
			break;

		default:
			info.setOperationType(BPMOperationType.OPT_COMMIT);
			break;
		}
	}

}
