package com.bokesoft.yes.design.utils;

import com.bokesoft.erp.mid.schema.ERPSchemaCheck;
import com.bokesoft.erp.mid.schema.ERPSchemaMaintance;
import com.bokesoft.erp.mid.schema.ERPSchemaProcess;
import com.bokesoft.erp.mid.schema.ERPSchemaViewDependSchemaTable;
import com.bokesoft.erp.mid.util.EnsureSingleDBStruct;
import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yes.design.constant.ConstantUtil;
import com.bokesoft.yes.design.io.DesignIOMetaUtil;
import com.bokesoft.yes.mid.connection.dbmanager.mysqls.MultiDBManager;
import com.bokesoft.yes.mid.parameterizedsql.SqlString;
import com.bokesoft.yes.mid.schemamgr.SchemaProProcess;
import com.bokesoft.yigo.common.def.DataObjectSecondaryType;
import com.bokesoft.yigo.common.def.MigrationUpdateStrategyType;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.dataobject.MetaDataSource;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.meta.dataobject.MetaTableCollection;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.form.MetaFormList;
import com.bokesoft.yigo.meta.form.MetaFormProfile;
import com.bokesoft.yigo.meta.schema.MetaIndex;
import com.bokesoft.yigo.meta.schema.MetaIndexCollection;
import com.bokesoft.yigo.meta.schema.MetaSchemaColumn;
import com.bokesoft.yigo.meta.schema.MetaSchemaTable;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.mid.connection.DBType;
import com.bokesoft.yigo.mid.connection.DataBaseInfo;
import com.bokesoft.yigo.mid.connection.IDBManager;
import com.bokesoft.yigo.mid.schemamgr.ISchemaCheck;
import com.bokesoft.yigo.mid.schemamgr.ISchemaCreate;
import com.bokesoft.yigo.mid.schemamgr.SchemaCreateFactory;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 根据配置文件重新建表，采用异步实现，专为设计器使用
 */
public class RebuildTableUtil {

    private static final Map<String, Set<String>> dataObjectViewCacheMap = new HashMap<>();

    public static void rebuildTable(DefaultContext defaultContext, String filePath) throws Throwable {
        String content = FileUtils.readFileToString(new File(filePath), "UTF-8");
        String newSecondLine = DesignIOMetaUtil.getSecondLine(content);
        if (newSecondLine.contains("<CommonDef>")) {
            return;
        }
        String formKey = DesignIOMetaUtil.getSubString(newSecondLine, "Key=\"([A-Za-z_]\\w*)\"");
        SAXReader reader = new SAXReader();
        // 加载文档
        Document document = reader.read(new File(filePath));
        // 获取的根节点是临时文件的根节点，再未保存前就修改临时文件
        Element root = document.getRootElement();
        String name = root.getQName().getName();
        //确保不是界面马甲
        IMetaFactory metaFactory = MetaFactory.getGlobalInstance();
        MetaFormList metaFormList = metaFactory.getMetaFormList();
        if (StringUtils.equals(name, ConstantUtil.DATA_OBJECT)) {
            for (MetaFormProfile profile : metaFormList) {
                MetaForm form = profile.getForm();
                if (Objects.nonNull(form)) {
                    MetaDataSource dataSource = form.getDataSource();
                    if (Objects.nonNull(dataSource)) {
                        MetaDataObject dataObject = dataSource.getDataObject();
                        if (Objects.nonNull(dataObject)) {
                            String key = dataObject.getKey();
                            String extend = dataObject.getExtend();
                            String mergeToSourceMapKey = dataObject.getMergeToSourceMapKey();
                            if (StringUtils.equalsAny(formKey, key, extend, mergeToSourceMapKey)) {
                                return;
                            }
                        }
                    }
                }
            }
            MetaDataObject dataObject = metaFactory.getDataObject(formKey);
            if (Objects.nonNull(dataObject)) {
                rebuildTable4VestDiff(defaultContext, dataObject);
            }
        } else {
            MetaFormProfile metaFormProfile = metaFormList.get(formKey);
            if(metaFormProfile == null) metaFormProfile = metaFactory.getExtFormList().get(formKey);
            if(metaFormProfile != null) {
                MetaForm form = metaFormProfile.getForm();
                if (metaFormProfile.getMergeToSource()) {
                    rebuildTable4VestDiff(defaultContext, form);
                } else {
                    if (!newSecondLine.contains(ConstantUtil.EXTEND)) {
                        if (form != null) {
                            rebuildTable4VestDiff(defaultContext, form);
                        }
                    }
                }
            }
        }
    }

    public static void rebuildTable4VestDiff(DefaultContext context, MetaForm metaForm) throws Throwable {
        MetaDataSource dataSource = metaForm.getDataSource();
        if (Objects.isNull(dataSource)) {
            return;
        }
        MetaDataObject dataObject = dataSource.getDataObject();
        if (Objects.isNull(dataObject)) {
            return;
        }
        rebuildTable4VestDiff(context, dataObject);
    }

    public static void rebuildTable4VestDiff(DefaultContext context, MetaDataObject dataObject) throws Throwable {
        LogSvr.getInstance().info("rebuildTable4VestDiff：" + dataObject.getKey() + "开始");
        DefaultContext newContext = null;
        try {
            newContext = new DefaultContext(context.getVE());

            final MetaTableCollection tableCollection = dataObject.getTableCollection();
            Map<String, MetaTable> metaTableMap = new HashMap<>();
            for (MetaTable metaTable : tableCollection) {
                metaTableMap.put(metaTable.getKey(), metaTable);
            }

            ArrayList<MetaSchemaTable> schemaTableList = dataObject.getSchemeTableList();
            if (schemaTableList != null) {
                IDBManager dbManager = newContext.getDBManager();
                final int secondaryType = dataObject.getSecondaryType();
                ERPSchemaProcess schemaProcess = RebuildTableUtil.getSchemaProcess(dbManager);
                DataBaseInfo info = schemaProcess.getInfo();
                for (MetaSchemaTable table : schemaTableList) {
                    SchemaProProcess.dataObjectProProcess(dbManager, table);
                    try {
                        tableRebuild(dbManager, table, schemaProcess, info, metaTableMap);
                    } catch (Throwable e) {
                        LogSvr.getInstance().error("tableRebuild：" + dataObject.getKey() + "出错", e);
                    }
                }
                if (secondaryType == DataObjectSecondaryType.MIGRATION && dataObject.getMigrationUpdateStrategy() == MigrationUpdateStrategyType.KEYS_TABLE_FOCUS_CHANGE
                        && Objects.nonNull(dataObject.getErpMigrationKeysTable())) {
                    MetaTable mainMetaTable = dataObject.getMainTable();
                    // 生成联合索引，如果有期间字段，那么就是联合索引就是(group_id,期间字段),否则只有分组id作为索引字段
                    try {
                        ERPSchemaMaintance.generateUnionIndex(dbManager, dataObject, schemaProcess);
                    } catch (Throwable e) {
                        LogSvr.getInstance().error("generateUnionIndex：" + dataObject.getKey() + "出错", e);
                    }
                    final Set<String> dataObjectViewCacheSet = dataObjectViewCacheMap.getOrDefault(dataObject.getKey(), Collections.emptySet());
                    ERPSchemaViewDependSchemaTable views = new ERPSchemaViewDependSchemaTable(dataObject);
                    List<String> viewNames = views.getViewNames();
                    try {
                        final Set<String> newDataObjectViewCacheSet = rebuildView(dbManager, mainMetaTable, views, viewNames, info, dataObjectViewCacheSet);
                        dataObjectViewCacheMap.put(dataObject.getKey(), newDataObjectViewCacheSet);
                    } catch (Throwable e) {
                        LogSvr.getInstance().error("rebuildView：" + dataObject.getKey() + "出错", e);
                    }
                }
                
                if(dbManager instanceof MultiDBManager) {
                	EnsureSingleDBStruct.buildTablesDataObject(dataObject, (MultiDBManager) dbManager);
                }
                newContext.commit();
                LogSvr.getInstance().info("rebuildTable4VestDiff：" + dataObject.getKey() + "结束");
            }
        } finally {
            if (newContext != null) {
                try {
                    newContext.close();
                } catch (Throwable e) {
                    LogSvr.getInstance().error("RebuildTable出错", e);
                }
            }
        }

    }

    private static List<MetaIndex> getiMetaIndexList(MetaSchemaTable table, ERPSchemaProcess schemaProcess) {
        List<MetaIndex> indexList = new ArrayList<>();
        Iterator<MetaIndex> iterator = table.getIndexCollection().iterator();
        while (iterator.hasNext()) {
            MetaIndex metaIndex = iterator.next();
            indexList.add(metaIndex);
//			schemaProcess.updateIndexColumns(table.getKey(), metaIndex);
        }
        return indexList;
    }

    public static boolean tableRebuild(IDBManager dbManager, MetaSchemaTable table, ERPSchemaProcess schemaProcess,
                                       DataBaseInfo info, Map<String, MetaTable> metaTableMap) throws Throwable {
    	if (dbManager instanceof MultiDBManager) {
    		((MultiDBManager) dbManager).setIgnoreXA(true);
    	}
        boolean result = false;
        // 如果是视图，就返回；否则进行表操作
        if (info.checkViewExist(table.getKey())) {
            return result;
        }
        ISchemaCreate schemaCreate = SchemaCreateFactory.getInstance().create();
        if (!info.checkTableExist(table.getKey())) {
            schemaCreate.createTable(dbManager, table);
            // dbManager.execPrepareUpdate(createTableSQL, new Object[0]);
            if (dbManager.getDBType() == DBType.EsgynDB) {
                dbManager.commit();
            }
            result = true;
            MetaIndexCollection indexCollection = table.getIndexCollection();
            final MetaIndex metaIndex = acceptTableIndexs(table, metaTableMap, indexCollection);
            if (Objects.nonNull(metaIndex)) {
                indexCollection.add(metaIndex);
            }
            if (indexCollection != null) {
                Iterator<MetaIndex> it = indexCollection.iterator();
                while (it.hasNext()) {
                    MetaIndex index = it.next();
                    schemaProcess.createIndex(dbManager, schemaCreate, table, index);
                    schemaProcess.updateIndexColumns(table.getKey(), index);
                }
            }
        } else {
            LogSvr.getInstance().info("rebuildTable4VestDiff：更新表" + table.getKey());
            ISchemaCheck schemaCheck = new ERPSchemaCheck(dbManager, table, info);
            List<MetaSchemaColumn> columns = schemaCheck.checkTable();
            if (columns.size() != 0) {
                String alterTableStr = dbManager.getAlterTableStr(table, columns);
                dbManager.execPrepareUpdate(alterTableStr);
                if (dbManager.getDBType() == DBType.EsgynDB) {
                    dbManager.commit();
                }
                result = true;
            }
            try {
                List<MetaIndex> indexList = getiMetaIndexList(table, schemaProcess);
                final MetaIndex metaIndex = acceptTableIndexs(table, metaTableMap, indexList);
                if (Objects.nonNull(metaIndex)) {
                    indexList.add(metaIndex);
                }
                //检查并删除需要删除的索引
                checkAndUpdateDeleteIndex(dbManager, table, schemaProcess, indexList, metaTableMap);
                //这里尝试去重建索引
                schemaProcess.checkUpdateIndex(dbManager, table, indexList, schemaCreate);
            } catch (Exception e) {
                LogSvr.getInstance().error("checkUpdateIndex：" + table.getKey() + "出错", e);
                throw e;
                //这里尝试去重建索引报错原因就是索引名称重复导致，可以忽略，就算不创建成功也不会影响建表，下次服务重启会自动重新创建正确的索引
            }

        }
        if (info != null) {
            info.updateInfo(table);
        }
        return result;
    }

    private static synchronized ERPSchemaProcess getSchemaProcess(IDBManager dbManager) throws Throwable {
        ERPSchemaProcess schemaProcess = ERPSchemaMaintance.getSchemaProcess();

        if (schemaProcess == null) {
            schemaProcess = new ERPSchemaProcess(dbManager);
            ERPSchemaMaintance.setSchemaProcess(schemaProcess);
        }
        return schemaProcess;

    }

    private static Set<String> rebuildView(IDBManager dbManager, MetaTable mainMetaTable, ERPSchemaViewDependSchemaTable views,
                                           List<String> viewNames, DataBaseInfo info, Set<String> dataObjectViewCacheSet) throws Throwable {
        // 删除视图
        for (String viewName : viewNames) {
            if (info.checkViewExist(viewName) || dataObjectViewCacheSet.contains(viewName)) {
                String sql = new SqlString().append("drop view ").append(viewName).getSql();
                if (dbManager.getDBType() == DBType.EsgynDB) {
                    sql = new SqlString().append("drop view ").append(viewName).append(" CASCADE").getSql();
                }
                dbManager.execPrepareUpdate(sql);
                if (dbManager.getDBType() == DBType.EsgynDB) {
                    dbManager.commit();
                }
            }
        }
        // 如果该视图的名称已经被表使用，那么先重命名表名，然后再创建视图
        changeTableName(dbManager, mainMetaTable.getBindingDBTableName());
        // 获取生成视图列表
        LinkedHashMap<String, String> createViewSqls = (LinkedHashMap<String, String>) views.generateViewSqls(dbManager);
        if (!createViewSqls.isEmpty()) {
            for (Map.Entry<String, String> entry : createViewSqls.entrySet()) {
                String viewSql = entry.getValue();
                dbManager.execPrepareUpdate(viewSql);
                if (dbManager.getDBType() == DBType.EsgynDB) {
                    dbManager.commit();
                }
            }
        }
        return createViewSqls.keySet();
    }

    private static void changeTableName(IDBManager dbManager, String tableName) throws Throwable {
        if (dbManager.checkTableExist(tableName)) {
            String oldTableName = tableName + "_old";
            // 如果_old后缀的表已经存在，那么重命名表为_old+时间戳
            if (dbManager.checkTableExist(oldTableName)) {
                renameTable(dbManager, oldTableName, oldTableName + System.currentTimeMillis());
            }
            renameTable(dbManager, tableName, oldTableName);
        }
    }

    /***
     * 重命名表
     * @param dbManager
     * @param sourceTable
     * @param targetTable
     * @return true, false
     */
    private static void renameTable(IDBManager dbManager, String sourceTable, String targetTable) throws Throwable {
        SqlString sql = new SqlString();
        switch (dbManager.getDBType()) {
            case DBType.MySql:
                sql.append("rename table ", sourceTable, " to ", targetTable);
                break;
            case DBType.Oracle:
            case DBType.DM:
            case DBType.EsgynDB:
                sql.append("alter table ", sourceTable, " rename to ", targetTable);
                break;
            case DBType.SqlServer:
                sql.append("exec sp_rename ", sourceTable, ",", targetTable);
                break;
            default:
                break;
        }
        dbManager.execPrepareUpdate(sql.getSql());
        if (dbManager.getDBType() == DBType.EsgynDB) {
            dbManager.commit();
        }
    }

    private static MetaIndex acceptTableIndexs(MetaSchemaTable table, Map<String, MetaTable> metaTableMap, Iterable<MetaIndex> indexList) {
        final MetaTable metaTable = metaTableMap.get(table.getKey());
        if (Objects.isNull(metaTable)) {
            return null;
        }
        final String soidIndexKey = metaTable.getIndexPrefix4Create() + "_SOID";
        boolean existFlag = false;
        for (MetaIndex metaIndex : indexList) {
            if (StringUtils.equalsIgnoreCase(metaIndex.getKey(), soidIndexKey)) {
                existFlag = true;
                metaIndex.setUnique(false);
            }
        }
        if (!existFlag && table.getColumnCollection().containsKey("SOID")) {
            MetaIndex metaIndex = new MetaIndex();
            metaIndex.setKey(soidIndexKey);
            metaIndex.setUnique(false);
            metaIndex.setColumns("SOID");
            return metaIndex;
        }
        return null;
    }

    private static void checkAndUpdateDeleteIndex(IDBManager dbManager, MetaSchemaTable table, ERPSchemaProcess schemaProcess, List<MetaIndex> indexList, Map<String, MetaTable> metaTableMap) {
        String tableKey = table.getKey();
        // 考虑到存在表单xml没有index配置，这种情况不维护
        MetaTable metaTable = metaTableMap.get(tableKey);
        if (metaTable == null || Objects.isNull(metaTable.getIndexCollection()) || metaTable.getIndexCollection().isEmpty()){
            return;
        }
        List<String> indexKeyList = indexList.stream().map(m -> m.getKey().toLowerCase()).collect(Collectors.toList());
        // 获取保存前所有索引
        HashMap<String, String> indexColumnSet = schemaProcess.getInfo().getIndexColumnSet(tableKey);
        // 有差异的索引
        HashMap<String, String> diff_indexColumnSet = new HashMap<>();
        indexColumnSet.forEach((key,value)->{
            if (!indexKeyList.contains(key.toLowerCase()) && !"PRIMARY".equalsIgnoreCase(key)){
                diff_indexColumnSet.put(key, value);
            }
        });
        if (diff_indexColumnSet.size() == 0){
            return;
        }
        try {
            MetaIndex metaIndex = new MetaIndex();
            for (Map.Entry<String, String> entry : diff_indexColumnSet.entrySet()) {
                String indexKey = entry.getKey();
                String columns = entry.getValue();
                metaIndex.setKey(indexKey);
                metaIndex.setColumns(columns);
                // 查询缓存
                Field indexTableMapField = ERPSchemaProcess.class.getDeclaredField("indexTableMap");
                Field columnsIndexMapField = ERPSchemaProcess.class.getDeclaredField("columnsIndexMap");
                String indexName = indexKey.toUpperCase();
                String key = tableKey.toUpperCase() + "." + columns.toUpperCase();
                indexTableMapField.setAccessible(true);
                columnsIndexMapField.setAccessible(true);
                Map<String, String> indexTableMap = (Map<String, String>) indexTableMapField.get(schemaProcess);
                Map<String, String> columnsIndexMap = (Map<String, String>) columnsIndexMapField.get(schemaProcess);
                if (Objects.isNull(indexTableMap.get(indexName)) && Objects.isNull(columnsIndexMap.get(key))){
                    // 缓存中不存在则表示已经删除
                    continue;
                }
                // 删除索引
                Method privateMethod = ERPSchemaProcess.class.getDeclaredMethod("dropExistsIndex", IDBManager.class, String.class, MetaIndex.class);
                privateMethod.setAccessible(true);
                privateMethod.invoke(schemaProcess, dbManager, tableKey, metaIndex);
                // 更新缓存
                indexTableMap.remove(indexName);
                columnsIndexMap.remove(key);
                indexTableMapField.set(schemaProcess, indexTableMap);
                columnsIndexMapField.set(schemaProcess, columnsIndexMap);
            }
        }catch (Exception e){
            LogSvr.getInstance().error("checkAndUpdateDeleteIndex：" + table.getKey() + "出错", e);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }

    }
}
