package com.bokesoft.yes.design.search;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Pattern;

import com.bokesoft.yes.design.search.text.FileMatch;
import com.bokesoft.yes.design.search.text.FileMatchResource;
import com.bokesoft.yes.design.search.text.LineElement;
import com.bokesoft.yes.design.search.text.Match;
import com.bokesoft.yes.design.search.text.TextSearchMatchAccess;

/**
 * 文件搜索
 */
public class FileSearchQuery {

	/** 输入的搜索文本 */
	private final String fSearchText;

	/** 正则 */
	private final boolean fIsRegEx;

	/** 区分大小写 */
	private final boolean fIsCaseSensitive;

	/** 全词匹配 */
	private final boolean fIsWholeWord;
	
	/** 文件类型 */
	private final String fFileTypes;

	/** 文件范围 */
	private final FileTextSearchScope fScope;

	/** 搜索结果 */
	private FileSearchResult fResult;

	public FileSearchQuery(String searchText, boolean isRegEx, boolean isCaseSensitive, FileTextSearchScope scope) {
		this(searchText, isRegEx, isCaseSensitive, false, "",scope);
	}

	public FileSearchQuery(String searchText, boolean isRegEx, boolean isCaseSensitive, boolean isWholeWord, String fileTypes,
			FileTextSearchScope scope) {
		fSearchText = searchText;
		fIsRegEx = isRegEx;
		fIsCaseSensitive = isCaseSensitive;
		fIsWholeWord = isWholeWord;
		fFileTypes = fileTypes;
		fScope = scope;
	}

	/**
	 * 获取文件搜索范围
	 * 
	 * @return
	 */
	public FileTextSearchScope getSearchScope() {
		return fScope;
	}

	/**
	 * 获取搜索字符串
	 * 
	 * @return
	 */
	public String getSearchString() {
		return fSearchText;
	}

	/**
	 * 是否正则
	 * 
	 * @return
	 */
	public boolean isRegexSearch() {
		return fIsRegEx;
	}

	/**
	 * 是否区分大小写
	 * 
	 * @return
	 */
	public boolean isCaseSensitive() {
		return fIsCaseSensitive;
	}

	/**
	 * 是否整词匹配
	 * 
	 * @return
	 */
	public boolean isWholeWord() {
		return fIsWholeWord;
	}

	/**
	 * 文件名搜索
	 * 
	 * @return
	 */
	public boolean isFileNameSearch() {
		return fSearchText.length() == 0;
	}

	public void run() {
		FileSearchResult textResult = getSearchResult();
		textResult.removeAll();

		Pattern searchPattern = getSearchPattern();
		TextSearchResultCollector collector = new TextSearchResultCollector(textResult, isFileNameSearch(), fFileTypes);

		TextSearchEngine.createDefault().search(collector, searchPattern);
	}

	/**
	 * 获取正则
	 * 
	 * @return
	 */
	protected Pattern getSearchPattern() {
		return PatternConstructor.createPattern(fSearchText, fIsRegEx, fIsCaseSensitive, fIsWholeWord);
	}

	/**
	 * 获取搜索结果
	 * 
	 * @return
	 */
	public FileSearchResult getSearchResult() {
		if (fResult == null) {
			fResult = new FileSearchResult();
		}
		return fResult;
	}

	private final static class TextSearchResultCollector extends TextSearchRequestor {
		private final FileSearchResult fResult;
		private final boolean fIsFileSearchOnly;
		private final String fileTypes;
		/** 剩余匹配数 */
		private int remainingResultQuota;

		private Map<FileMatchResource, ArrayList<FileMatch>> fCachedMatches;
		private Object fLock = new Object();

		private TextSearchResultCollector(FileSearchResult result, boolean isFileSearchOnly,String fTypes) {
			fResult = result;
			fIsFileSearchOnly = isFileSearchOnly;
			fileTypes = fTypes;
		}

		@Override
		public void beginReporting() {
			fCachedMatches = new HashMap<>();
			remainingResultQuota = 2000;
		}

		@Override
		public void endReporting() {
			flushMatches();
			synchronized (fLock) {
				fCachedMatches = null;
			}
		}

		@Override
		public boolean acceptFile(File file, File sfile) {
			if (!file.exists()) {
				return false;
			}

			if (!file.getName().endsWith(".xml")) {
				return false;
			}

			// 文件名搜索
//			if (fIsFileSearchOnly) {
//				synchronized (fLock) {
//					fResult.addMatch(new FileMatch(new FileMatchResource(sfile, file)));
//				}
//			}

			flushMatches();
			return true;
		}
		
		@Override
		public boolean acceptFileType(String type) {
			if (fileTypes == null || fileTypes.length() == 0) {
				return true;
			}
			
			if (fileTypes.contains(type)) {
				return true;
			}

			return false;
		}

		@Override
		public boolean acceptPatternMatch(TextSearchMatchAccess matchRequestor) {
			ArrayList<FileMatch> matches;

			synchronized (fLock) {
				if (remainingResultQuota < 0) {
					return false;
				}
				
				if (fCachedMatches == null) {
					return false;
				}
				matches = fCachedMatches.get(matchRequestor.getFile());
			}

			int matchOffset = matchRequestor.getMatchOffset();

			LineElement lineElement = getLineElement(matchOffset, matchRequestor, matches);
			if (lineElement != null) {
				FileMatch fileMatch = new FileMatch(matchRequestor.getFile(), matchOffset,
						matchRequestor.getMatchLength(), lineElement);
				synchronized (fLock) {
					if (fCachedMatches == null) {
						return false;
					}
					matches = fCachedMatches.get(matchRequestor.getFile());
					if (matches == null) {
						matches = new ArrayList<>();
						fCachedMatches.put(matchRequestor.getFile(), matches);
					}
					matches.add(fileMatch);
					
					remainingResultQuota --;
				}
			}
			
			return true;
		}

		private LineElement getLineElement(int offset, TextSearchMatchAccess matchRequestor,
				ArrayList<FileMatch> matches) {
			int lineNumber = 1;
			int lineStart = 0;

			if (matches != null) {
				FileMatch last = matches.get(matches.size() - 1);
				LineElement lineElement = last.getLineElement();
				if (lineElement.contains(offset)) {
					return lineElement;
				}

				lineStart = lineElement.getOffset() + lineElement.getLength();
				lineNumber = lineElement.getLine() + 1;
			}

			int i = lineStart;
			int contentLength = matchRequestor.getFileContentLength();
			while (i < contentLength) {
				char ch = matchRequestor.getFileContentChar(i++);
				if (ch == '\n' || ch == '\r') {
					if (ch == '\r' && i < contentLength && matchRequestor.getFileContentChar(i) == '\n') {
						i++;
					}
					if (offset < i) {
						String lineContent = getContents(matchRequestor, lineStart, i);
						return new LineElement(matchRequestor.getFile().getFile(), lineNumber, lineStart, lineContent);
					}
					lineNumber++;
					lineStart = i;
				}
			}
			if (offset < i) {
				String lineContent = getContents(matchRequestor, lineStart, i);
				return new LineElement(matchRequestor.getFile().getFile(), lineNumber, lineStart, lineContent);
			}

			return null;
		}

		private static String getContents(TextSearchMatchAccess matchRequestor, int start, int end) {
			StringBuilder buf = new StringBuilder();
			for (int i = start; i < end; i++) {
				char ch = matchRequestor.getFileContentChar(i);
				if (Character.isWhitespace(ch) || Character.isISOControl(ch)) {
					buf.append(' ');
				} else {
					buf.append(ch);
				}
			}
			return buf.toString();
		}

		/**
		 * 刷新当前文件缓存
		 */
		private void flushMatches() {
			synchronized (fLock) {
				if (fCachedMatches != null && !fCachedMatches.isEmpty()) {
					Iterator<ArrayList<FileMatch>> it = fCachedMatches.values().iterator();
					while (it.hasNext()) {
						ArrayList<FileMatch> matches = it.next();
						fResult.addMatches(matches.toArray(new Match[matches.size()]));
					}
					fCachedMatches.clear();
				}
			}
		}
	}
}
