package com.bokesoft.yes.design.context;

import com.bokesoft.yes.common.struct.StringHashMap;
import com.bokesoft.yes.design.DiffProperty;
import com.bokesoft.yes.erp.dev.MetaTableCache;
import com.bokesoft.yes.erp.scope.MetaFormAllFormulScopeCache;
import com.bokesoft.yes.erp.scope.MetaFormAllFormulaScope;
import com.bokesoft.yes.erp.scope.ScopeTree;
import com.bokesoft.yes.erp.scope.ScopeTreeBuilder;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.form.MetaFormList;
import com.bokesoft.yigo.meta.form.MetaFormProfile;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.BeanUtils;

import java.util.*;

public final class DesignerContext {
    private static final ThreadLocal<DesignerContext> THREAD_LOCAL = ThreadLocal.withInitial(DesignerContext::new);
    private final List<DiffProperty> propertyList = new ArrayList<>();
    private final Map<String, MetaForm> formMap = new StringHashMap<>();
    private final Set<String> changedDataObjectSet = new HashSet<>();
    private final Map<String, MetaFormAllFormulaScope> formulaCache = new StringHashMap<>();
    private final Map<String, ScopeTree> scopeCache = new StringHashMap<>();
    private final Pair<Map, Set<String>> tableCache = Pair.of(new HashMap<>(), new HashSet<>());

    public static void addProperty(DiffProperty property) {
        if (Objects.isNull(property) || Objects.isNull(property.property)) {
            return;
        }
        DesignerContext context = THREAD_LOCAL.get();
        context.propertyList.add(property);
    }

    public static void addForm(MetaFormProfile formProfile) {
        MetaForm form = formProfile.getForm();
        if (Objects.isNull(form)) {
            return;
        }
        String formKey = formProfile.getKey();
        DesignerContext context = THREAD_LOCAL.get();
        context.formMap.putIfAbsent(formKey, form);

        MetaFormAllFormulaScope formula = MetaFormAllFormulScopeCache.instance.getCache(formKey);
        if (Objects.nonNull(formula)) {
            context.formulaCache.putIfAbsent(formKey, formula);
        }

        ScopeTree scope = ScopeTreeBuilder.getCache(form);
        if (Objects.nonNull(scope)) {
            context.scopeCache.putIfAbsent(formKey, scope);
        }
    }

    public static boolean isChangedDataObject(MetaDataObject metaDataObject) {
        if (Objects.isNull(metaDataObject)) {
            return false;
        }
        String dataObjectKey = metaDataObject.getKey();
        DesignerContext context = THREAD_LOCAL.get();
        boolean isChanged = context.changedDataObjectSet.contains(dataObjectKey);
        if (isChanged) {
            return true;
        }
        context.changedDataObjectSet.add(dataObjectKey);
        return false;
    }

    public static void backupTableCache() {
        DesignerContext context = THREAD_LOCAL.get();
        if (context.tableCache.getLeft().isEmpty()) {
            context.tableCache.getLeft().putAll(MetaTableCache.getMetaTables());
            context.tableCache.getRight().addAll(MetaTableCache.getProcessDataObject());
        }
    }

    public static void revertTableCache() {
        DesignerContext context = THREAD_LOCAL.get();
        boolean changeTable = context.propertyList.isEmpty() || context.propertyList.stream().anyMatch(item -> item.property.changeTable);
        if (changeTable) {
            return;
        }
        if (MetaTableCache.getMetaTables().isEmpty()) {
            MetaTableCache.getMetaTables().putAll(context.tableCache.getLeft());
            MetaTableCache.getProcessDataObject().addAll(context.tableCache.getRight());
        }
    }

    public static void removeContext() throws Throwable {
        try {
            DesignerContext context = THREAD_LOCAL.get();
            boolean changeScope = context.propertyList.isEmpty() || context.propertyList.stream().anyMatch(item -> item.property.changeScope);
            if (changeScope) {
                return;
            }
            IMetaFactory metaFactory = MetaFactory.getGlobalInstance();
            MetaFormList metaFormList = metaFactory.getMetaFormList();
            for (Map.Entry<String, MetaForm> entry : context.formMap.entrySet()) {
                String formKey = entry.getKey();
                if (!metaFormList.containsKey(formKey)) {
                    continue;
                }
                MetaForm form = entry.getValue();
                MetaForm newForm = metaFactory.getMetaForm(formKey);
                BeanUtils.copyProperties(newForm, form);
                MetaFormProfile profile = metaFormList.get(formKey);
                profile.setForm(form);

                MetaFormAllFormulaScope formula = context.formulaCache.get(formKey);
                if (Objects.nonNull(formula)) {
                    MetaFormAllFormulScopeCache.instance.putCache(formKey, formula);
                }

                ScopeTree scopeTree = context.scopeCache.get(formKey);
                if (Objects.nonNull(scopeTree)) {
                    ScopeTreeBuilder.putCache(form, scopeTree);
                }
            }
        } finally {
            THREAD_LOCAL.remove();
        }
    }

}
