package com.bokesoft.distro.tech.yigosupport.extension.utils.yigo;

import java.util.Date;
import java.util.Map;
import java.util.UUID;

import com.bokesoft.yigo.common.def.OperationState;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.json.JSONObject;

import com.bokesoft.distro.tech.commons.basis.MiscUtil;
import com.bokesoft.yigo.common.def.DictStateMask;
import com.bokesoft.yigo.mid.auth.Login;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.mid.base.LoginInfo;
import com.bokesoft.yigo.mid.session.ISessionInfo;
import com.bokesoft.yigo.mid.session.ISessionInfoMap;
import com.bokesoft.yigo.mid.session.ISessionInfoProvider;
import com.bokesoft.yigo.mid.session.SessionInfoProviderHolder;
import com.bokesoft.yigo.mid.util.ContextBuilder;
import com.bokesoft.yigo.struct.dict.Item;
import com.bokesoft.yigo.struct.env.Env;
import com.bokesoft.yigo.tools.dict.IDictCache;

/**
 * 和 Yigo 登录/会话 相关的工具类
 */
public class SessionUtils {
    /**
     * Yigo 2.0 的登录模式, 1 代表 PC 模式登录
     */
    public static final int LOGIN_MODE_PC = 1;
    /**
     * Yigo 2.0的登录模式, 2 代表MOBILE 模式登录
     */
    public static final int LOGIN_MODE_MOBILE = 2;

    /**
     * 登录时 {@link Env#setTime(String)} 时使用的时间字符串格式.
     */
    private static final String ENV_TIME_PATTERN = "yyyy/MM/dd HH:mm:ss";

    /**
     * Yigo 登录服务
     * @param prefix 系统前缀, 比如移动应用可以以 “APP” 作为前缀
     * @param userCode 用户编码(登录名)
     * @param password 登录密码
     * @param mode 登录模式, 见 {@link #LOGIN_MODE_PC} 和 {@link #LOGIN_MODE_MOBILE}
     * @return
     */
    public static String login(final String prefix, final String userCode, final String password, final int mode) {

        return SessionUtils.processWithContext(null, ctx -> {

            Env env = ctx.getEnv();
            String clientId = prefix + "-" + UUID.randomUUID().toString();
            env.setClientID(clientId);

            String pass = (null == password) ? "" : password;
            LoginInfo loginInfo = new LoginInfo(userCode, pass, -1, mode, "", null);
            Login login = new Login(loginInfo);
            try {
                login.doLogin(ctx);
            } catch (Throwable ex) {
                throw MiscUtil.toRuntimeException(ex);
            }

            return clientId;
        });
    }

    /**
     * Yigo 后台登录 - 不需要密码
     * @param userCode 用户编码(登录名)
     * @return
     */
    public static String loginBackend(String userCode) {
        return loginBackend(userCode, LOGIN_MODE_PC);
    }

    /**
     * Yigo 后台登录 - 不需要密码
     * @param userCode 用户编码(登录名)
     * @param loginMode 登录模式, 见 {@link #LOGIN_MODE_PC} 和 {@link #LOGIN_MODE_MOBILE}
     * @return
     */
    public static String loginBackend(String userCode, int loginMode) {
        DefaultContext ctx = null;
        try {
            ctx = ContextBuilder.create();
            String clientId = loginBackend(ctx, userCode, loginMode, null);
            ctx.commit();
            return clientId;
        } catch (Throwable ex) {
            try {
                ctx.rollback();
            } catch (Throwable e) {
                //Ignore it
            }
            throw MiscUtil.toRuntimeException(ex);
        } finally {
            try {
                ctx.close();
            } catch (Throwable e) {
                //Ignore it
            }
        }
    }

    /**
     * 对一个 Yigo 的 Context 执行无密码登录操作, 从而得到一个具有合理身份的 Yigo 上下文
     * @param ctx Yigo的Context
     * @param userCode 用户账号
     * @param loginMode 登录模式, 见 {@link #LOGIN_MODE_PC} 和 {@link #LOGIN_MODE_MOBILE}
     * @return 返回登录会话的 client ID
     */
    public static String loginBackend(DefaultContext ctx, String userCode, int loginMode, Map<String, Object> params) {
        return loginBackend(ctx, userCode, null, loginMode, params);
    }

    /**
     * 对一个 Yigo 的 Context 执行无密码登录操作, 从而得到一个具有合理身份的 Yigo 上下文
     * @param ctx  Yigo 的 Context
     * @param userCode 用户账号
     * @param clientID 锚定clientID确保返回值如同预期,可以为空,
     * @param loginMode 登录模式, 见 {@link #LOGIN_MODE_PC} 和 {@link #LOGIN_MODE_MOBILE}
     * @return 返回登录会话的 clientID
     */
    public static String loginBackend(DefaultContext ctx, String userCode, String clientID, int loginMode, Map<String, Object> params) {
        Env env = ctx.getEnv();
        if (StringUtils.isEmpty(clientID)) {
            clientID = UUID.randomUUID().toString();
        }
        env.setClientID(clientID);
        LoginInfo loginInfo = new LoginInfo(userCode, "", -1, loginMode, "", params);
        SSOLogin ssoLogin = new SSOLogin(loginInfo);
        try {
            JSONObject loginResult = ssoLogin.doLogin(ctx);
            long userID = loginResult.getLong("UserID");
            wrapEnvSetUserID(env, userID);
            String userName = loginResult.getString("Name");
            env.setUserName(userName);
            long loginTime = loginResult.getLong("Time");
            env.setTime(DateFormatUtils.format(new Date(loginTime), ENV_TIME_PATTERN));
            //模拟 PC(loginMode=1) / APP(loginMode=2) 登录
            env.setMode(loginMode);
            return clientID;
        } catch (Throwable e) {
            throw MiscUtil.toRuntimeException(e);
        }
    }

    private static void wrapEnvSetUserID(Env env, long userID) {
        // 解决有些yigo版本返回Long而不是long
        if (env.getUserID() instanceof Long) {
            env.setUserID(Long.valueOf(userID));
        } else {
            env.setUserID(userID);
        }
    }

    public static boolean attachSession(DefaultContext ctx, String clientId) {
        return attachSession(ctx, clientId, LOGIN_MODE_PC); //模拟 PC 登录
    }

    /**
     * 向当前上下文附加一个 Yigo 会话
     * @param ctx Yigo的Context
     * @param clientId 会话ID
     * @return 返回 true 代表成功附加上会话信息, false 代表因为无法找到会话等原因未成功附加会话信息
     */
    public static boolean attachSession(DefaultContext ctx, String clientId, int loginMode) {
        Env env = ctx.getEnv();

        ISessionInfo si = getLoginSession(clientId);
        if (null != si) {
            env.setClientID(si.getClientID());
            env.setClientIP(si.getIP());
            env.setClusterid(si.getClusterID());
            env.setGuestUserID(si.getGuestUserID());
            env.setMode(si.getMode());
            env.setSessionParas(si.getSessionParas());
            env.setTicketID(si.getTicketID());
            wrapEnvSetUserID(env, si.getOperatorID());
            env.setRoleIDList(si.getRoleIDList());
            env.setMode(loginMode);
            env.setSessionParas(si.getSessionParas());
            //返回成功信息
            return true;
        } else {
            //找不到 session 返回失败标识
            return false;
        }
    }

    public static boolean attachBackgroundSession(DefaultContext ctx, String userCode) {
        return attachBackgroundSession(ctx, userCode, LOGIN_MODE_PC);
    }

    /**
     * 使用操作员的 Code 向当前上下文附加一个 Yigo 会话(并未实际登录)
     * @param ctx Yigo的Context
     * @param userCode 用户账号
     * @return
     */
    public static boolean attachBackgroundSession(DefaultContext ctx, String userCode, int loginMode) {
        Env env = ctx.getEnv();

        IDictCache cache = ctx.getDictCache();
        Item item = null;
        try {
            item = cache.locate("Operator", "Code", userCode, null, null, DictStateMask.All, null, null, OperationState.Default);
        } catch (Throwable e) {
            throw MiscUtil.toRuntimeException(e);
        }

        if (null != item) {
            Long userID = item.getID();

            env.setClientID("BG-" + UUID.randomUUID());
            wrapEnvSetUserID(env, userID);
            env.setUserCode(userCode);
            //模拟 PC 登录
            env.setMode(LOGIN_MODE_PC);

            //返回成功信息
            return true;
        } else {
            //找不到用户返回失败标识
            return false;
        }

    }

    /**
     * 通过 clientID 获得登录会话信息
     * @param clientID 会话ID
     * @return
     */
    public static ISessionInfo getLoginSession(String clientID) {
        if (StringUtils.isBlank(clientID)) {
            return null;
        }
        ISessionInfoProvider p = SessionInfoProviderHolder.getProvider(LOGIN_MODE_PC);

        ISessionInfoMap sim = p.getSessionInfoMap();
        ISessionInfo session = sim.get(clientID);

        if (null == session) {
            p = SessionInfoProviderHolder.getProvider(LOGIN_MODE_MOBILE);
            session = p.getSessionInfoMap().get(clientID);
        }

        return session;
    }

    /**
     * 不需要密码(SSO模式)的登录扩展
     */
    public static class SSOLogin extends Login {
        public SSOLogin(LoginInfo loginInfo) {
            super(loginInfo);
        }

        protected void passwordCheck(DefaultContext ctx) {
            //不作密码校验
        }
    }

    /**
     * 根据传入的 Client ID, 在指定的上下文中运行相关业务
     * @param clientId 会话ID
     * @param runner YigoRunnable实现类,以callback函数的方式注入后续处理
     * @return
     */
    public static <T> T processWithContext(String clientId, YigoRunnable<T> runner) {
        DefaultContext context = null;
        try {
            context = ContextBuilder.create();
            attachSession(context, clientId);
            T result = runner.run(context);
            context.commit();
            return result;
        } catch (Throwable ex) {
            if (null != context) {
                try {
                    context.rollback();
                } catch (Throwable e) {
                    //Ignore it
                }
            }
            throw MiscUtil.toRuntimeException(ex);
        } finally {
            if (null != context) {
                try {
                    context.close();
                } catch (Throwable e) {
                    //Ignore it
                }
            }
        }
    }

    /**
     * 根据传入的操作员代码, 在指定的上下文中运行相关业务
     * @param userCode 用户账号
     * @param runner
     * @return
     */
    public static <T> T processWithBackgroundContext(String userCode, YigoRunnable<T> runner) {
        DefaultContext context = null;
        try {
            context = ContextBuilder.create();
            attachBackgroundSession(context, userCode);
            T result = runner.run(context);
            context.commit();
            return result;
        } catch (Throwable ex) {
            if (null != context) {
                try {
                    context.rollback();
                } catch (Throwable e) {
                    //Ignore it
                }
            }
            throw MiscUtil.toRuntimeException(ex);
        } finally {
            if (null != context) {
                try {
                    context.close();
                } catch (Throwable e) {
                    //Ignore it
                }
            }
        }
    }

    /**
     * 配合 {@link SessionUtils#processWithContext(String, YigoRunnable)} 的执行器
     */
    public static interface YigoRunnable<T> {

        /**
         * 回调函数,给与yigo上下文,执行字定义过程
         * @param ctx yigo上下文
         * @return 执行结果
         * @throws Throwable FIXME 因为 Yigo 的很多操作都抛出 Throwable，此处只能这样
         */
        public T run(DefaultContext ctx) throws Throwable;
    }
}
