package com.bokesoft.distro.tech.commons.basis.io.internal;

import com.bokesoft.base.bokebase.instance.ProcessInstanceUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


public class TempWriterManager {
    private static final int DEFAULT_BLOCK_TIME_MS = 5_000;
    /**
     * 文件对象的泛型类对应的writer参数缓存,用于提供创建class对应writer的必要参数
     */
    static final Map<String, WriterParam> PARAMS_CACHE = new ConcurrentHashMap<>();
    /**
     * Writer实例的缓存
     */
    static final Map<String, LocalFileTempWriter<?>> WRITERS_CACHE = new ConcurrentHashMap<>();
    /**
     * 监察线程轮询间隔,单位:秒
     */
    static long MONITOR_ROLLING_TIME = 10;
    /**
     * 监察线程实例
     */
    private static Thread monitor = null;
    /** 版本标记变量,建议较大改动,应升级版本 */
    public static final String VERSION = "v2";

    /**
     * 创建 对应对象存储泛型的 writer 预制参数缓存
     * @param clazz 内存对象实例的java类
     * @param tmpDir 临时区地址
     * @param readyDir 准备区地址
     */
    public static void bind(Class<?> clazz,String tmpDir, String readyDir, String errDir){
        PARAMS_CACHE.put(clazz.getName(),new WriterParam(tmpDir, readyDir, errDir, DEFAULT_BLOCK_TIME_MS));
    }

    /**
     * 修改文件区块时间
     * @param clazz
     * @param blockTime
     */
    public static void modifyBlockTime(Class<?> clazz, long blockTime){
        PARAMS_CACHE.get(clazz.getName()).setBlockTime(blockTime);
    }

    /**
     * 获取文件区块对应的对象输出流实例,一般创建后放入输出流缓存池,在同一文件区块时间内,获取相同writer实例
     * @param clazz
     * @return
     * @throws IOException
     */
    public static LocalFileTempWriter<?> getWriter(Class<?> clazz){
        String fileName = buildTmpFileName(clazz);
        return WRITERS_CACHE.computeIfAbsent(fileName,(key) -> {
            return new LocalFileTempWriter<>(fileName, clazz, PARAMS_CACHE.get(clazz.getName()));
        });
    }

    public static List<String> getActiveTmpFiles(){
        return new ArrayList<>(WRITERS_CACHE.keySet());
    }

    /**
     * 根据 对象泛型类,构造存储文件名
     * @param clazz 对象java类
     * @return
     */
    private static String buildTmpFileName(Class<?> clazz) {
        long blockTime = PARAMS_CACHE.get(clazz.getName()).getBlockTime();
        long timeCode = System.currentTimeMillis() / blockTime * blockTime;
        String name = clazz.getSimpleName()+"."+timeCode+"."+ ProcessInstanceUtil.getInstanceId()+"."+VERSION;
        name = changeIllegalCharacter(name);
        return name;
    }

    protected static String changeIllegalCharacter (String name) {
        return name.replaceAll("(:|;|\\\\|/|\\?|\\*)","_");
    }

    /**
     * 启动检查线程
     */
    public static void startMonitor(){
        if(null == monitor){
            monitor = new Thread(new WriterMonitor());
            synchronized (monitor) {
                monitor.start();
            }
        }
    }
}
