package com.bokesoft.distro.tech.commons.basis.auth;

import com.bokesoft.distro.tech.commons.basis.auth.crossauth.CrossAuthData;
import com.bokesoft.distro.tech.commons.basis.auth.crossauth.CrossAuthTokenSetting;
import com.bokesoft.distro.tech.commons.basis.auth.crossauth.ICrossAuthTokenSettingProvider;
import com.bokesoft.yes.common.encrypt.RSA;
import com.bokesoft.yes.common.util.Base64;
import com.bokesoft.yigo.common.dependency.DependencySortCore;
import com.bokesoft.yigo.common.dependency.IDependencySortable;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * 用于应用间访问的授权验证(CrossAuth)的基础工具类
 */
public class CrossAuthUtil {
    private static Logger logger = LoggerFactory.getLogger(CrossAuthUtil.class);

    /** 加密信息字符串以 HTTP Header 或 Parameter的方式传递,存放加密信息的 Header Name 或 Parameter Name */
    public static final String AUTH_WILDCARD = "x-bk-cross-auth";

    private static List<ICrossAuthTokenSettingProvider> providers;

    /**
     * 静态类初始化方法,注册{@link ICrossAuthTokenSettingProvider}集合,
     * 并且{@link ICrossAuthTokenSettingProvider}该集合会按{@link IDependencySortable}做依赖排序
     * @param providers
     */
    public static void setup(List<ICrossAuthTokenSettingProvider> providers){
        // 使用DependencySortCore 按依赖排序
        CrossAuthUtil.providers = DependencySortCore.sort(providers);
    }

    /**
     * 构造{@link CrossAuthData}对象的加密字符串
     * @param publicKey 用于加密{@link CrossAuthData}的公钥
     * @param authData 认证对象
     * @return 公钥加密后的字符串
     */
    public static String build(String publicKey, CrossAuthData authData) {
        try {
            RSA rsa = new RSA();
            ObjectMapper objectMapper = new ObjectMapper();
            String jsonStr = objectMapper.writeValueAsString(authData);
            byte[] data = jsonStr.getBytes();
            data = rsa.encryptByPublic(data, publicKey);
            String result = new String(Base64.encode(data));
            return result;
        } catch (Exception e) {
            return ExceptionUtils.rethrow(e);
        }
    }

    /**
     * 验证{@link CrossAuthData}对象的加密字符串
     * @param privateKey 用于解密{@link CrossAuthData}的私钥
     * @param authStr 加密的{@link CrossAuthData}字符串
     * @return 身份认证的结果,true为认证通过,false为认证失败
     */
    public static boolean doCheck(String privateKey, String authStr){
        CrossAuthData authData = decrypt(privateKey, authStr);
        CrossAuthTokenSetting setting = matchReleatdTokenSetting(authData);
        if( null != setting) {
            return checkTokenData(authData, setting);
        }else{
            logger.warn("未匹配到合适 callid = {} 的 Token 验证配置,无法验证有效性",authData.getCallerId());
            return false;
        }
    }

    /**
     * 将混淆后的{@link CrossAuthData} json字符串，以 privateKey 解密,并反序列化成@link OpsAuthRequest}
     * @param authStr
     * @return
     */
    protected static CrossAuthData decrypt(String privateKey, String authStr){
        try {
            byte[] data = Base64.decode(authStr.getBytes());
            data = (new RSA()).decryptByPrivate(data, privateKey);
            String jsonStr = new String(data);
            ObjectMapper mapper = new ObjectMapper();
            CrossAuthData request = mapper.readValue(jsonStr, CrossAuthData.class);
            return request;
        }catch (Throwable e){
            logger.error("无法解析CrossAuth加密信息",e);
            return ExceptionUtils.rethrow(e);
        }
    }

    /**
     * 将{@OpsAuthRequest} 对象与{@link CrossAuthTokenSetting}配置实例比较,校验是否通过token验证
     * @param request
     * @return
     */
    protected static boolean checkTokenData(CrossAuthData request, CrossAuthTokenSetting setting) {
        // 请求验证时间间隔前后不能小于5分钟
        if( System.currentTimeMillis() < request.getApplyTime() + setting.getMaxTimeDriftMs()
          && System.currentTimeMillis() > request.getApplyTime() - setting.getMaxTimeDriftMs()){
           String refToken = setting.getToken();
            // 请求的token和配置token比对
            if (request.getToken().equals(refToken)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 根据{@link CrossAuthData}实例的 callerid,找到对应比较的 token 验证配置
     * @param authData
     * @return
     */
    private static CrossAuthTokenSetting matchReleatdTokenSetting(CrossAuthData authData){
        for(ICrossAuthTokenSettingProvider authTokenSettingProvider:providers){
            CrossAuthTokenSetting setting = authTokenSettingProvider.getAuthTokenSetting(authData.getCallerId());
            if(null != setting){
                return setting;
            }
        }
        return null;
    }


}
