package com.bokesoft.distro.tech.bootsupport.starter.utils;

import com.bokesoft.distro.tech.bootsupport.starter.deployment.SpringResourceMultiSolutionMetaResolverFactory;
import com.bokesoft.distro.tech.yigosupport.deployment.resource.intf.IResourceIO;
import com.bokesoft.distro.tech.yigosupport.extension.base.log.LogSvrSlf4j;
import com.bokesoft.yes.mid.base.MidGlobalEnv;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ResourceLoader;

import java.io.*;
import java.net.URI;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

public class YigoInitUtil {
	private static final Logger log = LoggerFactory.getLogger(YigoInitUtil.class);
	
	/**
	 * 初始化 Yigo 环境(但不运行), 主要用于分析工具等开发目的
	 */
	public static void initYigoEnv(ResourceLoader resourceLoader, MultiSolutionPathProvider solutions,
			RunAfterYigoInitialized runner, boolean keepSilent) throws IOException {
		
		//首先初始化 SpringResourceMultiSolutionMetaResolverFactory - 从实现可见此处实际初始化了一个静态变量
		(new SpringResourceMultiSolutionMetaResolverFactory()).setResourceLoader(resourceLoader);
		
		//准备工作目录
		File workDir = Files.createTempDirectory(YigoInitUtil.class.getName()+"-").toFile();
		String workPath = workDir.getCanonicalPath();
		
		//在工作目录下准备好必要的 properites 文件
		log.info("开始初始化准备, 工作目录='{}' ...", workPath);
		prepareProperties(workDir, solutions.getSolutionResources());
		
		try {
			log.info("开始初始化 Yigo 环境 ...");
			long startMs = System.currentTimeMillis();
			//Yigo 环境初始化
			YigoPropPreparationHelper.PrepareResult prepareResult = new YigoPropPreparationHelper.PrepareResult(workPath, false);
			YigoServiceInitHelper.initYigoService(prepareResult);
			log.info("完成 Yigo 环境初始化, 耗时 {} 秒 .", (System.currentTimeMillis()-startMs)/1000);
			
			log.info("开始执行 Yigo 设计工具任务 ...");
			IMetaFactory metaFactory = MidGlobalEnv.getInstance().getMetaFactory();
			runner.run(metaFactory);
			
			log.info("Yigo 初始化完成, 工作目录='{}' .", workPath);
		}catch(Exception ex) {
			if (keepSilent) {
				log.error("Yigo 启动过程出错: {}", ExceptionUtils.getRootCauseMessage(ex));
			} else {
				log.error("Yigo 启动过程出错", ex);
				ExceptionUtils.rethrow(ex);
			}
		}
	}
	
	private static void prepareProperties(File workDir, String[] slnResources) throws IOException {
		final Properties coreProps = new Properties();

		//处理 Solution
		List<String> solutions = new ArrayList<>();
		for (int i=0; i<slnResources.length; i++){
		    String slnRes = slnResources[i];
		    String slnName = "sln-"+i;
		    solutions.add(slnName);
		    Properties slnProp = new Properties();

		    slnProp.setProperty("IMPL", SkipSthMetaResolverFactory.class.getName());

		    String paraRootRes = "PARA."+SpringResourceMultiSolutionMetaResolverFactory.PARA_KEY_ROOT_RESOURCE;
			slnProp.setProperty(paraRootRes ,slnRes);
			
		    savePropFile(workDir, slnName, slnProp, "Solution properties for "+slnRes);
		}
		//处理 core.properties
		coreProps.setProperty("SolutionPath", workDir.getCanonicalPath());
	    coreProps.setProperty("SOLUTIONS", StringUtils.join(solutions, ","));
	    coreProps.setProperty("LOGSVR", LogSvrSlf4j.class.getName());
	    coreProps.setProperty("DEBUG", "true");
	    savePropFile(workDir, "core", coreProps, "Yigo core properties");
	}

	private static void savePropFile(File workDir, String fileName, Properties properties, String desc)
			throws IOException {
		File propFile = new File(workDir, fileName+".properties");
		try(OutputStream out = Files.newOutputStream(propFile.toPath())){
			properties.store(out, desc);
		}
	}
	
	public static interface MultiSolutionPathProvider {
		public String[] getSolutionResources();
	}
	
	public static interface RunAfterYigoInitialized {
		public void run(IMetaFactory metaFactory);
	}
	
	public static class SkipSthMetaResolverFactory extends SpringResourceMultiSolutionMetaResolverFactory {
		/** 忽略二次开发相关的文件, 这样加载时不会调用各类初始化工作 */
		private static final String[] SKIP_FILES = new String[]{"/Enhance.xml", "/service_cmd.xml", "/service_filter.xml"};
		
		@Override
		protected IResourceIO getResourceIO() {
			final IResourceIO inner = super.getResourceIO();
			return new IResourceIO() {
				@Override
				public List<String> list() throws IOException {
					return inner.list();
				}
				@Override
				public byte[] read(String resource) throws IOException {
					for(String skip: SKIP_FILES) {
						if (resource.endsWith(skip)) {
							log.warn("文件 '{}' 已被忽略 .", resource);
							return null;
						}
					}
					return inner.read(resource);
				}
				@Override
				public void write(String resource, byte[] data) throws IOException {
					inner.write(resource, data);
				}
				@Override
				public String getInstanceId() {
					return inner.getInstanceId();
				}
				@Override
				public URI getURI(String resource) throws IOException {
					return inner.getURI(resource);
				}
				
			};
		}
	}
}
