package com.bokesoft.distro.tech.bootsupport.starter.readonly.impl;

import com.bokesoft.distro.tech.bootsupport.starter.config.ReadOnlyDbCheckConfig;
import com.bokesoft.distro.tech.bootsupport.starter.i18n.StringTable;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.struc.DBReadOnlyStatus;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.struc.DBWatchData;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.intf.IDBWatcher;
import com.bokesoft.distro.tech.commons.basis.trace.TraceUtil;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.mid.connection.DBType;
import com.bokesoft.yigo.mid.connection.IDBManager;
import com.bokesoft.yigo.struct.datatable.DataTable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;

public class MySQLDBWatcher implements IDBWatcher {
    private static final Logger logger= LoggerFactory.getLogger(MySQLDBWatcher.class);

    @Override
    public DBWatchData doCheck(DefaultContext context, ReadOnlyDbCheckConfig config) throws Throwable {
        String sql="show slave status";
        IDBManager dbManager = context.getDBManager();
        DataTable dt = dbManager.execPrepareQuery(sql);
        MysqlDBWatchData mysqlDBWatchData = null;
        if (dt.size() != 0) {
            mysqlDBWatchData = new MysqlDBWatchData(dt, config.getReplicationDelayThresholdS());
        } else {
            mysqlDBWatchData =  MysqlDBWatchData.ERROR(dbErrI18N());
        }
        return new DBWatchData(mysqlDBWatchData.getReason(),mysqlDBWatchData.getDescription(),
                mysqlDBWatchData.getStatus());
    }

    @Override
    public boolean support(int dbType) {
        return DBType.MySql == dbType;
    }

    private String dbErrI18N(){
        return StringTable.i18N(null,StringTable.MSG_DB_READONLY_CHECK_FAILD,TraceUtil.getTraceId());
    }

     static class MysqlDBWatchData {

        /** 同步主库服务的主机名,可以是IP地址 */
        private String masterHost = "unkown";
        /** 同步主库服务的端口 */
        private int masterPort = -1;
        /** 同步对应主库服务的binlog文件名 */
        private String masterLogFile = "unkown";
        /** 同步对应主库服务的binlog文件读取位置 */
        private int readMasterLogPos = -1;
        /** 从库同步日志的文件名 */
        private String relayLogFile = "unkown";
        /** 同步对应从库服务的relaylog文件读取位置 */
        private int relayLogPos = -1;
        /** 同步对应从库服务的relaylog文件执行结束的位置 */
        private int execMasterLogPos = -1;
        /** 主从同步的IO状态  */
        private String slaveIOState = "unkown";
        /** 主从同步的Sql执行状态  */
        private String slaveSQLRunningState = "unkown";
        /** 最近一次IO异常错误编号 */
        private int lastIOErrno = -1;
        /** 最近一次IO异常原因 */
        private String lastIOError = "unkown";
        /** 最近一次IO异常发生时间 */
        private String lastIOErrorTimestamp = "unkown";
        /** 最近一次同步SQL执行异常错误编号 */
        private int lastSQLErrno = -1;
        /** 最近一次同步SQL执行异常原因*/
        private String lastSQLError = "unkown";
        /** 最近一次同步SQL执行异常发生时间 */
        private String lastSQLErrorTimestamp = "unkown";
        /** 主从数据同步延迟多少秒 */
        private int secondsBehindMaster = -1;
        /** 主从同步数据延迟判断设置,单位秒 */
        private int delayThreshold = -1;
        /** 主从同步异常的原因 */
        private String reason;
        /** 主从同步健康状态 */
        private DBReadOnlyStatus status = DBReadOnlyStatus.ERROR;

        private MysqlDBWatchData(){

        }

        public MysqlDBWatchData(DataTable dt, int delayThreshold) throws SQLException {
            this.slaveIOState = dt.getString("Slave_IO_Running");
            this.slaveSQLRunningState = dt.getString("Slave_SQL_Running");
            this.masterHost = dt.getString("Master_Host");
            this.masterPort = dt.getInt("Master_Port");
            this.masterLogFile = dt.getString("Master_Log_File");
            this.readMasterLogPos = dt.getInt("Read_Master_Log_Pos");
            this.relayLogFile = dt.getString("Relay_Log_File");
            this.relayLogPos = dt.getInt("Relay_Log_Pos");
            this.execMasterLogPos = dt.getInt("Exec_Master_Log_Pos");
            this.lastIOErrno = dt.getInt("Last_IO_Errno");
            this.lastIOError = dt.getString("Last_IO_Error");
            this.lastIOErrorTimestamp = dt.getString("Last_IO_Error_Timestamp");
            this.lastSQLErrno = dt.getInt("Last_SQL_Errno");
            this.lastSQLError = dt.getString("Last_SQL_Error");
            this.lastSQLErrorTimestamp = dt.getString("Last_SQL_Error_Timestamp");
            this.secondsBehindMaster = dt.getInt("Seconds_Behind_Master");
            this.delayThreshold = delayThreshold;

            if(slaveSQLRunningState.equalsIgnoreCase("yes") && slaveIOState.equalsIgnoreCase("yes")){
                // 传输通道正常,但是数据延迟超过overtime,也认为同步异常
                if(secondsBehindMaster > delayThreshold){
                    this.status = DBReadOnlyStatus.ERROR;
                    this.reason = i18N4RsyncDelay();
                }else {
                    this.status =  DBReadOnlyStatus.OK;
                }
            }else {
                this.status =  DBReadOnlyStatus.ERROR;
                if(StringUtils.isNotBlank(lastIOError)) {
                    this.reason = i18N4RsyncIOError();
                }else if(StringUtils.isNotBlank(lastSQLError)){
                    this.reason = i18N4RsyncSqlError();
                }else{
                    this.reason = i18N4RsyncOtherError();
                }
            }
        }

        public String getReason() {
            return reason;
        }

        public DBReadOnlyStatus getStatus() {
            return this.status;
        }

        public String getDescription() {
            String result = "\n=====================================================\n";
            result += "master sever: " +masterHost+":"+masterPort+"\n";
            result += "slave status: "+ status +"\n";
            result += "error reason:" + reason +"\n";
            result += "masterLogFile: "+masterLogFile+", readMasterLogPos: "+ readMasterLogPos+" \n";
            result += "relayLogFile: "+relayLogFile+", relayLogPos: "+relayLogPos+", execMasterLogPos: "+execMasterLogPos+" \n";
            result += "slaveIOState: "+slaveIOState+ "\n";
            result += "slaveSQLRunningState: "+slaveSQLRunningState+ ", secondsBehindMaster: "+secondsBehindMaster+"\n";
            result += "lastIOErrno: "+lastIOErrno+", lastIOErrorTimestamp: "+lastIOErrorTimestamp+"\n";
            result += "lastIOError: "+lastIOError+"\n";
            result += "lastSQLErrno: "+lastSQLErrno+", lastSQLErrorTimestamp: "+lastSQLErrorTimestamp+"\n";
            result += "lastSQLError: "+lastSQLError+"\n";
            result += "=====================================================";
            return  result;
        }

        public static MysqlDBWatchData ERROR(String reason){
            MysqlDBWatchData data = new MysqlDBWatchData();
            data.reason = reason;
            return data;
        }



        private String i18N4RsyncDelay() {
            return StringTable.i18N(null,StringTable.MSG_MYSQL_RSYNC_DELAY,
                    delayThreshold,secondsBehindMaster,TraceUtil.getSpanId());
        }

        private String i18N4RsyncIOError() {
            return StringTable.i18N(null,StringTable.MSG_MYSQL_RSYNC_IO_ERROR,
                    lastIOErrorTimestamp,lastIOError,TraceUtil.getSpanId());
        }

        private String i18N4RsyncSqlError() {
            return StringTable.i18N(null,StringTable.MSG_MYSQL_RSYNC_SQL_ERROR,
                    lastSQLErrorTimestamp,lastSQLError,TraceUtil.getSpanId());
        }

        private String i18N4RsyncOtherError() {
            return StringTable.i18N(null,StringTable.MSG_MYSQL_RSYNC_OTHER_ERROR,TraceUtil.getSpanId());
        }
    }
}
