package com.bokesoft.distro.tech.bootsupport.starter.execctl;

import java.util.*;

import com.bokesoft.distro.tech.bootsupport.starter.config.ExecTimeoutCtrlConfig;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.exception.ExecTimeoutException;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.model.StartTimeObject;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.utils.ExecutionControlUtils;
import com.bokesoft.distro.tech.yigosupport.extension.utils.yigo.ServiceIdPartsUtil;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.mid.base.IServiceContext;
import com.bokesoft.yigo.struct.document.Document;

public class ExecutionTimeoutManager {

    /**
     * 进入服务层时缓存服务的serviceId,主要时用来计算相关的超时时间配置
     */
    private static ThreadLocal<String> tl_serviceId = new ThreadLocal<>();

    /**
     * 缓存dataObjectPath到栈中,根据栈先进后出的特性，在计算DataObject的配置时，可以获取最后进入DataObject的path进行计算
     */
    private static ThreadLocal<Stack<String>> tl_dataObjectPaths = new ThreadLocal<>();

    /**
     * 并行集合,可以根据instanceId命中单个计算节点
     */
    private static ThreadLocal<List<StartTimeObject>> tl_parallelSTOContainer = new ThreadLocal<>();

    /**
     * 串行集合以计算整条链的超时的原则，所以全部涵盖在计算范围内
     */
    private static ThreadLocal<List<StartTimeObject>> tl_unParallelSTOContainer = new ThreadLocal<>();

    /**
     * 超时时间配置管控
     */
    private static ExecTimeoutCtrlConfig executionTimeoutControl;

    public static void setExecutionTimeoutControl(ExecTimeoutCtrlConfig executionTimeoutControl) {
        ExecutionTimeoutManager.executionTimeoutControl = executionTimeoutControl;
    }

    /**
     * 初始化 ThreadLocal
     */
    public static void init() {
        Stack<String> dataObjectInstanceIds = new Stack<>();
        tl_dataObjectPaths.set(dataObjectInstanceIds);
        List<StartTimeObject> parallelSTOContainer = new ArrayList<>();
        tl_parallelSTOContainer.set(parallelSTOContainer);
        List<StartTimeObject> unParallelSTOContainer = new ArrayList<>();
        tl_unParallelSTOContainer.set(unParallelSTOContainer);
    }

    /**
     * 清理所有缓存
     */
    public static void reset() {
        tl_serviceId.remove();
        tl_dataObjectPaths.remove();
        tl_parallelSTOContainer.remove();
        tl_unParallelSTOContainer.remove();
    }

    /**
     * 进入服务,根据serviceId计算相关超时时间的配置
     */
    public static void enterService(IServiceContext serviceContext, Map<String, Object> args) throws Throwable {
        String serviceId = ServiceIdPartsUtil.patchServiceId(serviceContext, args);
        //记录serviceId
        enterService(serviceId);
    }

    static void enterService(String serviceId) {
        //记录serviceId
        tl_serviceId.set(serviceId);
    }

    /**
     * 离开服务,清理缓存
     */
    public static void exitService() {
        tl_serviceId.remove();
    }

    /**
     * 进入数据对象,根据dataObjectId计算相关的超时时间配置
     */
    public static String enterDataObject(MetaDataObject dataObject) {
        String dataObjectPath = dataObject.getProjectKey() + "/" + dataObject.getKey();
        return enterDataObject(dataObjectPath);
    }

    static String enterDataObject(String dataObjectPath) {
        pushDataObjectPath(dataObjectPath);
        return dataObjectPath;
    }

    /**
     * 离开数据对象
     */
    public static void exitDataObject() {
        //DataObject的instanceId弹栈
        popDataObjectPath();
    }

    /**
     * 获取DataObject的开始时间对象
     */
    public static StartTimeObject getStartTimeObject(Document doc) {
        String dataObjectInstanceId = (null == doc.getExpandData(StartTimeObject.class.getName())) ? "" : (String) doc.getExpandData(StartTimeObject.class.getName());
        List<StartTimeObject> startTimeObjects = tl_unParallelSTOContainer.get();
        if (null != startTimeObjects && startTimeObjects.size() > 0) {
            for (int i = startTimeObjects.size() - 1; i >= 0; i--) {
                StartTimeObject startTimeObject = startTimeObjects.get(i);
                String instanceId = startTimeObject.getInstanceId();
                if (dataObjectInstanceId.equals(instanceId)) {
                    return startTimeObject;
                }
            }
        }
        return null;
    }

    /**
     * 加入开始时间对象
     */
    public static void addStartTimeObject(StartTimeObject startTimeObject) {
        if (null != startTimeObject) {
            List<StartTimeObject> startTimeObjects;
            if (startTimeObject.isParallel()) {
                startTimeObjects = tl_parallelSTOContainer.get();
            } else {
                startTimeObjects = tl_unParallelSTOContainer.get();
            }
            if (null != startTimeObjects) {
                startTimeObjects.add(startTimeObject);
            }
        }
    }

    /**
     * 移除开始时间对象
     */
    public static void removeStartTimeObject(StartTimeObject startTimeObject) {
        if (null != startTimeObject) {
            List<StartTimeObject> startTimeObjects;
            if (startTimeObject.isParallel()) {
                startTimeObjects = tl_parallelSTOContainer.get();
            } else {
                startTimeObjects = tl_unParallelSTOContainer.get();
            }
            if (null != startTimeObjects) {
                startTimeObjects.remove(startTimeObject);
            }
        }
    }

    /**
     * 分配剩余时间
     */
    public static long allocRemainedTime() throws ExecTimeoutException {
        return allocRemainedTime(null);
    }

    /**
     * 分配剩余时间
     */
    public static long allocRemainedTime(String instanceId) throws ExecTimeoutException {
        return ExecutionControlUtils.calcAllocRemainedTime(executionTimeoutControl, tl_parallelSTOContainer.get(),
                tl_unParallelSTOContainer.get(), tl_dataObjectPaths.get(), tl_serviceId.get(), instanceId);
    }

    /**
     * DataObject的instanceId入栈
     */
    private static void pushDataObjectPath(String dataObjectPath) {
        Stack<String> dataObjectPaths = tl_dataObjectPaths.get();
        if (null != dataObjectPaths) {
            dataObjectPaths.push(dataObjectPath);
        }
    }

    /**
     * DataObject的instanceId弹栈
     */
    private static String popDataObjectPath() {
        String dataObjectPath = "";
        Stack<String> dataObjectPaths = tl_dataObjectPaths.get();
        if (null != dataObjectPaths && dataObjectPaths.size() > 0) {
            dataObjectPath = dataObjectPaths.pop();
        }
        return dataObjectPath;
    }
}
