package com.bokesoft.distro.tech.bootsupport.starter.deployment;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternUtils;

import com.bokesoft.distro.tech.bootsupport.starter.utils.URLPathUtils;
import com.bokesoft.distro.tech.yigosupport.deployment.resource.intf.IResourceIO;

import jodd.util.Wildcard;

/**
 * 使用 Spring {@link ResourceLoader} 的资源读写功能实现
 */
public class SpringResourceIO implements IResourceIO {
    private static final Logger log = LoggerFactory.getLogger(SpringResourceIO.class);
    
    /** 资源根路径 */
    private String rootResourcePath;
    /** Spring resource loader, 用于支持以 resource 的方式 (classpath://... 、file://... 、 http://... ) 加载资源 */
    private ResourceLoader resourceLoader;
    /** 需要忽略的资源路径(相对路径) */
    private List<String> ignoreResourcePaths;
    /** 需要忽略的资源路径, 支持以路径通配符的方式定义. */
    private List<String> ignoreResourceFullPathPatterns;

    public SpringResourceIO(
            String rootResourcePath, ResourceLoader resourceLoader,
            List<String> ignoreResourcePaths, List<String> ignoreResourceFullPathPatterns){
        this.rootResourcePath = rootResourcePath;
        this.resourceLoader = resourceLoader;

        this.ignoreResourcePaths = ignoreResourcePaths;
        this.ignoreResourceFullPathPatterns = ignoreResourceFullPathPatterns;

        log.info("Spring 资源读写: instance=[{}], rootResourcePath='{}', ignoreResourcePaths='{}', ignoreResourceFullPathPatterns='{}'.",
                this.getInstanceId(), rootResourcePath, ignoreResourcePaths, ignoreResourceFullPathPatterns);
    }

    private void lazyValidate(){
        if (StringUtils.isBlank(rootResourcePath)){
            throw new UnsupportedOperationException("Field 'rootResourcePath' is not invalid.");
        }
        if (null==resourceLoader){
            throw new UnsupportedOperationException("Field 'resourceLoader' is null.");
        }

    }

    @Override
    public List<String> list() throws IOException{
        lazyValidate();

        Resource root = resourceLoader.getResource(rootResourcePath);
        String rootPath = root.getURL().getPath();
        
        String pattern = _fullPath("/**/*.*");
        ResourcePatternResolver patternResolver = ResourcePatternUtils.getResourcePatternResolver(resourceLoader);
        Resource[] resources = patternResolver.getResources(pattern);
        
        List<String> result = new ArrayList<>();
        for (Resource r: resources) {
            String path = r.getURL().getPath();
            path = URLPathUtils.relativizeOf(path, rootPath);
            result.add(path);
        }
        return result;
    }

    private boolean isSkip(String resource){
        if (null==resource){
            return true;
        }

        if (null != ignoreResourcePaths) {
            for (String pattern : ignoreResourcePaths) {
                if (Wildcard.matchPath(resource, pattern)) {
                    log.info("[{}]: 忽略资源 '{}' (匹配 '{}') .",
                            this.getInstanceId(), resource, pattern);
                    return true;
                }
            }
        }

        //与 ignoreResourcePaths, 来自全局配置的忽略资源定义中包含完整资源路径
        if (null!=ignoreResourceFullPathPatterns){
            String fullPath = _fullPath(resource);
            for (String pattern: ignoreResourceFullPathPatterns){
                if (Wildcard.matchPath(fullPath, pattern)) {
                    log.info("[{}]: 忽略资源 '{}' (匹配 Path Wildcard '{}') .",
                            this.getInstanceId(), fullPath, pattern);
                    return true;
                }
            }
        }

        return false;
    }
    private String _fullPath(String resource){
        String fullPath = URLPathUtils.get(rootResourcePath, resource);
        return fullPath;
    }
    private Resource getResource(String resource){
        if (isSkip(resource)) {
            return null;
        }
        String resPath = _fullPath(resource);
        Resource res = resourceLoader.getResource(resPath);
        return res;
    }
    @Override
    public byte[] read(String resource) throws IOException {
        lazyValidate();

        Resource res = getResource(resource);
        if (null!=res && res.exists()) {
            try (InputStream is = res.getInputStream()) {
                return IOUtils.toByteArray(is);
            }
        } else {
            return null;
        }
    }
    @Override
    public void write(String resource, byte[] data) throws IOException{
        lazyValidate();

        Resource res = getResource(resource);
        if (null==res){
            throw new UnsupportedOperationException(
                "["+this.getInstanceId()+"]: Can't get resource '"+resource+"' to write.");
        }
        if (res.isFile()){
            File f = res.getFile();
            FileUtils.writeByteArrayToFile(f, data);
        }else{
            throw new UnsupportedOperationException(
                "["+this.getInstanceId()+"]: Writing resource '"+resource+"' is not supported because of it's not File.");
        }
    }


    @Override
    public URI getURI(String resource) throws IOException {
        lazyValidate();

        Resource res = getResource(resource);
        if (null==res){
            return null;
        }
        return res.getURI();
    }

    @Override
    public String getInstanceId() {
        return rootResourcePath + "@" + Integer.toHexString(this.hashCode());
    }

}
