package com.bokesoft.distro.tech.bootsupport.starter.configurer;

import com.bokesoft.distro.tech.bootsupport.starter.config.ReadOnlyDbCheckConfig;
import com.bokesoft.distro.tech.bootsupport.starter.i18n.StringTable;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.impl.MySQLDBWatcher;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.struc.DBWatchData;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.intf.IDBWatcher;
import com.bokesoft.distro.tech.bootsupport.starter.readonly.struc.DBReadOnlyStatus;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.YigoInstanceManager;
import com.bokesoft.distro.tech.commons.basis.dependency.DependencySortCore;
import com.bokesoft.distro.tech.commons.basis.trace.TraceUtil;
import com.bokesoft.distro.tech.commons.basis.trace.intf.ITraceSupplier;
import com.bokesoft.distro.tech.yigosupport.extension.utils.yigo.SessionUtils;
import com.bokesoft.yes.mid.base.ServerSetting;
import com.bokesoft.yigo.common.trace.TraceRecorder;
import com.bokesoft.yigo.common.trace.TraceSystemManager;
import com.bokesoft.yigo.mid.connection.IDBManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;


@Configuration
@EnableScheduling
public class ReadOnlyWatchConfigurer {

    private static final Logger logger= LoggerFactory.getLogger(ReadOnlyWatchConfigurer.class);

    private static final List<IDBWatcher> dbWatcherList = new ArrayList<>();

    @Autowired
    private YigoInstanceManager yigoInstanceManager;

    @Autowired
    private ReadOnlyDbCheckConfig config;

    @PostConstruct
    public void initDBWatcher(){
        ServiceLoader<IDBWatcher> loader = ServiceLoader.load(IDBWatcher.class);
        List<IDBWatcher> cache = new ArrayList<>();
        Iterator<IDBWatcher> it = loader.iterator();
        while(it.hasNext()){
            cache.add(it.next());
        }
        dbWatcherList.addAll(DependencySortCore.sort(cache));
    }

    /**
     * 每5分钟,做一次主从可靠性状态检查
     */
    @Scheduled(cron="${yigoee.tech.bootsupport.readonly-follower.db-watch.schedule: 0 */5 * * * ?}")
    public void task(){
        ServerSetting serverSetting = ServerSetting.getInstance();
        if (null != serverSetting && serverSetting.isReadOnly()) {
            if (yigoInstanceManager.isInitializationComplete()) {
                TraceSystemManager.withTraceLog(
                        () -> {
                            doReadOnlyCheck();
                            return null;
                        }, this, true, logger, Level.INFO
                );
            }
        }
    }

    private void doReadOnlyCheck(){
        try {
            SessionUtils.processWithContext(null, context -> {
                IDBManager idbManager = context.getDBManager();
                IDBWatcher watcher = getSuitableDBWather(idbManager);
                if (null == watcher) {
                    return null;  // 当前数据库类型没有可用的Watcher
                } else {
                    DBWatchData watchData = watcher.doCheck(context, config);
                    if(null == watchData){
                        logger.error("只读服务健康检查: 返回数据不正确( watchData = null )");
                        yigoInstanceManager.markUnavaliableReason(ReadOnlyWatchConfigurer.class,
                                watchDataIsNullI18N());
                    }else if (DBReadOnlyStatus.OK.equals(watchData.getStatus())) {
                        logger.info("只读服务健康检查: 状态正常");
                        yigoInstanceManager.cleanUnavaliableEvent(ReadOnlyWatchConfigurer.class);
                    } else {
                        logger.error("只读服务健康检查: 状态异常, {}", watchData.getDescription());
                        yigoInstanceManager.markUnavaliableReason(ReadOnlyWatchConfigurer.class,
                                watchData.getReason());
                    }
                }
                return null;
            });
        }catch (Exception e){
            logger.error(e.getMessage(),e);
            yigoInstanceManager.markUnavaliableReason(ReadOnlyWatchConfigurer.class,
                    dbQueryErrI18N());
        }
    }

    /**
     * 获取适配的DBWatcher处理程序
     * @param dbManager 数据连接管理起
     * @return
     */
    private IDBWatcher getSuitableDBWather(IDBManager dbManager){
        int dbType = dbManager.getDBType();
        for(IDBWatcher watcher: dbWatcherList){
            if(watcher.support(dbType)){
                return watcher;
            }
        }
        logger.warn("只读服务健康检查: 找不到 dbType {} 对应的 IDBWatcher", dbType);
        return null;
    }

    private String dbQueryErrI18N(){
        return StringTable.i18N(null,StringTable.MSG_DB_READONLY_CHECK_FAILD, TraceRecorder.getTraceId());
    }

    private String watchDataIsNullI18N(){
        return StringTable.i18N(null,StringTable.MSG_WATCHDATA_IS_NULL, TraceRecorder.getTraceId());
    }

}
