package com.bokesoft.distro.tech.bootsupport.starter.beans;

import com.bokesoft.distro.tech.bootsupport.starter.event.YigoReloadRequestEvent;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.web.server.WebServer;
import org.springframework.boot.web.servlet.context.ServletWebServerApplicationContext;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.stereotype.Component;

import com.bokesoft.distro.tech.bootsupport.starter.runtime.YigoInstanceManager;
import org.springframework.util.Assert;

/**
 * 在 {@link ContextRefreshedEvent} 事件中完成 Yigo 初始化
 */
@Component
public class YigoInitStage2 implements ApplicationListener<ApplicationEvent> {
    private static Logger log = LoggerFactory.getLogger(YigoInitStage2.class);

    @Value("${server.port:-1}")
    int mainWebPort;

    @Autowired
    private YigoInstanceManager yigoInstanceManager;

    @Override
    public void onApplicationEvent(ApplicationEvent appEvt) {
        boolean reload = false;
        ApplicationContext ctx;
        if (appEvt instanceof YigoReloadRequestEvent) {
            YigoReloadRequestEvent event = (YigoReloadRequestEvent) appEvt;
            ctx = event.getApplicationContext();
            reload = true;
            log.info("开始 Yigo 启动(YigoReloadRequestEvent), ApplicationContext={} ...", ctx.getDisplayName());
        } else if (appEvt instanceof ContextRefreshedEvent){
            ContextRefreshedEvent event = (ContextRefreshedEvent) appEvt;
            ctx = event.getApplicationContext();

            //只能响应顶层 ApplicationContext 的 ContextRefreshedEvent
            if (! isMainWebContext(ctx)){
                log.debug("忽略非当前主 Web 应用的 ApplicationContext {} .", ctx.getDisplayName());
                return;
            }

            log.info("开始 Yigo 启动(Main ServletWebServerApplicationContext), ApplicationContext={} ...", ctx.getDisplayName());
        }else{
            log.debug("忽略的 ApplicationEvent: {} .", appEvt);
            return;
        }

        try {
            YigoInstanceManager.BOOTING_WITH_SPRING_BOOT = true;
            boolean isSilentStart = yigoInstanceManager.bootPrepare(ctx,reload);
            if (!isSilentStart) {
                yigoInstanceManager.bootLoad();
            }
        } catch (Exception ex) {
            log.error("Yigo 启动过程出错，系统退出", ex);
            //Yigo 初始化出错之后, 强制系统退出
            log.error("Yigo 启动过程出错，系统退出: {}", ExceptionUtils.getRootCauseMessage(ex));
            // 保证exitcode不为0
            System.exit(SpringApplication.exit(ctx, () -> -1));
        } finally {
            YigoInstanceManager.BOOTING_WITH_SPRING_BOOT = false;
        }
        log.info("Yigo 启动完成, ApplicationContext={} .", ctx.getDisplayName());
    }

    private boolean isMainWebContext(ApplicationContext ctx){
        Assert.isTrue(this.mainWebPort != -1,
                "目前必须借助 server.port 配置来判断主服务端口, 此配置项不可缺失");

        String ctxName = ctx.getDisplayName();
        if (! (ctx instanceof ServletWebServerApplicationContext) ){
            log.debug("忽略非 ServletWebServerApplicationContext 类型的 ApplicationContext: {} .", ctxName);
            return false;
        }
        ServletWebServerApplicationContext webCtx = (ServletWebServerApplicationContext) ctx;
        WebServer webServer = webCtx.getWebServer();
        if(webServer==null){
            log.info("webServer 为空,当前应用运行在war模式下 .");
            return true;
        }
        int port = webServer.getPort();
        if (port == this.mainWebPort){
            log.info("通过 {} 端口定位当前应用的主 Web ApplicationContext: {} .", port, ctxName);
            return true;
        }else{
            log.debug("忽略端口({}) 不等于 {} 的 ApplicationContext: {} .", port, this.mainWebPort, ctxName);
            return false;
        }
    }

}