package com.bokesoft.distro.tech.bootsupport.starter.api.ctx;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoRawConfiger;
import com.bokesoft.distro.tech.commons.basis.MiscUtil;
import com.bokesoft.yigo.meta.factory.IMetaResolverFactory;

/**
 * {@link YigoRawConfiger} 的运行上下文
 */
public abstract class ConfigerContext {
	private static final Logger log = LoggerFactory.getLogger(ConfigerContext.class);

	private static final String FILENAME_CORE_PROPERTIES = "core.properties";
	private static final String PROP_PARA_PREFIX = "PARA.";
	private static final String PROP_IMPL = "IMPL";
	private static final String PROP_PRIMARY = "PRIMARY";

	private static final String PROP_SOLUTIONS = "SOLUTIONS";
	
	/** 附加文件内容; Key-文件名、Value-文件内容 */
	private Map<String, String> additionalFiles = new HashMap<String, String>();

	/** 主 Solution 的名称 */
	private String primarySolutionName = null;
	
	/**
	 * 增加配置文件
	 * @param fileName
	 * @param content
	 */
	private void addFile(String fileName, String content) {
		additionalFiles.put(fileName, content);
	}
	
	/**
	 * 增加配置文件
	 * @param fileName
	 * @param properties
	 */
	private void addFile(String fileName, Properties properties) {
		try (Writer w = new StringWriter()){
			properties.store(w, "Modified by "+this.getClass().getName());
			String content = w.toString();
			addFile(fileName, content);
		}catch(IOException e) {
			throw MiscUtil.toRuntimeException(e);
		}
	}

	/**
	 * 增加一个 Solution（注意：只支持多 Solution 的情况）
	 * @param solutionName
	 * @param resolverFac
	 * @param properties
	 */
	public synchronized void addSolution(String solutionName,
			Class<? extends IMetaResolverFactory> resolverFac, Map<String, String> properties) {
		addSolution(solutionName, false, resolverFac, properties);
	}
	
	/**
	 * 增加一个 Solution（注意：只支持多 Solution 的情况）
	 * @param solutionName
	 * @param asPrimary 是否作为一个主 Solution 加入
	 * @param resolverFac
	 * @param properties
	 */
	public synchronized void addSolution(String solutionName, boolean asPrimary,
			Class<? extends IMetaResolverFactory> resolverFac, Map<String, String> properties) {
		String fileName = solutionName+".properties";
		if (null==properties) {
			properties = new HashMap<>();
		}
		
		Properties target = new Properties();
		//属性写入 solution 定义的目标文件时, 需要加上 PARA. 前缀
		for(Map.Entry<String, String> en: properties.entrySet()) {
			String key = PROP_PARA_PREFIX+en.getKey();
			String val = en.getValue();
			target.setProperty(key, val);
		}
		//写入实现类
		target.setProperty(PROP_IMPL, resolverFac.getName());
		//处理需要作为主 Solution 加入的情况
		if (asPrimary){
			if (null==this.primarySolutionName){
				this.primarySolutionName = solutionName;
				target.setProperty(PROP_PRIMARY, Boolean.TRUE.toString());
				log.info("已将 Solution '{}' 设置为主 Solution .", solutionName);
			}else{
				throw new UnsupportedOperationException(
					"无法设置 '"+solutionName+"' 为主 Solution: 与当前主 Solution 设置 '"+this.primarySolutionName+"' 冲突");
			}
		}
		addFile(fileName, target);
		
		//修改 core.properties 中的 SOLUTIONS 属性
		String core = _getFile(FILENAME_CORE_PROPERTIES);
		MiscUtil.$assert(StringUtils.isBlank(core),
				"未找到 "+FILENAME_CORE_PROPERTIES+", 无法执行 addSolution '"+solutionName+"'");
		Properties coreProp = new Properties();
		try(Reader r = new StringReader(core)){
			coreProp.load(r);
		}catch(IOException e) {
			throw MiscUtil.toRuntimeException(e);
		}
		String solutions = coreProp.getProperty(PROP_SOLUTIONS);
		if(StringUtils.isBlank(solutions)){
			coreProp.setProperty(PROP_SOLUTIONS, solutionName);
		}else {
			coreProp.setProperty(PROP_SOLUTIONS, solutions+","+solutionName);
		}
		addFile(FILENAME_CORE_PROPERTIES, coreProp);
	}

	private String _getFile(String fileName) {
		String content = additionalFiles.get(fileName);
		if (StringUtils.isNotEmpty(content)) {
			return content;
		}else {
			return getFile(fileName);
		}
	}

	/**
	 * 获得配置文件的内容; 此方法需要子类实现。
	 * @param fileName
	 * @return
	 */
	public abstract String getFile(String fileName);

	/**
	 * 获取附加的文件内容, Key-文件名、Value-文件内容
	 * @return
	 */
	public Map<String, String> getAdditionalFiles() {
		return additionalFiles;
	}

}