package com.bokesoft.yes.meta.persist.dom.xml;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.DefaultNode;
import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.DefaultNodeDefine;
import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.LinkedData;
import com.bokesoft.yes.meta.persist.dom.xml.node.AbstractNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.CDataNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.TagNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.TextNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.XmlTree;

/**
 * 根据原始文件和模板生成标志格式的xml文件
 *
 * @author chenbinbin
 *
 */
public class XmlCreator {

	private Document doc;

	private XmlTree orgTree;

	private LinkedHashMap<String, TagNode> mapNode = new LinkedHashMap<String, TagNode>();

	private DefaultNodeDefine defualtDefine = DefaultNodeDefine.getInstance();

	public XmlCreator(Document doc, XmlTree orgTree) {
		this.doc = doc;
		this.orgTree = orgTree;
	}

	public XmlTree createXmlTree() {
		mapNode.clear();
		XmlTree newTree = new XmlTree();
		Element elem = doc.getDocumentElement();
		String key = getKey(elem);
		TagNode node = (orgTree == null ? null : orgTree.getTagNode(key));
		TagNode newRoot = createNewNodeByBaseNode(elem, node);
		createTagNode(elem, newRoot, true);
		if (orgTree != null) {
			newRoot.addPreComment(orgTree.getRoot().getPreComment());
			newRoot.addLastComment(orgTree.getRoot().getLastComment());
		}
		newTree.setRoot(newRoot);
		return newTree;
	}

	public String createXml() {
		XmlTree xmlTree = createXmlTree();
		createComments(xmlTree);
		return xmlTree.getRoot().toXml(0);
	}

	public HashMap<String, TagNode> getTagNodeMap() {
		return this.mapNode;
	}

	/**
	 * 根据原文件信息添加注释
	 *
	 * @param xmlTree
	 */
	private void createComments(XmlTree xmlTree) {
		if (orgTree == null) {
			return;
		}
		TagNode root = xmlTree.getRoot();
		TagNode virtual = new TagNode();
		virtual.addNode(root);
		List<AbstractNode> listNode = orgTree.getNodesWithComment();
		for (AbstractNode nodeWithComment : listNode) {
			Stack<AbstractNode> stackPath  = getCommentPath(nodeWithComment);
			TagNode levelNode = virtual;
			boolean bFind = false;
			AbstractNode pathNode = null;
			while(!stackPath.isEmpty() && levelNode != null) {
				pathNode = stackPath.pop();
				if (pathNode instanceof TagNode) {
					TagNode tagNode = (TagNode) pathNode;
					levelNode = levelNode.findChildByTagNode((TagNode) tagNode);
					bFind = (levelNode != null);
				} else {
					bFind = false;
					break;
				}
			}
			if (bFind) {
				levelNode.addPreComment(pathNode.getPreComment());
				levelNode.addLastComment(pathNode.getLastComment());
			}
		}
	}

	private Stack<AbstractNode> getCommentPath(AbstractNode nodeWithComment) {
		Stack<AbstractNode> stack = new Stack<AbstractNode>();
		AbstractNode parent = nodeWithComment;
		while (parent != null) {
			stack.push(parent);
			parent = parent.getParent();
		}
		return stack;
	}

	private void createTagNode(Element elem, TagNode parentTagNode, boolean isRoot) {
		String key = getKey(elem);
		TagNode node = (orgTree == null ? null : orgTree.getTagNode(key));
		//根据模板和原始文件节点设置新节点
		TagNode newTagNode = null;
		if (!isRoot) {
			newTagNode = createNewNodeByBaseNode(elem, node);
			parentTagNode.addNode(newTagNode);
			String par="";
			if (newTagNode.getParent()!=null){
				par = newTagNode.getParent().getPrimaryKey();
				if (newTagNode.getParent().getParent()!=null){
					par = newTagNode.getParent().getParent().getPrimaryKey()+"->"+par;
				}
			}
			mapNode.put(par+"->"+newTagNode.getPrimaryKey(), newTagNode);
		} else {
			newTagNode = parentTagNode;
		}


		Node child = null;
		String sContent = null;
		NodeList nodeList = elem.getChildNodes();
		for(int n=0; n<nodeList.getLength(); n++) {
			child = nodeList.item(n);
			if (Element.ELEMENT_NODE == child.getNodeType()) {
				createTagNode((Element) child, newTagNode, false);
			} else if (Element.CDATA_SECTION_NODE == child.getNodeType()) {
				//CDataNode cDataNode = createCDataNode(elem);
				newTagNode.addNode(new CDataNode(child.getNodeValue()));
			} else if(Element.TEXT_NODE == child.getNodeType()){
				sContent = child.getTextContent();
				if (sContent != null && sContent.trim().length() > 0) {
					newTagNode.addNode(new TextNode(sContent));
				}
			}
		}
	}

	/**
	 * 根据当前配置文件文件中已有节点，生成新的节点
	 * @param document
	 * @param curElem
	 * @param baseFileXmlNode
	 * @return
	 */
	private TagNode createNewNodeByBaseNode(Element curElem, TagNode baseTagNode) {
		if (baseTagNode == null) {
			return createNewNodeByDefault(curElem);
		}

		String curNodeName = curElem.getNodeName();
		DefaultNode newElementAttrs = new DefaultNode(curNodeName);
		//添加当前dom节点对象中存在， 且文件节点对象中也存在的属性
		Set<String> keySet = baseTagNode.getAttributes().keySet();
		Iterator<String> iter = keySet.iterator();
		while(iter.hasNext()) {
			String attrName = iter.next();
			if (curElem.hasAttribute(attrName)) {
				newElementAttrs.addNode(new LinkedData(attrName));
			}
		}

		//添加当前dom节点对象中存在且默认定义文件中也存在,但当前文件节点对象中不存在的属性
		DefaultNode defaultNode = defualtDefine.getDefaultNode(curNodeName);
		NamedNodeMap curAttrs = curElem.getAttributes();
		for (int i=0; i<curAttrs.getLength(); i++) {
			String attrName = curAttrs.item(i).getNodeName();
			if (!newElementAttrs.hasAttribute(attrName) ) {
				if(defaultNode != null && defaultNode.hasAttribute(attrName)) {
					LinkedData lastDefinedAttr = findLastDefinedAttr(newElementAttrs, defaultNode);
					newElementAttrs.addNode(new LinkedData(attrName), lastDefinedAttr);
				} else {
					newElementAttrs.addNode(new LinkedData(attrName));
				}
			}
		}

		// 生成新的顺序节点，用于保存
		LinkedData firstAttr = newElementAttrs.first();
		LinkedData tmp = firstAttr;
		TagNode newTagNode = new TagNode(curNodeName);
		while(tmp != null) {
			String attrName = tmp.getValue();
			newTagNode.setAttribute(attrName, curElem.getAttribute(attrName));
			tmp = tmp.next;
		}

		//Node cdataNode = getCDataNode(curElem);
		//if (cdataNode!=null && StringUtil.isBlankOrNull(o)) {
		//	newTagNode.addNode(new CDataNode(cdataNode.getNodeValue()));
		//}
		return newTagNode;
	}
	/**
	 * 根据当前配置文件文件中已有节点，生成新的节点
	 * @param document
	 * @param curElem
	 * @param baseFileXmlNode
	 * @return
	 */
	public TagNode createNewNodeByBaseNode(TagNode curElem, TagNode baseTagNode) {
		if (baseTagNode == null) {
			return createNewNodeByDefault(curElem);
		}

		String curNodeName = curElem.getTagName();
		DefaultNode newElementAttrs = new DefaultNode(curNodeName);
		//添加当前dom节点对象中存在， 且文件节点对象中也存在的属性
		Set<String> keySet = baseTagNode.getAttributes().keySet();
		Iterator<String> iter = keySet.iterator();
		while(iter.hasNext()) {
			String attrName = iter.next();
			if (curElem.hasAttribute(attrName)) {
				newElementAttrs.addNode(new LinkedData(attrName));
			}
		}

		//添加当前dom节点对象中存在且默认定义文件中也存在,但当前文件节点对象中不存在的属性
		DefaultNode defaultNode = defualtDefine.getDefaultNode(curNodeName);


		Map<String, String> attributes = curElem.getAttributes();
		attributes.forEach((key, abs) -> {
			if (!newElementAttrs.hasAttribute(key)) {
				if (defaultNode != null && defaultNode.hasAttribute(key)) {
					LinkedData lastDefinedAttr = findLastDefinedAttr(newElementAttrs, defaultNode);
					newElementAttrs.addNode(new LinkedData(key), lastDefinedAttr);
				} else {
					newElementAttrs.addNode(new LinkedData(key));
				}
			}
		});
		// 生成新的顺序节点，用于保存
		LinkedData firstAttr = newElementAttrs.first();
		LinkedData tmp = firstAttr;
		TagNode newTagNode = new TagNode(curNodeName);
		while(tmp != null) {
			String attrName = tmp.getValue();
			newTagNode.setAttribute(attrName, curElem.getAttributes().get(attrName));
			tmp = tmp.next;
		}

		//Node cdataNode = getCDataNode(curElem);
		//if (cdataNode!=null && StringUtil.isBlankOrNull(o)) {
		//	newTagNode.addNode(new CDataNode(cdataNode.getNodeValue()));
		//}
		return newTagNode;
	}

	/**
	 * 根据默认定义文件顺序生成节点
	 *
	 * @param document
	 * @param curElem
	 * @return
	 */
	private TagNode createNewNodeByDefault(Element curElem) {
		String nodeName = curElem.getNodeName();
		TagNode newTagNode = new TagNode(nodeName);
		DefaultNode defaultNode = defualtDefine.getDefaultNode(nodeName);
		if (defaultNode != null) {
			for (LinkedData attrData : defaultNode.values()) {
				String attrName = attrData.getValue();
				if (curElem.hasAttribute(attrName)) {
					newTagNode.setAttribute(attrName, curElem.getAttribute(attrName));
				}
			}
		}

		NamedNodeMap curAttrs = curElem.getAttributes();
		for (int i=0; i<curAttrs.getLength(); i++) {
			String attrName = curAttrs.item(i).getNodeName();
			if (!newTagNode.hasAttribute(attrName)) {
				newTagNode.setAttribute(attrName, curElem.getAttribute(attrName));
			}
		}
		return newTagNode;
	}
	/**
	 * 根据默认定义文件顺序生成节点
	 *
	 * @param document
	 * @param curElem
	 * @return
	 */
	private TagNode createNewNodeByDefault(TagNode curElem) {
		String nodeName = curElem.getTagName();
		TagNode newTagNode = new TagNode(nodeName);
		DefaultNode defaultNode = defualtDefine.getDefaultNode(nodeName);
		if (defaultNode != null) {
			for (LinkedData attrData : defaultNode.values()) {
				String attrName = attrData.getValue();
				if (curElem.hasAttribute(attrName)) {
					newTagNode.setAttribute(attrName, curElem.getAttributes().get(attrName));
				}
			}
		}
		Map<String, String> attributes = curElem.getAttributes();
		attributes.forEach((key,abs)->{
			if (!newTagNode.hasAttribute(key)) {
				newTagNode.setAttribute(key, abs);
			}
		});
		return newTagNode;
	}

	private LinkedData findLastDefinedAttr(DefaultNode linkAttrs, DefaultNode defaultAttrs) {
		LinkedData tmpData = linkAttrs.last();
		while (tmpData != null) {
			if(defaultAttrs.hasAttribute(tmpData.getValue())) {
				return tmpData;
			}
			tmpData = tmpData.previous;
		}
		return null;
	}

	/**
	 * 去节点cdata内容，如果其中有其他Element节点，则认为不存在cdata
	 * @param elem
	 * @return
	 */
	private Node getCDataNode(Element elem) {
		Node node = null;
		NodeList nodeList = elem.getChildNodes();
		for(int n=0; n<nodeList.getLength(); n++) {
			node = nodeList.item(n);
			if (Element.CDATA_SECTION_NODE == node.getNodeType()) {
				return node;
			} else if (Element.ELEMENT_NODE == node.getNodeType()) {
				return null;
			}
		}
		return null;
	}

	private String getKey(Element elem) {
		String tagName = elem.getTagName();
		String primaryKey = defualtDefine.getPrimaryKey(tagName);
		String attrKey = elem.getAttribute(primaryKey);
		Node parentNode = elem.getParentNode();
		String parentPrimaryKey = "";
		parentPrimaryKey = getParentPrimaryKey(parentNode, parentPrimaryKey);
		return parentPrimaryKey.concat("->").concat(tagName).concat("@").concat(attrKey);
	}

	/**
	 * 获取父节点Key
	 * @param parentNode
	 * @param parentPrimaryKey
	 * @return
	 */
	private String getParentPrimaryKey(Node parentNode, String parentPrimaryKey) {
		if (parentNode != null && parentNode instanceof  Element) {
			Element parentNode1 = (Element) parentNode;
			String tagName1 = parentNode1.getTagName();
			String key = parentNode1.getAttribute("Key");
			if (StringUtils.isEmpty(key)){
				parentPrimaryKey = parentPrimaryKey.concat(getParentPrimaryKey(parentNode1.getParentNode(), parentPrimaryKey));
			}else {
				parentPrimaryKey = tagName1.concat("@").concat(key);
			}
		}
		return parentPrimaryKey;
	}
}
