import { ENodeType, ETokenType } from "../enum/Enums";
import Var from "../struct/Var";
import ErrorUtil from "../util/ErrorUtil";
import StringBuilder from "../util/StringBuilder";
import StringUtil from "../util/StringUtil";
import IXmlEntity from "./IXmlEntity";
import IXmlLexicalAnalysisListener from "./IXmlLexicalAnalysisListener";
import INode from "./node/INode";
import INodeFactory from "./node/INodeFactory";
import ITagNode from "./node/ITagNode";
import Token from "./Token";
import XmlUtil from "./util/XmlUtil";
import XmlLexicalAnalysis from "./XmlLexicalAnalysis";
/*
XML --> XMLNote + XMLNode

XMLNote --> HEAD_START  + XMLAttrs + HEAD_END

XMLNode - > LAB + TEXT + XMLAttrs + RAB + [XMLNode | &  | CDATA] + TagTail;

XMLNode - > LAB + TEXT + XMLAttrs + SRAB

TagTail --> LABS + TEXT + RAB

XMLAttrs -> XMLAttr + XMLAttrs

XMLAttr --> TEXT + EQ + STRING

CDATA -> CDATA_START + CDATA_END

COMMENT -> COMMENT_START + COMMENT_END

*/
/**
 * xml语法分析
 * 
 * @author: chenbb
 */
export default class XmlParser {
	
	private position: number = 0;
	
	private total: number = 0;
	
	private tokens: Token[] = [];
	
	private root!: ITagNode;
	
	private tagStack: ITagNode[] = [];
	
	private nodeFactory!: INodeFactory; 

	public XmlParser() {
	}

	public parse(content: string, toEntity: IXmlEntity, lexiAnalysisListener?: IXmlLexicalAnalysisListener): void {
		this.nodeFactory = toEntity.getNodeFactory();
		this.root = toEntity.getRoot();
		// 词法分析
		let lexicalAnalysis = new XmlLexicalAnalysis(content);
		lexicalAnalysis.setAnalysisListener(lexiAnalysisListener);
		this.tokens = lexicalAnalysis.analysis();
		this.position = 0;
		this.total = this.tokens.length;
		// 语法分析
		this.start(this.root);

		toEntity.getTagRoot().resetLineIndex(new Var(0));
	}

	private start(node: ITagNode): void {
		let curToken: Token = this.token();
		while (this.position < this.total && curToken != Token.TK_END) {
			switch (curToken.getType()) {
			case ETokenType.HEAD_START:
				this.head(node);
				break;
			case ETokenType.LAB:
				this.node(node);
				break;
			case ETokenType.COMMENT_START:
				this.comment(node);
				break;
			case ETokenType.CDATA_START:
				this.cdata(node);
				break;
			case ETokenType.TEXT:
				this.text(node);
				break;
			case ETokenType.LABS:
				this.labs();
				break;
			default:
				ErrorUtil.throwError("Unknow Token Type: " + curToken.getType());
				break;
			}
			curToken = this.nextToken();
		}
	}

	private text(node: ITagNode): void {
		let token = this.token();
		let textNode = this.nodeFactory.createNode(ENodeType.TEXT, token.getValue());
		node.addChild(textNode);
	}
	
	/**
	 * 左尖括号带斜杠  </
	 * @throws Throwable 
	 */
	private labs(): void {
		let startTagNode = this.tagStack.pop();
		var startTag = startTagNode ? startTagNode.getTagName() : "";
		var textToken = this.nextToken();
		if (textToken.getValue() != startTag) {
			ErrorUtil.throwError("unexpected tag end:" +  textToken.getValue() + " with " + startTag);
		}
		//跳过一个右括号的位置
		this.position ++;
		
		var sizeOfStack = this.tagStack.length;
		if (sizeOfStack != 0) {
			this.position ++;
			var curTop = this.tagStack[sizeOfStack - 1];
			this.start(curTop);
		}
	}

	private node(parentNode: ITagNode): void {
		let token = this.nextToken();
		let node: ITagNode = this.nodeFactory.createTagNode(token.getValue());
		parentNode.addChild(node);

		token = this.nextToken();
		while (token.getType() != ETokenType.RAB && token.getType() != ETokenType.SRAB) {
			this.attr(node);
			token = this.nextToken();
		}
		
		if (token.getType() == ETokenType.SRAB) {
			return;
		} else if (token.getType() == ETokenType.RAB) {
			this.tagStack.push(node);
			this.position ++;
			this.start(node);
		}
	}
	
	private attr(tagNode: ITagNode): void {
		let textToken = this.token();
		// 跳过 EQ
		this.position ++;
		let strToken = this.nextToken();
		var attrValue = strToken.getValue();
		tagNode.addAttribute(textToken.getValue(), XmlUtil.escape(attrValue));
	}

	private comment(parentNode: ITagNode): void {
		let token1 = this.token();
		let token2 = this.nextToken();
		if (token2.getType() != ETokenType.COMMENT_END) {
			ErrorUtil.throwError("unexpected comment end!");
		}
		
		var node: INode = this.nodeFactory.createNode(ENodeType.COMMENT, token1.getValue() + token2.getValue());
		if (node.getNodeType() != ENodeType.EMPTY) {
			parentNode.addChild(node);
		}
	}
	
	private cdata(parentNode: ITagNode): void {
		var token1 = this.token();
		var token2 = this.nextToken();
		if (token2.getType() != ETokenType.CDATA_END) {
			ErrorUtil.throwError("unexpected cdata end!");
		}
		var node = this.nodeFactory.createNode(ENodeType.CDATA, token1.getValue() + token2.getValue());
		parentNode.addChild(node);
	}
	
	private head(parentNode: ITagNode): void {
		let sb = new StringBuilder();
		sb.append(this.token().getValue()).append(" ");
		var token = this.nextToken();
		while (token.getType() != ETokenType.HEAD_END) {
			sb.append(token.getValue());
			this.position ++;
			sb.append("=");
			token = this.nextToken();
			sb.append("\"").append(token.getValue()).append("\"").append(StringUtil.STR_SPACE);
			token = this.nextToken();
		}
		
		sb.deleteLastChar();
		
		if (this.tokens[this.position].getType() != ETokenType.HEAD_END) {
			ErrorUtil.throwError("can not find HEAD_END");
		}
		sb.append(token.getValue());

		var headNode = this.nodeFactory.createNode(ENodeType.HEAD, sb.toString());
		parentNode.addChild(headNode);
	}
	
	private token(): Token {
		return this.tokens[this.position];
	}
	
	private nextToken(): Token {
		this.position ++;
		if (this.position < this.total) {
			return this.token();
		}
		return Token.TK_END;
	}
}

