package com.bokesoft.yes.design.xml;

import java.util.*;

import com.bokesoft.yes.design.Property;
import com.bokesoft.yes.meta.persist.dom.form.MetaConstants;
import com.bokesoft.yigo.meta.form.component.control.MetaTextEditor;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.DefaultNode;
import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.DefaultNodeDefine;
import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.LinkedData;
import com.bokesoft.yes.meta.persist.dom.xml.node.AbstractNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.CDataNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.TagNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.TextNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.XmlTree;

public class XmlCreator {

	private final Document doc;

	private final XmlTree orgTree;

	private final LinkedHashMap<String, TagNode> mapNode = new LinkedHashMap<>();

	private final DefaultNodeDefine defaultDefine = DefaultNodeDefine.getInstance();

	public XmlCreator(Document doc, XmlTree orgTree) {
		this.doc = doc;
		this.orgTree = orgTree;
	}

	public XmlTree createXmlTree() {
		mapNode.clear();

		Element elem = doc.getDocumentElement();
		TagNode newRoot;
		if (Objects.nonNull(orgTree)) {
			String key = this.getKey(elem);
			TagNode node = orgTree.getTagNode(key);
			newRoot = this.createNewNodeByBaseNode(elem, node);
		} else {
			newRoot = this.createNewNodeByDefault(elem);
		}

		this.createTagNode(elem, newRoot, true);

		if (Objects.nonNull(orgTree)) {
			final TagNode root = orgTree.getRoot();
			newRoot.addPreComment(root.getPreComment());
			newRoot.addLastComment(root.getLastComment());
		}

		XmlTree newTree = new XmlTree();
		newTree.setRoot(newRoot);
		return newTree;
	}

	public String createXml() {
		XmlTree xmlTree = createXmlTree();
		createComments(xmlTree);
		return xmlTree.getRoot().toXml(0);
	}

	public HashMap<String, TagNode> getTagNodeMap() {
		return this.mapNode;
	}

	private void createComments(XmlTree xmlTree) {
		if (Objects.isNull(orgTree)) {
			return;
		}
		TagNode root = xmlTree.getRoot();
		TagNode virtual = new TagNode();
		virtual.addNode(root);
		List<AbstractNode> listNode = orgTree.getNodesWithComment();
		for (AbstractNode nodeWithComment : listNode) {
			Stack<AbstractNode> stackPath  = getCommentPath(nodeWithComment);
			TagNode levelNode = virtual;
			boolean bFind = false;
			AbstractNode pathNode = null;
			while(!stackPath.isEmpty() && levelNode != null) {
				pathNode = stackPath.pop();
				if (pathNode instanceof TagNode) {
					TagNode tagNode = (TagNode) pathNode;
					levelNode = levelNode.findChildByTagNode(tagNode);
					bFind = (levelNode != null);
				} else {
					bFind = false;
					break;
				}
			}
			if (bFind) {
				levelNode.addPreComment(pathNode.getPreComment());
				levelNode.addLastComment(pathNode.getLastComment());
			}
		}
	}

	private Stack<AbstractNode> getCommentPath(AbstractNode nodeWithComment) {
		Stack<AbstractNode> stack = new Stack<>();
		AbstractNode parent = nodeWithComment;
		while (parent != null) {
			stack.push(parent);
			parent = parent.getParent();
		}
		return stack;
	}

	private void createTagNode(Element elem, TagNode parentTagNode, boolean isRoot) {
		TagNode newTagNode;
		if (isRoot) {
			newTagNode = parentTagNode;
		} else {
			if (Objects.nonNull(orgTree)) {
				String key = getKey(elem);
				TagNode node = orgTree.getTagNode(key);
				newTagNode = createNewNodeByBaseNode(elem, node);
			} else {
				newTagNode = createNewNodeByDefault(elem);
			}
			parentTagNode.addNode(newTagNode);
			String par="";
			if (newTagNode.getParent()!=null){
				par = newTagNode.getParent().getPrimaryKey();
			}
			mapNode.put(par+"->"+newTagNode.getPrimaryKey(), newTagNode);
		}

		NodeList nodeList = elem.getChildNodes();
		for (int n = 0; n < nodeList.getLength(); n++) {
			Node child = nodeList.item(n);
			if (child.getNodeType() == Element.ELEMENT_NODE) {
				createTagNode((Element) child, newTagNode, false);
			}
			if (child.getNodeType() == Element.CDATA_SECTION_NODE) {
				newTagNode.addNode(new CDataNode(child.getNodeValue()));
			}
			if (child.getNodeType() == Element.TEXT_NODE) {
				String sContent = child.getTextContent();
				if (sContent != null && sContent.trim().length() > 0) { // TODO 这里不敢改
					newTagNode.addNode(new TextNode(sContent));
				}
			}
		}
	}

	private TagNode createNewNodeByBaseNode(Element curElem, TagNode baseTagNode) {
		if (curElem.hasAttribute(Property.CssClass.name)) {
			curElem.removeAttribute(MetaConstants.COMPONENT_CLASS);
		}

		if (Objects.isNull(baseTagNode)) {
			return createNewNodeByDefault(curElem);
		}
		String curNodeName = curElem.getNodeName();
		DefaultNode newElementAttrs = new DefaultNode(curNodeName);
		//添加当前dom节点对象中存在， 且文件节点对象中也存在的属性
		Set<String> keySet = baseTagNode.getAttributes().keySet();
		for (String attrName : keySet) {
			if (curElem.hasAttribute(attrName)) {
				newElementAttrs.addNode(new LinkedData(attrName));
			}
		}

		//添加当前dom节点对象中存在且默认定义文件中也存在,但当前文件节点对象中不存在的属性
		DefaultNode defaultNode = defaultDefine.getDefaultNode(curNodeName);
		NamedNodeMap curAttrs = curElem.getAttributes();
		for (int i=0; i<curAttrs.getLength(); i++) {
			String attrName = curAttrs.item(i).getNodeName();
			if (!newElementAttrs.hasAttribute(attrName) ) {
				if(defaultNode != null && defaultNode.hasAttribute(attrName)) {
					LinkedData lastDefinedAttr = findLastDefinedAttr(newElementAttrs, defaultNode);
					newElementAttrs.addNode(new LinkedData(attrName), lastDefinedAttr);
				} else {
					newElementAttrs.addNode(new LinkedData(attrName));
				}
			}
		}

		// 生成新的顺序节点，用于保存
		LinkedData tmp = newElementAttrs.first();
		TagNode newTagNode = new TagNode(curNodeName);
		while(tmp != null) {
			String attrName = tmp.getValue();
			newTagNode.setAttribute(attrName, curElem.getAttribute(attrName));
			tmp = tmp.next;
		}
		return newTagNode;
	}
	public TagNode createNewNodeByBaseNode(TagNode curElem, TagNode baseTagNode) {
		if (Objects.isNull(baseTagNode)) {
			return createNewNodeByDefault(curElem);
		}

		String curNodeName = curElem.getTagName();
		DefaultNode newElementAttrs = new DefaultNode(curNodeName);
		//添加当前dom节点对象中存在， 且文件节点对象中也存在的属性
		Set<String> keySet = baseTagNode.getAttributes().keySet();
		for (String attrName : keySet) {
			if (curElem.hasAttribute(attrName)) {
				newElementAttrs.addNode(new LinkedData(attrName));
			}
		}
		//添加当前dom节点对象中存在且默认定义文件中也存在,但当前文件节点对象中不存在的属性
		DefaultNode defaultNode = defaultDefine.getDefaultNode(curNodeName);
		Map<String, String> attributes = curElem.getAttributes();
		attributes.forEach((key, abs) -> {
			if (!newElementAttrs.hasAttribute(key)) {
				if (defaultNode != null && defaultNode.hasAttribute(key)) {
					LinkedData lastDefinedAttr = findLastDefinedAttr(newElementAttrs, defaultNode);
					newElementAttrs.addNode(new LinkedData(key), lastDefinedAttr);
				} else {
					newElementAttrs.addNode(new LinkedData(key));
				}
			}
		});
		// 生成新的顺序节点，用于保存
		LinkedData tmp = newElementAttrs.first();
		TagNode newTagNode = new TagNode(curNodeName);
		while(tmp != null) {
			String attrName = tmp.getValue();
			newTagNode.setAttribute(attrName, curElem.getAttributes().get(attrName));
			tmp = tmp.next;
		}
		return newTagNode;
	}

	private TagNode createNewNodeByDefault(Element curElem) {
		String nodeName = curElem.getNodeName();
		TagNode newTagNode = new TagNode(nodeName);
		DefaultNode defaultNode = defaultDefine.getDefaultNode(nodeName);
		if (defaultNode != null) {
			for (LinkedData attrData : defaultNode.values()) {
				String attrName = attrData.getValue();
				if (curElem.hasAttribute(attrName)) {
					newTagNode.setAttribute(attrName, curElem.getAttribute(attrName));
				}
			}
		}

		NamedNodeMap curAttrs = curElem.getAttributes();
		for (int i=0; i<curAttrs.getLength(); i++) {
			String attrName = curAttrs.item(i).getNodeName();
			if (!newTagNode.hasAttribute(attrName)) {
				newTagNode.setAttribute(attrName, curElem.getAttribute(attrName));
			}
		}
		return newTagNode;
	}
	private TagNode createNewNodeByDefault(TagNode curElem) {
		String nodeName = curElem.getTagName();
		TagNode newTagNode = new TagNode(nodeName);
		DefaultNode defaultNode = defaultDefine.getDefaultNode(nodeName);
		if (defaultNode != null) {
			for (LinkedData attrData : defaultNode.values()) {
				String attrName = attrData.getValue();
				if (curElem.hasAttribute(attrName)) {
					newTagNode.setAttribute(attrName, curElem.getAttributes().get(attrName));
				}
			}
		}
		Map<String, String> attributes = curElem.getAttributes();
		attributes.forEach((key,abs)->{
			if (!newTagNode.hasAttribute(key)) {
				newTagNode.setAttribute(key, abs);
			}
		});
		return newTagNode;
	}

	private LinkedData findLastDefinedAttr(DefaultNode linkAttrs, DefaultNode defaultAttrs) {
		LinkedData tmpData = linkAttrs.last();
		while (tmpData != null) {
			if(defaultAttrs.hasAttribute(tmpData.getValue())) {
				return tmpData;
			}
			tmpData = tmpData.previous;
		}
		return null;
	}

	private String getKey(Element elem) {
		String tagName = elem.getTagName();
		String primaryKey = defaultDefine.getPrimaryKey(tagName);
		String attrKey = elem.getAttribute(primaryKey);
		Node parentNode = elem.getParentNode();
		String parentPrimaryKey = "";
		parentPrimaryKey = getParentPrimaryKey(parentNode, parentPrimaryKey);
		return parentPrimaryKey.concat("->").concat(tagName).concat("@").concat(attrKey);
	}

	private String getParentPrimaryKey(Node parentNode, String parentPrimaryKey) {
		if (parentNode instanceof Element) {
			Element parentNode1 = (Element) parentNode;
			String tagName1 = parentNode1.getTagName();
			String key = parentNode1.getAttribute("Key");
			if (StringUtils.isEmpty(key)){
				parentPrimaryKey = parentPrimaryKey.concat(getParentPrimaryKey(parentNode1.getParentNode(), parentPrimaryKey));
			}else {
				parentPrimaryKey = tagName1.concat("@").concat(key);
			}
		}
		return parentPrimaryKey;
	}
}
