package com.bokesoft.yes.design.search;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 正则
 */
public class PatternConstructor {

	private PatternConstructor() {

	}

	/**
	 * 根据条件创建正则模式
	 * 
	 * @param pattern
	 * @param isRegex
	 * @param isCaseSensitive
	 * @param isWholeWord
	 * @return
	 * @throws PatternSyntaxException
	 */
	public static Pattern createPattern(String pattern, boolean isRegex, boolean isCaseSensitive, boolean isWholeWord)
			throws PatternSyntaxException {
		if (pattern == null || pattern.length() == 0) {
			throw new RuntimeException("Cannot create regex from empty string");
		}

		if (!isRegex) {
			pattern = escapeRegExpCharacters(pattern);
		}

		if (isWholeWord) {
			int len = pattern.length();

			if (isWordChar(pattern.charAt(0))) {
				pattern = "\\b" + pattern;
			}

			if (isWordChar(pattern.charAt(len - 1))) {
				pattern = pattern + "\\b";
			}
		}

		int regexOptions = Pattern.MULTILINE;

		if (!isCaseSensitive) {
			regexOptions |= Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE;
		}

		return Pattern.compile(pattern, regexOptions);
	}

	/**
	 * 创建pattern
	 * 
	 * @param pattern         搜索字符串
	 * @param isRegex         字符串是正则格式
	 * @param isStringMatcher
	 * @param isCaseSensitive 区分大小写
	 * @param isWholeWord     整词匹配
	 * @return
	 */
	public static Pattern createPattern(String pattern, boolean isRegex, boolean isStringMatcher,
			boolean isCaseSensitive, boolean isWholeWord) throws PatternSyntaxException {
		if (isRegex) {
			if (isWholeWord) {
				throw new RuntimeException("isWholeWord unsupported together with isRegex");
			} else {
				pattern = substituteLinebreak(pattern);
			}

		} else {
			int len = pattern.length();
			StringBuilder buffer = new StringBuilder(len + 10);

			if (isWholeWord && len > 0 && isWordChar(pattern.charAt(0))) {
				buffer.append("\\b");
			}

			appendAsRegEx(isStringMatcher, pattern, buffer);

			if (isWholeWord && len > 0 && isWordChar(pattern.charAt(len - 1))) {
				buffer.append("\\b");
			}

			pattern = buffer.toString();
		}

		int regexOptions = Pattern.MULTILINE;

		if (!isCaseSensitive) {
			regexOptions |= Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE;
		}

		return Pattern.compile(pattern, regexOptions);
	}

	/**
	 * 替换换行
	 * 
	 * @param findString
	 * @return
	 * @throws PatternSyntaxException
	 */
	private static String substituteLinebreak(String findString) throws PatternSyntaxException {
		int length = findString.length();
		StringBuilder buf = new StringBuilder(length);

		int inCharGroup = 0;
		int inBraces = 0;
		boolean inQuote = false;
		for (int i = 0; i < length; i++) {
			char ch = findString.charAt(i);
			switch (ch) {
			case '[':
				buf.append(ch);
				if (!inQuote)
					inCharGroup++;
				break;

			case ']':
				buf.append(ch);
				if (!inQuote)
					inCharGroup--;
				break;

			case '{':
				buf.append(ch);
				if (!inQuote && inCharGroup == 0)
					inBraces++;
				break;

			case '}':
				buf.append(ch);
				if (!inQuote && inCharGroup == 0)
					inBraces--;
				break;

			case '\\':
				if (i + 1 < length) {
					char ch1 = findString.charAt(i + 1);
					if (inQuote) {
						if (ch1 == 'E')
							inQuote = false;
						buf.append(ch).append(ch1);
						i++;

					} else if (ch1 == 'R') {
						if (inCharGroup > 0 || inBraces > 0) {
							throw new PatternSyntaxException("Illegal position for \\\\R", findString, i);
						}
						buf.append("(?>\\r\\n?|\\n)");
						i++;

					} else {
						if (ch1 == 'Q') {
							inQuote = true;
						}
						buf.append(ch).append(ch1);
						i++;
					}
				} else {
					buf.append(ch);
				}
				break;

			default:
				buf.append(ch);
				break;
			}

		}
		return buf.toString();
	}

	/**
	 * 字母或数字
	 * 
	 * @return
	 */
	private static boolean isWordChar(char c) {
		return Character.isLetterOrDigit(c);
	}

	public static String escapeRegExpCharacters(String pattern) {
		int len = pattern.length();
		StringBuilder buffer = new StringBuilder(len + 10);

		boolean isEscaped = false;

		for (int i = 0; i < pattern.length(); i++) {
			char c = pattern.charAt(i);
			switch (c) {
			case '\\':
				if (!isEscaped) {
					isEscaped = true;
				} else {
					buffer.append("\\\\");
					isEscaped = false;
				}

				break;
			case '(':
			case ')':
			case '{':
			case '}':
			case '.':
			case '[':
			case ']':
			case '$':
			case '^':
			case '+':
			case '|':
				if (isEscaped) {
					buffer.append("\\\\");
					isEscaped = false;
				}
				buffer.append("\\");
				buffer.append(c);
				break;
			case '?':
				if (!isEscaped) {
					buffer.append('.');
				} else {
					buffer.append('\\');
					buffer.append(c);
					isEscaped = false;
				}
				break;
			case '*':
				if (!isEscaped) {
					buffer.append(".*");
				} else {
					buffer.append('\\');
					buffer.append(c);
					isEscaped = false;
				}
				break;
			default:
				if (isEscaped) {
					buffer.append("\\\\");
					isEscaped = false;
				}
				buffer.append(c);
				break;
			}
		}
		if (isEscaped) {
			buffer.append("\\\\");
			isEscaped = false;
		}

		return buffer.toString();
	}

	/**
	 * 
	 * 
	 * @return
	 */
	public static StringBuilder appendAsRegEx(boolean isStringMatcher, String pattern, StringBuilder buffer) {
		boolean isEscaped = false;

		for (int i = 0; i < pattern.length(); i++) {
			char c = pattern.charAt(i);
			switch (c) {
			case '\\':
				if (isStringMatcher && !isEscaped) {
					isEscaped = true;
				} else {
					buffer.append("\\\\");
					isEscaped = false;
				}

				break;
			case '(':
			case ')':
			case '{':
			case '}':
			case '.':
			case '[':
			case ']':
			case '$':
			case '^':
			case '+':
			case '|':
				if (isEscaped) {
					buffer.append("\\\\");
					isEscaped = false;
				}
				buffer.append("\\");
				buffer.append(c);
				break;
			case '?':
				if (isStringMatcher && !isEscaped) {
					buffer.append('.');
				} else {
					buffer.append('\\');
					buffer.append(c);
					isEscaped = false;
				}
				break;
			case '*':
				if (isStringMatcher && !isEscaped) {
					buffer.append(".*");
				} else {
					buffer.append('\\');
					buffer.append(c);
					isEscaped = false;
				}
				break;
			default:
				if (isEscaped) {
					buffer.append("\\\\");
					isEscaped = false;
				}
				buffer.append(c);
				break;
			}
		}
		if (isEscaped) {
			buffer.append("\\\\");
			isEscaped = false;
		}

		return buffer;
	}
}
