package com.bokesoft.yes.design.bpm.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bokesoft.erp.WebDesignerConfiguration;
import com.bokesoft.yes.common.util.DateUtil;
import com.bokesoft.yes.design.XmlTreeWithPath;
import com.bokesoft.yes.design.bpm.po.Path;
import com.bokesoft.yes.design.cmd.XmlFileProcessor;
import com.bokesoft.yes.design.enums.NodeKeyAndCaptionEnum;
import com.bokesoft.yes.design.io.LoadFileTree;
import com.bokesoft.yes.design.constant.ConstantUtil;
import com.bokesoft.yes.design.utils.DesignReloadMetaObject;
import com.bokesoft.yes.design.utils.XMLWriter;
import com.bokesoft.yes.design.vo.JsonQuestVo;
import com.bokesoft.yes.design.vo.ResponseResult;
import com.bokesoft.yes.helper.DataObjectHelper;
import com.bokesoft.yes.helper.FilePathHelper;
import com.bokesoft.yes.meta.persist.dom.bpm.MetaProcessSave;
import com.bokesoft.yes.design.xml.node.TagNode;
import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.bpm.process.MetaProcess;
import com.bokesoft.yigo.meta.bpm.process.ProcessDefinitionProfile;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.solution.MetaProject;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.apache.commons.collections.CollectionUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Method;
import java.nio.file.Paths;
import java.util.*;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import static java.lang.Class.forName;

/***
 * bpm转换为xml文件操作工具类
 * @date 2020-09-25
 * @author zhsy
 */
public class BpmOperToXmlUtil {

	private static final Logger logger = Logger.getLogger(BpmOperToXmlUtil.class.getName());
	/***BPM目录*/
	private static final String BPM_DIRECTORY = "BPM";
	/***xml扩展名*/
	private static final String XML_EXTENSION = ".xml";
	/***删除操作类型OPER_TYPE_THREE*/
	public static final int OPER_TYPE_THREE = 3;

	/***
	 * 新增xml文件
	 * @param projectKey 项目key,例如:hrconfig,mmconfig等
	 * @param key 工作流key
	 * @param caption 工作流描述
	 * @param version 版本号
	 * @Return {"result":true或false,"filePath":""}
	 */
	public Map<String, String> createXml(String projectKey, String key, String caption, String version) {
		Map<String, String> result = new HashMap<>(10);
		result.put("result", "true");
		String filePath = "";
		try {
			// 校验
			if (validateForCreateXml(projectKey, key, caption, version, result)) {
				return result;
			}

			int verionNo = Integer.parseInt(version) + 1;
			if (key.contains("_V")) {
				key = key.substring(0, key.indexOf("_V")) + "_V" + verionNo;
			} else {
				if (verionNo > 1) {
					key = key + "_V" + verionNo;
				}
			}

			if (LoadFileTree.isExistKey(key)) {
				result.put("result", "false");
				result.put("filePath", "");
				result.put("msg", "新增的版本V" + verionNo + "已存在，不能重复新增版本");
				return result;
			}
			MetaProcess process = new MetaProcess();
			process.setKey(key);
			process.setCaption(caption);
			process.setVersion(verionNo);
			IMetaFactory globalInstance = MetaFactory.getGlobalInstance();
			IMetaResolver resolver = globalInstance.getProjectResolver(projectKey);
			String newFileName = key + XML_EXTENSION;
			// 保存xml文件到磁盘
			MetaProcessSave metaProcessSave = new MetaProcessSave(process);
			metaProcessSave.save(resolver, File.separator + BPM_DIRECTORY + File.separator + newFileName);
			String versionStr = "0".equals(version) ? "" : " Version=\"" + verionNo + "\"";
			// 调用新增xml文件方法
			filePath = LoadFileTree.newBpmFile(projectKey, newFileName, key, caption);
			String fileContent = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n" +
					"<Process Caption=\"" + caption + "\" Key=\"" + key + "\"" + versionStr + ">\n" +
					"    <SwimlineCollection/>\n" +
					"    <BPMInfoCollection/>\n" +
					"    <DMTable/>\n" +
					"    <PermCollection/>\n" +
					"    <AttachmentCollection/>\n" +
					"</Process>";
			XmlFileProcessor.instance.saveTempFile(filePath, fileContent,projectKey);

			result.put("result", "true");
			result.put("filePath", filePath);
			result.put("version", String.valueOf(verionNo));
			result.put("fileName", newFileName.replaceAll(".xml", ""));
			//处理bpm实体
			ProcessDefinitionProfile formProfile = new ProcessDefinitionProfile();
			HashMap<String, ProcessDefinitionProfile> profileMap = globalInstance.getMetaBPM().getProfileMap();
			MetaProject metaProject = globalInstance.getMetaProject(projectKey);
			formProfile.setKey(key);
			formProfile.setCaption(caption);
			formProfile.setVersion(verionNo);
			formProfile.setResource(projectKey + "/BPM/" + key + ".xml");
			formProfile.setProject(metaProject);
			profileMap.put(key, formProfile);
		} catch (Throwable e) {
			logger.warning("新增xml文件异常,异常为:" + ExceptionUtils.getStackTrace(e));
			if (StringUtils.isNotEmpty(filePath)) {
				try {
					LoadFileTree.removeFilePath(filePath);
				} catch (Throwable e1) {
					logger.warning("删除文件异常,异常为:" + ExceptionUtils.getStackTrace(e1));
				}
			}
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", e.getMessage());
		}
		return result;
	}

	/***
	 * 创建xml文件校验
	 * @param projectKey 项目标识，如：hrconfig
	 * @param key 标识
	 * @param caption 名称
	 * @param version 版本
	 * @param result 校验结果
	 * @return true:校验失败；false:校验成功
	 */
	private boolean validateForCreateXml(String projectKey, String key, String caption, String version,
										 Map<String, String> result) {
		// 非空校验
		if (StringUtils.isEmpty(projectKey)) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "项目不能为空");
			return true;
		}
		if (StringUtils.isEmpty(key)) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "标识不能为空");
			return true;
		}
		if (StringUtils.isEmpty(caption)) {
			result.put("result", "false");
			result.put("filePath", "名称不能为空");
			result.put("msg", "");
			return true;
		}

		String pattern = "([A-Za-z_]\\w*)";
		boolean isMatch = Pattern.matches(pattern, key);
		if (!isMatch) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "标识不正确，不是字母或下划线开始的");
			return true;
		}

		// 如果是新增文件，并且文件已经存在，直接返回错误信息
		if (LoadFileTree.isExistKey(key) && "0".equals(version)) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "文件'" + key + "'已存在");
			return true;
		}
		return false;
	}

	/***
	 * 更新xml文件
	 * @param jsonQuestVo 前端传过来的数据对象
	 * @return 新增结果
	 */
	public ResponseResult<JSONObject> updateXml(JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> result = new ResponseResult<>();
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		try {
			// xml文件路径
			String filePath = jsonQuestVo.getFilePath();
			String tempPath = XmlFileProcessor.instance.getTmpFile(filePath);
			if (StringUtils.isBlank(tempPath)) {
				tempPath = filePath;
			}
			// 前端传过来的要更新的json
			JSONObject frontUpdateJson = JSON.parseObject(jsonQuestVo.getContent());
			// 要更新键值
			String key = frontUpdateJson.getString("key");
			// 更新后的值
			String value = frontUpdateJson.getString("value");

			// 创建SAX读取器
			SAXReader reader = new SAXReader();
			// 加载文档
			Document document = reader.read(new File(tempPath));
			// 获取根节点
			Element root = document.getRootElement();
			switch (key) {
				case "DMTable":
					if (null != root.element(key)) {
						root.remove(root.element(key));
					}
					if (!"null".equals(value) && StringUtils.isNotEmpty(value)) {
						root.addElement(key);
						JSONArray array = JSON.parseArray(value);
						if (!CollectionUtils.isEmpty(array)) {
							JSONObject obj;
							Element field;
							String type;
							String detail;
							for (int i = 0; i < array.size(); i++) {
								obj = array.getJSONObject(i);
								field = root.element(key).addElement("Field");
								if (StringUtils.isEmpty(obj.getString(ConstantUtil.KEY))) {
									result.setCode(1);
									result.setMsg("第" + (i + 1) + "行'迁移字段的标识'为空，不能进行操作");
									return result;
								}
								field.addAttribute(ConstantUtil.KEY, obj.getString(ConstantUtil.KEY));
								field.addAttribute(ConstantUtil.CAPTION, obj.getString(ConstantUtil.CAPTION));
								type = obj.getString(ConstantUtil.TYPE);
								if ("Field".equals(type)) {
									if (StringUtils.isEmpty(obj.getString("Value"))) {
										result.setCode(1);
										result.setMsg("第" + (i + 1) + "行'来源字段'为空，不能进行操作");
										return result;
									}
									field.addAttribute("SourceFieldKey", obj.getString("Value"));
								} else if ("Const".equals(type)) {
									field.addAttribute(ConstantUtil.TYPE, type);
									field.addAttribute("ConstValue", obj.getString("Value"));
								}
								detail = obj.getString("Detail");
								field.addAttribute("Detail", detail.replaceAll("\"", "'"));
							}
						}
					}
					break;
				case ConstantUtil.FORM_KEY:
				case ConstantUtil.CAPTION:
				case "IgnoreFormState":
				case "KillInstanceTrigger":
				case "TemplateKey":
					if (StringUtils.isNotEmpty(value)) {
						root.addAttribute(key, value);
					} else {
						deleteXmlElementAttribute(root, root.attribute(key));
					}
					break;
				case "QueryRetreatWorkitem":
					if ("false".equals(value)) {
						root.addAttribute(key, value);
					} else {
						deleteXmlElementAttribute(root, root.attribute(key));
					}
					break;
				case "LockWorkitem":
					if ("true".equals(value)) {
						root.addAttribute(key, value);
					} else {
						deleteXmlElementAttribute(root, root.attribute(key));
					}
					break;
				case "PermCollection":
					if (null != root.element(key)) {
						root.remove(root.element(key));
					}
					if (StringUtils.isNotEmpty(value)) {
						JSONObject permJson = JSON.parseObject(value);
						if (null != permJson) {
							JSONArray optPermArray = permJson.getJSONArray("OptPerm");
							JSONArray visiblePermArray = permJson.getJSONArray("VisiblePerm");
							JSONArray enablePermArray = permJson.getJSONArray("EnablePerm");
							boolean hasPerm = !CollectionUtils.isEmpty(optPermArray) ||
									!CollectionUtils.isEmpty(visiblePermArray) ||
									!CollectionUtils.isEmpty(enablePermArray);
							if (hasPerm) {
								Element permCollection = root.addElement(key);
								permCollection.addElement("Perm");
								List<String> keys = Arrays.asList("OptPerm", "VisiblePerm", "EnablePerm");
								keys.forEach(q -> setXmlPerm(permJson, permCollection, q,
										q + "Item"));
							}
						}
					}
					break;
				case "BPMGraphInfoPath":
					if (StringUtils.isNotEmpty(value)) {
						if (null == root.element("Expand")) {
							root.addElement("Expand");
						}
						root.element("Expand").addAttribute(key, value);
					} else {
						if (null != root.element("Expand")) {
							if (null == root.element("Expand").attribute("WorkFlowHookPath")) {
								root.remove(root.element("Expand"));
							} else {
								deleteXmlElementAttribute(root.element("Expand"),
										root.element("Expand").attribute(key));
							}
						}
					}
					break;
				case "WorkFlowHookPath":
					if (StringUtils.isNotEmpty(value)) {
						if (null == root.element("Expand")) {
							root.addElement("Expand");
						}
						root.element("Expand").addAttribute(key, value);
					} else {
						if (null != root.element("Expand")) {
							if (null == root.element("Expand").attribute("BPMGraphInfoPath")) {
								root.remove(root.element("Expand"));
							} else {
								deleteXmlElementAttribute(root.element("Expand"),
										root.element("Expand").attribute(key));
							}
						}
					}
					break;
				default:
					break;
			}
			OutputFormat format = getOutputFormat();
			String solutionPath = FilePathHelper.getWorkspacePath() + File.separator;
			String newFilePath = Paths.get(WebDesignerConfiguration.getDesignerDataPath(), "tmp", filePath.substring(solutionPath.length()).replace(File.separator, "__") + "." + System.currentTimeMillis()).toString();
			fileOutputStream = new FileOutputStream(newFilePath);
			writer = new XMLWriter(fileOutputStream, format);
			writer.write(document);
			XmlFileProcessor.stackput(filePath, newFilePath);
			result.setCode(0);
			result.setMsg("更新xml成功");
		} catch (Exception e) {
			logger.warning("更新xml异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result.setCode(999);
			result.setMsg("更新xml失败，失败消息为:" + e.getMessage());
		} catch (Throwable throwable) {
			logger.warning(throwable.getMessage());
		} finally {
			closeFileStream(fileOutputStream, writer);
		}
		return result;
	}

	/***
	 * 写xml格式化
	 * @return 格式化类
	 */
	private OutputFormat getOutputFormat() {
		// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
		OutputFormat format = OutputFormat.createPrettyPrint();
		// 设置xml标签和Process标签之间没有空行
		format.setNewLineAfterDeclaration(false);
		format.setIndentSize(4);
		format.setExpandEmptyElements(false);
		format.setPadText(false);
		return format;
	}

	/***
	 * 部署工作流
	 * @param jsonQuestVo 前端传过来的数据对象
	 * @return 新增结果
	 */
	public ResponseResult<JSONObject> deployBpm(JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> result = new ResponseResult<>();
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		try {
			// 工作流xml文件路径
			String filePath = jsonQuestVo.getFilePath();
			// 获取发布BPM的路径
			File workFlowFile = new File(filePath);
			String parentPath = workFlowFile.getParent();
			String bpmFilePath = parentPath.substring(0, parentPath.length() - 4) + File.separator + "BPM.xml";
			// 前端传过来的要更新的json
			JSONObject frontUpdateJson = JSON.parseObject(jsonQuestVo.getContent());
			if (null == frontUpdateJson) {
				result.setCode(1);
				result.setMsg("工作流标识为空，不能部署");
				return result;
			}

			// 创建SAX读取器
			SAXReader reader = new SAXReader();
			// 加载文档
			Document document = reader.read(new File(bpmFilePath));
			// 获取根节点
			Element root = document.getRootElement();

			// 工作流key
			String key = frontUpdateJson.getString(ConstantUtil.KEY);

			if (null == root.element("DeployInfoCollection")) {
				root.addElement("DeployInfoCollection");
			}

			// 若存在，先删除
			if (null != root.element("DeployInfoCollection").elements("DeployInfo")) {
				List<Element> deployInfoList = root.element("DeployInfoCollection").elements("DeployInfo");
				for (Element ele : deployInfoList) {
					if (key.equals(ele.attributeValue(ConstantUtil.KEY))) {
						root.element("DeployInfoCollection").remove(ele);
					}
				}
			}

			// 后新增
			root.element("DeployInfoCollection").addElement("DeployInfo")
					.addAttribute("InitDate",
							DateUtil.getDateFormatText(null, "yyyy-MM-dd HH:mm:ss"))
					.addAttribute(ConstantUtil.KEY, key);

			// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
			OutputFormat format = getOutputFormat();

			fileOutputStream = new FileOutputStream(bpmFilePath);
			writer = new XMLWriter(fileOutputStream, format);
			writer.write(document);
			result.setCode(0);
			result.setMsg("部署工作流成功");
		} catch (Exception e) {
			logger.warning("部署工作流异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result.setCode(999);
			result.setMsg("部署工作流失败，失败消息为:" + e.getMessage());
		} catch (Throwable throwable) {
			logger.warning(throwable.getMessage());
		} finally {
			closeFileStream(fileOutputStream, writer);
		}
		return result;
	}

	/***
	 * 流程单据关联设置
	 * @param jsonQuestVo 前端传过来的数据对象
	 * @return 新增结果
	 */
	public ResponseResult<List<JSONObject>> workflowAndBillsConfig(JsonQuestVo jsonQuestVo) {
		ResponseResult<List<JSONObject>> result = new ResponseResult<>();
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		try {
			// 工作流xml文件路径
			String filePath = jsonQuestVo.getFilePath();
			if (StringUtils.isEmpty(filePath)) {
				result.setCode(1);
				result.setMsg("参数filePath为空，不能设置");
				return result;
			}
			// 获取发布BPM的路径
			File workFlowFile = new File(filePath);
			// 创建SAX读取器
			SAXReader reader = new SAXReader();
			// 加载文档
			Document document = reader.read(workFlowFile);
			// 获取根节点
			Element root = document.getRootElement();
			if (null != root.element("ProcessMapCollection")) {
				//先将之前绑定表单上新增得操作按钮全部删除
//				List<Element> oldElements = root.element("ProcessMapCollection").elements("ProcessMap");
//				for (int j = 0; j < oldElements.size(); j++) {
//					Element element = oldElements.get(j);
//					String formKey = element.attributeValue(ConstantUtil.KEY);
					// 删除流程单据关联设置时，删除表单添加得操作
//					removeOperation(formKey);
//				}
				root.remove(root.element("ProcessMapCollection"));
			}
			root.addElement("ProcessMapCollection");

			// 前端传过来的要更新的json
			JSONArray arrayJson = null;
			if (!"null".equals(jsonQuestVo.getContent()) && StringUtils.isNotEmpty(jsonQuestVo.getContent())) {
				String json = jsonQuestVo.getContent();
				json = json.replaceAll("\"\\{", "{")
						.replaceAll("}\"", "}")
						.replaceAll("(\\\\r\\\\n)|\\\\n", "\\\n")
						.replaceAll("\\\\", "");
				arrayJson = JSON.parseArray(json);
			}
			JSONObject obj;
			String type;
			Element processMapEle;
			/*if (CollectionUtils.isEmpty(arrayJson)) {
			 *//*   result.setCode(1);
                result.setMsg("前端传过来的json为空，不能设置");*//*
                result.setCode(0);
                result.setMsg("流程单据关联设置成功");
                return result;
            }*/
			int i = 0;
			while (i < arrayJson.size()) {
				obj = arrayJson.getJSONObject(i);
				// 绑定标识为空的时候就就跳出 进入下一次循环
				if (StringUtils.isEmpty(obj.getString(ConstantUtil.KEY))) {
					i++;
					continue;
				}
				processMapEle = root.element("ProcessMapCollection").addElement("ProcessMap")
						.addAttribute("InitDate",
								DateUtil.getDateFormatText(null, "yyyy-MM-dd HH:mm:ss") + " CST")
						.addAttribute(ConstantUtil.KEY, obj.getString(ConstantUtil.KEY))
						.addAttribute("StartAction", obj.getString("StartAction"))
						.addAttribute("StartCaption", obj.getString("StartCaption"));
				type = obj.getString(ConstantUtil.TYPE);
				if ("Bill".equals(type)) {
					processMapEle.addAttribute("ProcessKey", obj.getString("ProcessKey"));
				} else if (ConstantUtil.DATA_OBJECT.equals(type)) {
					processMapEle.addAttribute(ConstantUtil.TYPE, type);
					processMapEle.addAttribute("ProcessKeyFormula", obj.getString("ProcessKey"));
				}
				if ("true".equals(obj.getString("IsDynamic"))) {
					processMapEle.addAttribute("IsDynamic", obj.getString("IsDynamic"));
					processMapEle.addAttribute("ProcessKeyFormula", obj.getString("ProcessKey"));
				}
				String v = obj.getString("Perm");
				// Perm的设置
				if (StringUtils.isNotEmpty(v)) {
					v = v.replaceAll("\"\"\\[", "\"[")
							.replaceAll("]\"\"", "]")
							.replaceAll("\"\\[", "[")
							.replaceAll("]\"", "]")
							.replaceAll("(\\\\r\\\\n)|\\\\n", "\\\n");
					JSONObject permJson = JSON.parseObject(v);
					if (null != permJson) {
						processMapEle.addElement("Perm");
						// 设置操作权限
						setXmlPerm(permJson, processMapEle, "OptPerm",
								"OptPermItem");
						// 设置可见性
						setXmlPerm(permJson, processMapEle, "VisiblePerm",
								"VisiblePermItem");
						// 设置可用性
						setXmlPerm(permJson, processMapEle, "EnablePerm",
								"EnablePermItem");
					}
				}
				i++;
			}
			// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
			OutputFormat format = getOutputFormat();
			fileOutputStream = new FileOutputStream(filePath);
			writer = new XMLWriter(fileOutputStream, format);
			writer.write(document);
			result.setCode(0);
			result.setMsg("流程单据关联设置成功");
			List<JSONObject> jsonObjects = new ArrayList<>();
			//关闭流
			closeFileStream(fileOutputStream, writer);
			//当设置成功之后去当前操作的表单里面添加InstanceID、WORKITEM、BPM这三个属性
			//setOperation(jsonObjects, jsonQuestVo, format, fileOutputStream, writer);
			IMetaFactory globalInstance = MetaFactory.getGlobalInstance();
			globalInstance.reloadMetaBPM();
			result.setData(jsonObjects);
		} catch (Exception e) {
			logger.warning("流程单据关联设置异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result.setCode(999);
			result.setMsg("流程单据关联设置失败，失败消息为:" + e.getMessage());
		} catch (Throwable throwable) {
			logger.warning(throwable.getMessage());
		} finally {
			closeFileStream(fileOutputStream, writer);
		}
		return result;
	}
	/**
	 * @param metaForm
	 * @param mainTableKey
	 * @return 检查主表中是否已经存在流程标识，有的话就不再进行添加
	 * @throws Throwable
	 */
	private boolean checkColumn(MetaForm metaForm, String mainTableKey) throws Throwable {
		boolean ishas = false;
		XmlTreeWithPath xmlTree;
		if (StringUtils.isNotEmpty(metaForm.getDataSource().getRefObjectKey())) {//引入数据对象表单
			MetaDataObject dataObject = metaForm.getDataSource().getDataObject();
			String filePath = LoadFileTree.loadFilePathByDataObjectFieldKey(dataObject.getKey());
			xmlTree = XmlTreeWithPath.parseFilePath(filePath);
		} else {
			String filePath = LoadFileTree.getPathByFormKey(metaForm.getKey());
			xmlTree = XmlTreeWithPath.parseFilePath(filePath);
		}
		List<TagNode> table = xmlTree.xmlTree.getRoot().findNodesByTagName("Table");
		for (TagNode tag : table) {
			if (tag.getAttributes().get(ConstantUtil.KEY).equals(mainTableKey)) {
				//去判断主表中是否已经存在流程标识
				List<TagNode> column = tag.findNodesByTagName("Column");
				for (TagNode tg : column) {
					if (tg.getAttributes().get(ConstantUtil.KEY).equals("InstanceID")) {
						ishas = true;
						break;
					}
				}
			}
		}
		return ishas;
	}

	/**
	 * @param element
	 * @return 从表单中获取TableNode节点
	 */
	private List<Element> findTableNode(Element element) {
		List<Element> elements = element
				.element(ConstantUtil.DATA_SOURCE)
				.element(ConstantUtil.DATA_OBJECT)
				.element("TableCollection")
				.elements("Table");
		return elements;
	}

	/**
	 * @param element
	 * @return 从数据对象中获取TableNode节点
	 */
	private List<Element> findDataObjectTableNode(Element element) {
		List<Element> elements = element
				.element("TableCollection")
				.elements("Table");
		return elements;
	}

	/***
	 * 设置Perm的相关属性
	 * @param permJson 权限json
	 * @param node 当前操作的节点
	 * @param firstElementName xml中Perm元素下第一级元素名称
	 * @param secondElementName xml中Perm元素下第二级元素名称
	 */
	private void setXmlPerm(JSONObject permJson, Element node,
							String firstElementName, String secondElementName) {
		JSONArray permArray = permJson.getJSONArray(firstElementName);
		if (!CollectionUtils.isEmpty(permArray)) {
			if (null != node.element("Perm").element(firstElementName)) {
				node.element("Perm").remove(node.element("Perm").element(firstElementName));
			}
			Element permElement = node.element("Perm").addElement(firstElementName);
			JSONObject eachObj;
			for (Object o : permArray) {
				eachObj = (JSONObject) o;
				permElement.addElement(secondElementName).addAttribute(ConstantUtil.KEY, eachObj.getString(ConstantUtil.KEY));
			}
		} else {
			node.element("Perm").remove(node.element("Perm").element(firstElementName));
		}
	}

	/***
	 * 新增版本
	 * @param jsonQuestVo 前端传过来的数据对象
	 * @return 新增结果
	 */
	public ResponseResult<JSONObject> newVersion(JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> result = new ResponseResult<>();
		JSONObject data = new JSONObject();
		// 新文件路径
		String newFilePath = "";
		try {
			// 工作流xml文件路径
			String filePath = jsonQuestVo.getFilePath();
			if (StringUtils.isEmpty(filePath)) {
				result.setCode(1);
				result.setMsg("参数filePath为空，不能新增版本");
				return result;
			}
			// 前端传过来的要更新的json
			JSONObject frontUpdateJson = JSON.parseObject(jsonQuestVo.getContent());
			if (null == frontUpdateJson) {
				result.setCode(1);
				result.setMsg("参数为空，不能新增版本");
				return result;
			}
			if (StringUtils.isEmpty(frontUpdateJson.getString(ConstantUtil.KEY))) {
				result.setCode(1);
				result.setMsg("参数Key为空，不能新增版本");
				return result;
			}
			if (StringUtils.isEmpty(frontUpdateJson.getString(ConstantUtil.CAPTION))) {
				result.setCode(1);
				result.setMsg("参数Caption为空，不能新增版本");
				return result;
			}
			if (StringUtils.isEmpty(frontUpdateJson.getString(ConstantUtil.VERSION))) {
				result.setCode(1);
				result.setMsg("参数Version为空，不能新增版本");
				return result;
			}
			// 获取项目标识
			String projectKey = getProjectKeyByFilePath(filePath);
			String key = frontUpdateJson.getString(ConstantUtil.KEY);
			String caption = frontUpdateJson.getString(ConstantUtil.CAPTION);
			Long version = frontUpdateJson.getLong(ConstantUtil.VERSION);
			Map<String, String> res = this.createXml(projectKey, key, caption, version.toString());
			if ("true".equals(res.get("result"))) {
				newFilePath = res.get("filePath");
				// 成功
				data.put("code", 0);
				data.put("msg", "新增版本成功");
				data.put("filePath", newFilePath);
				data.put("fileName", res.get("fileName"));
				data.put("caption", caption);
				data.put("version", res.get("version"));
			} else {
				result.setCode(999);
				result.setMsg("新增版本失败，失败消息为:" + res.get("msg"));
				return result;
			}
			result.setCode(0);
			result.setMsg("新增版本成功");
			result.setData(data);
		} catch (Exception e) {
			logger.warning("新增版本异常，异常为:" + ExceptionUtils.getStackTrace(e));
			if (StringUtils.isNotEmpty(newFilePath)) {
				try {
					LoadFileTree.removeFilePath(newFilePath);
				} catch (Throwable e1) {
					logger.warning("删除文件异常,异常为:" + ExceptionUtils.getStackTrace(e1));
				}
			}
			result.setCode(999);
			result.setMsg("新增版本失败，失败消息为:" + e.getMessage());
		}
		return result;
	}

	/***
	 * 从文件路径总获取项目标识
	 * @param filePath 文件路径
	 * @return 项目标识
	 */
	private String getProjectKeyByFilePath(String filePath) {
		String parentPath = new File(filePath).getParent();
		return parentPath.substring(parentPath.indexOf("solution") + 9, parentPath.lastIndexOf(File.separator));
	}

	/***
	 * 根据文件路径获取根元素
	 * @param filePath 文件路径
	 * @return 根元素
	 * @throws DocumentException 文档异常
	 */
	private Element getRootElementByFilePath(String filePath) throws DocumentException {
		// 创建SAX读取器
		SAXReader reader = new SAXReader();
		// 加载文档
		Document document = reader.read(new File(filePath));
		// 获取根节点
		return document.getRootElement();
	}

	/***
	 * 条件获取json数据
	 * @param filePath 文件路径
	 * @return json数据
	 */
	public ResponseResult<JSONArray> getWorkflowAndBillsTableData(String filePath) {
		ResponseResult<JSONArray> result = new ResponseResult<>();
		try {
			JSONArray jsonArray = new JSONArray();
			// 根据路径获取根元素
			Element root = getRootElementByFilePath(filePath);

			// [{id:1,Type:'Bill',Key:'',DynamicBinding:false,ProcessKey:'',StartAction:'',StartCaption:'',Perm:{}}]
			if (!CollectionUtils.isEmpty(root.element("ProcessMapCollection").elements("ProcessMap"))) {
				final List<Element> elements = root.element("ProcessMapCollection").elements("ProcessMap");
				JSONObject obj;
				Element ele;
				for (int i = 0; i < elements.size(); i++) {
					ele = elements.get(i);
					obj = new JSONObject();
					obj.put("id", (i + 1));
					if (StringUtils.isNotEmpty(ele.attributeValue(ConstantUtil.TYPE))) {
						obj.put(ConstantUtil.TYPE, ConstantUtil.DATA_OBJECT);
					} else {
						obj.put(ConstantUtil.TYPE, "Bill");
					}
					obj.put(ConstantUtil.KEY, ele.attributeValue(ConstantUtil.KEY));
					obj.put("IsDynamic", StringUtils.isNotEmpty(ele.attributeValue("IsDynamic")));
					if (StringUtils.isNotEmpty(ele.attributeValue("ProcessKey"))) {
						obj.put("ProcessKey", ele.attributeValue("ProcessKey"));
					}
					if (StringUtils.isNotEmpty(ele.attributeValue("ProcessKeyFormula"))) {
						obj.put("ProcessKey", ele.attributeValue("ProcessKeyFormula"));
					}
					obj.put("StartAction", ele.attributeValue("StartAction"));
					obj.put("StartCaption", ele.attributeValue("StartCaption"));
					obj.put("Perm", JSON.toJSONString(setPerm(ele)));
					jsonArray.add(obj);
				}
			}

			result.setCode(0);
			result.setMsg("获取json数据成功");
			result.setData(jsonArray);
		} catch (Exception e) {
			logger.warning("获取json数据异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result.setCode(999);
			result.setMsg("获取json数据失败，失败消息为:" + e.getMessage());
		}
		return result;
	}

	/***
	 * 设置任务的权限设置
	 * @param node 节点
	 * @return 权限的json设置
	 */
	private JSONObject setPerm(Element node) {
		List<Element> optPermItems = null;
		List<Element> visiblePermItems = null;
		List<Element> enablePermItems = null;
		if (null != node.element("Perm") && null != node.element("Perm").element("OptPerm")) {
			optPermItems = node.element("Perm").element("OptPerm").elements("OptPermItem");
		}
		if (null != node.element("Perm") && null != node.element("Perm").element("VisiblePerm")) {
			visiblePermItems = node.element("Perm").element("VisiblePerm").elements("VisiblePermItem");
		}
		if (null != node.element("Perm") && null != node.element("Perm").element("EnablePerm")) {
			enablePermItems = node.element("Perm").element("EnablePerm").elements("EnablePermItem");
		}

		JSONObject value = new JSONObject();
		List<String> keyList = Arrays.asList("OptPerm", "VisiblePerm", "EnablePerm");
		JSONArray array;
		List<Element> elements = null;
		for (String k : keyList) {
			switch (k) {
				case "OptPerm":
					elements = optPermItems;
					break;
				case "VisiblePerm":
					elements = visiblePermItems;
					break;
				case "EnablePerm":
					elements = enablePermItems;
					break;
				default:
					break;
			}
			if (!CollectionUtils.isEmpty(elements)) {
				array = new JSONArray();
				JSONObject jsonObj;
				for (Element ele : elements) {
					jsonObj = new JSONObject();
					jsonObj.put(ConstantUtil.KEY, ele.attributeValue(ConstantUtil.KEY));
					array.add(jsonObj);
				}
				value.put(k, array);
			}
		}
		return value;
	}

	/***
	 * 创建xml中的节点和路径
	 * @param jsonQuestVo 前端传过来的数据对象
	 * @param operType 操作类型，1:节点和路径的新增和修改；2：属性的修改；3：删除
	 * @return 新增结果
	 */
	public ResponseResult<JSONObject> handleNodesAndPaths(JsonQuestVo jsonQuestVo, Integer operType) {
		ResponseResult<JSONObject> result = new ResponseResult<>();
		try {
			// xml文件路径
			String filePath = jsonQuestVo.getFilePath();
			// 前端传过来的全部json内容
			JSONObject frontFullJson = JSON.parseObject(jsonQuestVo.getContent());
			// 返回给前端的json结果对象
			JSONObject operJsonResultObject;
			// 当前操作的节点id
			String nodeId = jsonQuestVo.getNodeId();

			// 如果是删除，走删除xml的逻辑
			if (OPER_TYPE_THREE == operType) {
				return deleteXmlNodeAndPaths(filePath, frontFullJson, nodeId);
			}

			// frontFullJson中要替换的字符串属性
			String replaceStr;
			// 根据nodeId从frontFullJson中获取到对应的json内容
			JSONObject operNodeJson = frontFullJson.getJSONObject("states").getJSONObject(nodeId);
			if (null != operNodeJson) {
				replaceStr = "states";
				// 处理节点
				operJsonResultObject = handleNodes(filePath, nodeId, operNodeJson);
			} else {
				operNodeJson = frontFullJson.getJSONObject("paths").getJSONObject(nodeId);
				if (null == operNodeJson) {
					result.setCode(1);
					result.setMsg("操作失败,该节点不存在");
					return result;
				}
				replaceStr = "paths";
				// 处理路径
				operJsonResultObject = new Path().save(filePath, frontFullJson, nodeId, operNodeJson);
			}

			// 填充节点数据到全量json中
			this.fillFrontFullJson(frontFullJson, operJsonResultObject, nodeId, replaceStr);

			// 整个工作流json
			JSONObject restoreJson = new JSONObject();
			restoreJson.put("restore", frontFullJson);
			result.setCode(0);
			result.setMsg("操作成功");
			result.setData(restoreJson);
		} catch (Throwable e) {
			logger.warning("操作异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result.setCode(999);
			result.setMsg("操作失败，失败消息为:" + e.getMessage());
		}
		return result;
	}

	/***
	 * 处理节点
	 * @param filePath 文件路径
	 * @param nodeId 节点id
	 * @param operNodeJson 数据对象
	 * @return 结果
	 * @throws Exception 异常
	 */
	@SuppressWarnings("unchecked")
	private JSONObject handleNodes(String filePath, String nodeId, JSONObject operNodeJson) throws Throwable {
		JSONObject result;
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		Element node;
		// 获取节点类型
		String nodeType = operNodeJson.getString("type");
		try {
			//获取临时文件路径
			String tempPath = XmlFileProcessor.instance.getTmpFile(filePath);
			if (StringUtils.isBlank(tempPath)) {
				tempPath = filePath;
			}
			// 创建SAX读取器
			SAXReader reader = new SAXReader();
			// 加载文档
			Document document = reader.read(new File(tempPath));
			// 获取根节点
			Element root = document.getRootElement();

			// 处理保存到xml文件逻辑
			String packagePath = "com.bokesoft.yes.design.bpm.po";
			Class<Object> c;
			c = (Class<Object>) forName(packagePath + "." + nodeType);
			Object instance = c.newInstance();

			// 调用公共的保存方法
			Method commonMethod =
					c.getMethod("save", JSONObject.class, Element.class, String.class, String.class);
			//调用Method类的方法invoke运行sleep方法
			node = (Element) commonMethod.invoke(instance, operNodeJson, root, nodeType, nodeId);
			// 保存自己属性
			Method selfMethod = c.getMethod("saveSelfAttributesToXml", JSONObject.class, Element.class);
			selfMethod.invoke(instance, operNodeJson, node);

			// 调用设置前端默认值方法
			Method frontDefaultValuesMethod = c.getMethod("setFrontDefaultValues");
			result = (JSONObject) frontDefaultValuesMethod.invoke(instance);

			// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
			OutputFormat format = getOutputFormat();
			String solutionPath = FilePathHelper.getWorkspacePath() + File.separator;
			String newFilePath = Paths.get(WebDesignerConfiguration.getDesignerDataPath(), "tmp", filePath.substring(solutionPath.length()).replace(File.separator, "__") + "." + System.currentTimeMillis()).toString();
			fileOutputStream = new FileOutputStream(newFilePath);
			writer = new XMLWriter(fileOutputStream, format);
			XmlFileProcessor.stackput(filePath, newFilePath);
			/**
			 * 这边使用dom4j转译xml时CDATA 部件两边的尖括号会被转译成&lt 和&gt这边  对xml文件进行强制格式化
			 * */
			writer.setEscapeText(false); //可能存在问题这边使用try catch 进行简单处理，后期看如何修改
			try {
				writer.write(document);
			} catch (IOException e) {
				logger.warning("格式化XML文档发生异常，请检查！");
			}
		} catch (Exception e) {
			logger.warning("保存属性到xml文件异常，异常为:" + ExceptionUtils.getStackTrace(e));
			throw e;
		} finally {
			this.closeFileStream(fileOutputStream, writer);
		}
		return result;
	}

	/***
	 * 删除xml中的节点和路径功能
	 * @param filePath 文件路径(包含文件名)
	 * @param frontFullJson 前端传过来的json
	 * @param nodeId 当前要删除的节点
	 * @return 删除结果
	 */
	private ResponseResult<JSONObject> deleteXmlNodeAndPaths(String filePath, JSONObject frontFullJson, String nodeId) {
		ResponseResult<JSONObject> result = new ResponseResult<>();
		try {
			String key;
			// 根据nodeId获取到Key
			JSONObject json = frontFullJson.getJSONObject("states").getJSONObject(nodeId);
			if (null == json) {
				json = frontFullJson.getJSONObject("paths").getJSONObject(nodeId);
			}
			key = json.getJSONObject("props").getJSONObject(ConstantUtil.KEY).getString("value");
			String nodeType = json.getJSONObject("props").getJSONObject("NodeType").getString("value");
			Boolean delResult = this.handleDelNodesAndPaths(filePath, key, nodeType);
			if (delResult) {
				result.setCode(0);
				result.setMsg("删除成功");
			} else {
				result.setCode(1);
				result.setMsg("删除失败");
			}
			// 整个工作流json
			JSONObject restoreJson = new JSONObject();
			restoreJson.put("restore", frontFullJson);
			result.setData(restoreJson);
		} catch (Exception e) {
			logger.warning("删除xml中的节点和路径功能异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result.setCode(999);
			result.setMsg("删除失败");
		}
		return result;
	}

	/***
	 * 关闭文件流
	 * @param fileOutputStream 文件输出流
	 * @param writer 写对象
	 */
	private void closeFileStream(FileOutputStream fileOutputStream, XMLWriter writer) {
		try {
			if (null != fileOutputStream) {
				fileOutputStream.close();
			}
			if (null != writer) {
				writer.close();
			}
		} catch (Exception e) {
			logger.warning("关闭writer异常，异常为:" + ExceptionUtils.getStackTrace(e));
		}
	}

	/***
	 * 删除xml中的节点或路径
	 * @param filePath 文件路径
	 * @param key 属性key的值
	 * @param nodeType 节点类型
	 * @return 删除结果
	 */
	private Boolean handleDelNodesAndPaths(String filePath, String key, String nodeType) {
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		boolean result = true;
		try {
			//获取临时文件路径
			String tempPath = XmlFileProcessor.instance.getTmpFile(filePath);
			if (StringUtils.isBlank(tempPath)) {
				tempPath = filePath;
			}
			// 创建SAX读取器
			SAXReader reader = new SAXReader();
			// 加载文档
			Document document = reader.read(new File(tempPath));
			// 获取根节点
			Element root = document.getRootElement();

			// 获取跟元素下的所有子元素
			List<Element> elements = root.elements();

			// if:SequenceFlow、Association、ExceptionFlow是路径的删除；
			// else:是非路径的删除
			if (NodeKeyAndCaptionEnum.SequenceFlow.getKey().equals(nodeType) ||
					NodeKeyAndCaptionEnum.Association.getKey().equals(nodeType) ||
					NodeKeyAndCaptionEnum.ExceptionFlow.getKey().equals(nodeType)) {
				Element transitionCollectionEle;
				for (Element ele : elements) {
					transitionCollectionEle = ele.element("TransitionCollection");
					if (null == transitionCollectionEle) {
						continue;
					}
					List<Element> paths = transitionCollectionEle.elements(nodeType);
					if (!CollectionUtils.isEmpty(paths)) {
						for (Element e : paths) {
							if (null != e && key.equals(e.attributeValue(ConstantUtil.KEY))) {
								transitionCollectionEle.remove(e);
								break;
							}
						}
					}
				}
			} else {
				// 先删除目标节点为该节点的路径节点
				Element transitionCollection;
				List<String> paths = Arrays.asList("SequenceFlow", "Association", "ExceptionFlow");
				List<Element> pathEles;
				for (Element ele : elements) {
					transitionCollection = ele.element("TransitionCollection");
					if (null == transitionCollection) {
						continue;
					}
					for (String path : paths) {
						if (!CollectionUtils.isEmpty(transitionCollection.elements(path))) {
							pathEles = transitionCollection.elements(path);
							for (Element e : pathEles) {
								if (key.equals(e.attributeValue("TargetNodeKey"))) {
									logger.info("删除路径:" + e.attributeValue(ConstantUtil.KEY) + "成功");
									transitionCollection.remove(e);
								}
							}
						}
					}
				}
				// 然后删除节点本身
				for (Element ele : elements) {
					if ("SwimlineCollection".equals(ele.getName())
							&& NodeKeyAndCaptionEnum.Swimline.getKey().equals(nodeType)) {
						List<Element> swimlines = ele.elements("Swimline");
						if (CollectionUtils.isEmpty(swimlines)) {
							continue;
						}
						for (Element e : swimlines) {
							if (key.equals(e.attributeValue(ConstantUtil.KEY))) {
								ele.remove(e);
								logger.info("删除元素" + key + "成功");
								break;
							}
						}
					} else {
						if (key.equals(ele.attributeValue(ConstantUtil.KEY))) {
							root.remove(ele);
							logger.info("删除元素" + key + "成功");
							break;
						}
					}
				}
			}

			// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
			OutputFormat format = getOutputFormat();
			String solutionPath = FilePathHelper.getWorkspacePath() + File.separator;
			String newFilePath = Paths.get(WebDesignerConfiguration.getDesignerDataPath(), "tmp", filePath.substring(solutionPath.length()).replace(File.separator, "__") + "." + System.currentTimeMillis()).toString();

			fileOutputStream = new FileOutputStream(newFilePath);
			writer = new XMLWriter(fileOutputStream, format);

			XmlFileProcessor.stackput(filePath, newFilePath);
			writer.write(document);
		} catch (Exception e) {
			logger.warning("删除异常，异常为:" + ExceptionUtils.getStackTrace(e));
			result = false;
		} catch (Throwable throwable) {
			logger.warning(throwable.getMessage());
		} finally {
			closeFileStream(fileOutputStream, writer);
		}
		return result;
	}

	/***
	 * 填充前端json
	 * @param frontFullJson 前端json
	 * @param operJsonObject 操作的json对象
	 * @param nodeId 节点id
	 * @param replaceStr 要替换的字符串属性
	 */
	private void fillFrontFullJson(JSONObject frontFullJson, JSONObject operJsonObject,
								   String nodeId, String replaceStr) {
		JSONObject nodeIdJsonObj = frontFullJson.getJSONObject(replaceStr).getJSONObject(nodeId);
		if (null != nodeIdJsonObj) {
			nodeIdJsonObj.put("props", operJsonObject);
			frontFullJson.put(nodeId, nodeIdJsonObj);
		}
	}

	/***
	 * 删除xml元素属性
	 * @param element 元素
	 * @param attribute 属性
	 */
	private void deleteXmlElementAttribute(Element element, Attribute attribute) {
		if (null != attribute) {
			element.remove(attribute);
		}
	}
}
