package com.bokesoft.distro.tech.commons.basis.io;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import com.bokesoft.distro.tech.commons.basis.io.internal.FileUnit;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.distro.tech.commons.basis.io.internal.LocalFileTempWriter;
import com.bokesoft.distro.tech.commons.basis.io.internal.TempWriterManager;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 基于本地文件实现的临时对象数据存储
 */
public class LocalFileTempStorage<T> {
    private static final Logger log = LoggerFactory.getLogger(LocalFileTempStorage.class);

    private String baseDir;
    private Class<T> clazz;

    private String tmpDir;
    private String readyDir;
    private String historyDir;
    private String errDir;

    private LocalFileTempWriter<T> writer;

    private LocalFileTempStorage(String baseDir, Class<T> clazz){
        this.baseDir = baseDir;
        this.clazz = clazz;
    }

    /**
     * 构建 LocalFileTempStorage
     * @param baseDir 数据存储的根目录
     * @param clazz 存储数据对象的 class
     * @param <T>
     * @return
     */
    public static <T> LocalFileTempStorage<T> build(String baseDir, Class<T> clazz) {
        LocalFileTempStorage<T> storage = new LocalFileTempStorage<>(baseDir, clazz);
        storage.tmpDir = getTmpFolder(storage.baseDir, storage.clazz.getName());
        storage.readyDir = getReadyFolder(storage.baseDir, storage.clazz.getName());
        storage.historyDir = getHistoryFolder(storage.baseDir, storage.clazz.getName());
        storage.errDir = getErrorFolder(storage.baseDir, storage.clazz.getName());
        TempWriterManager.bind(clazz, storage.tmpDir, storage.readyDir, storage.errDir);
        return storage;
    }

    /**
     * 修改 writer 的 文件区块切分时间
     * @param blockTime
     * @return
     */
    public LocalFileTempStorage<T> blockTime(long blockTime){
        TempWriterManager.modifyBlockTime(this.clazz,blockTime);
        return this;
    }

    /**
     * 开启监控线程,准备使用
     * @return
     */
    public LocalFileTempStorage<T> start() throws IOException {
        FileUtils.forceMkdir(new File(this.tmpDir));
        FileUtils.forceMkdir(new File(this.readyDir));
        FileUtils.forceMkdir(new File(this.historyDir));
        FileUtils.forceMkdir(new File(this.errDir));

        TempWriterManager.startMonitor();
        log.info("已启动 LocalFileTempStorage: class={}, baseDir={} .", this.clazz, this.baseDir);

        return this;
    }


    /**
     * 将实例通过对象序列化方式,临时保存在文件中
     * @param value
     * @throws IOException
     */
    @SuppressWarnings("unchecked")
    public void save(T value) throws IOException {
        if (log.isDebugEnabled()){
            log.debug("开始 LocalFileTempStorage [{}] 数据写入: {} ...", this.tmpDir, value);
        }

        this.writer = (LocalFileTempWriter<T>) TempWriterManager.getWriter(this.clazz);
        writer.writeObject(value);
        
        if (log.isDebugEnabled()){
            log.debug("LocalFileTempStorage [{}] 数据写入完成 .", this.tmpDir);
        }
    }

    /**
     * 将实例通过对象集合序列化方式,临时保存在文件中,支持 List和Set
     * @param values
     * @throws IOException
     */
    public void save(Collection<T> values) throws IOException {
        for (T value: values){
            save(value);
        }
    }

    /**
     * 清除过期的文件
     * @param expiredDays 过期天数, 删除这些天之前的数据, 小于等于 0 则删除全部数据
     * @throws IOException
     */
    public void clearHistoryFile(int expiredDays) throws IOException {
        CalendarSerialFolderManager.clearExpiredFiles(historyDir, expiredDays);
    }

    public void doPrepare() throws IOException {
        String[] fileNames = getTmpFiles();
        if(null != fileNames && fileNames.length > 0 ) {
            List<String> activeTmpFiles =  TempWriterManager.getActiveTmpFiles();
            List<String> errorFiles = new LinkedList<>();
            for (String fileName : fileNames) {
                if(activeTmpFiles.contains(fileName)){
                    continue;
                }
                try {
                    FileUnit.transfTmpFiletoReady(fileName, this.clazz, this.tmpDir, this.readyDir, this.errDir);
                }catch (IOException e){
                    errorFiles.add(fileName);
                }
            }
            if(errorFiles.size()>0) {
                log.error("以下tmp区文件:[" + StringUtils.join(errorFiles, ",") + "],迁移至ready区失败");
            }
        }
    }

    /**
     * 将临时保存在文件中的对象反序列化到内存, 并做通过 callback机制处理数据
     * @param consumer
     * @throws IOException
     */
    public void consume(DataConsumer<T> consumer) throws IOException {
        String[] fileNames = getReadyFiles();
        if(null != fileNames && fileNames.length > 0 ) {
            for (String fileName : fileNames) {
                if (log.isDebugEnabled()){
                    log.debug("开始 LocalFileTempStorage [{}] 数据处理: {} ...", this.tmpDir, fileName);
                }

                boolean success = false;
                try {
                    File jsonFile = new File(this.readyDir,fileName);
                    if(jsonFile.canRead()) {
                        List<T> records = loadJsonFromFile(jsonFile, clazz);
                        if (null != records) {
                            success = consumer.process(records);
                            if (log.isDebugEnabled()) {
                                log.debug("LocalFileTempStorage [{}] 数据处理完成: {}, 结果为 {} .", this.tmpDir, fileName, success);
                            }
                        }
                    }
                } catch (Exception e) {
                    success = false;    //出错后, 当前文件不会被 confirm
                    String msg = "LocalFileTempStorage ["+this.tmpDir+"] 数据处理失败: {"+fileName+"}, 错误信息: "+e.getMessage();
                    log.error(msg, e);
                }

                // 成功回调后,使用confirm,移除临时文件
                if(success) {
                    confirm(fileName);
                    if (log.isDebugEnabled()){
                        log.debug("归档 LocalFileTempStorage [{}] 数据完成: {} .", this.tmpDir, fileName);
                    }    
                }
            }
        }
    }

    /**
     * 根据文件名,读取准备区目录中的文件数据,并按对象泛型转化
     * @param jsonFile json字符串的文件
     * @param clazz json对应的class类型
     * @return
     * @throws IOException
     */
    private static <T> List<T> loadJsonFromFile(File jsonFile,Class<T> clazz) throws IOException {
        List<T> result = new LinkedList<>();
        List<String> jsonStrList = FileUtils.readLines(jsonFile,"UTF-8");
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES,false);
        for(String jsonStr:jsonStrList){
            T data = objectMapper.readValue(jsonStr,clazz);
            result.add(data);
        }
        return result;
    }

    /**
     * 确认文件已处理,则从准备区移至历史区
     * @param fileName 文件名
     * @throws IOException
     */
    private void confirm(String fileName) throws IOException {
        Path srcPath = Paths.get(readyDir,fileName);
        Path tagPath = Paths.get(CalendarSerialFolderManager.prepareFolder(this.historyDir),fileName);
        Files.move(srcPath,tagPath);
    }

    /**
     * 获取准备区目录中的所有文件的文件名
     * @return
     * @throws IOException
     */
    private String[] getReadyFiles() throws IOException {
        File folder = new File(readyDir);
        return getSonFiles(folder);
    }

    /**
     * 获取临时区目录中的所有文件的文件名
     * @return
     * @throws IOException
     */
    private String[] getTmpFiles() throws IOException {
        File folder = new File(tmpDir);
        return getSonFiles(folder);
    }

    private String[] getSonFiles(File folder) {
        String[] files = folder.list();
        Arrays.sort(files);
        List<String> matchFileList = new LinkedList<>();
        for(String file:files){
            File sonFile = Paths.get(folder.getAbsolutePath(), file).toFile();
            if(sonFile.isFile() && sonFile.getName().endsWith(TempWriterManager.VERSION)){
                matchFileList.add(sonFile.getName());
            }
        }
        return matchFileList.toArray(new String[]{});
    }

    /**
     * 获取 class 对应的临时区目录存在
     * @param baseDir
     * @param className
     * @return 临时区目录地址
     * @throws IOException
     */
    private static String getTmpFolder(String baseDir, String className) {
        Path folderPath = Paths.get(baseDir+"/tmp/"+className);
        return getFolderDir(folderPath);
    }

    /**
     * 获取 class 对应的准备区目录存在
     * @param baseDir
     * @param className
     * @return 准备区目录地址
     * @throws IOException
     */
    private static String getReadyFolder(String baseDir, String className) {
        Path folderPath = Paths.get(baseDir+"/ready/"+className);
        return getFolderDir(folderPath);
    }

    /**
     * 获取 class 对应的历史区目录存在
     * @param baseDir
     * @param className
     * @return 历史区目录地址
     * @throws IOException
     */
    private static String getHistoryFolder(String baseDir, String className) {
        Path folderPath = Paths.get(baseDir+"/history/"+className);
        return getFolderDir(folderPath);
    }

    /**
     * 获取 class 对应的错误区目录存在
     * @param baseDir
     * @param className
     * @return 历史区目录地址
     * @throws IOException
     */
    private static String getErrorFolder(String baseDir, String className) {
        Path folderPath = Paths.get(baseDir+"/error/"+className);
        return getFolderDir(folderPath);
    }

    private static String getFolderDir(Path folderPath) {
        return folderPath.toAbsolutePath().toString();
    }

    /**
     * process过程中,callback函数回调接口
     * @param <T> callback的参数泛型
     */
    public static interface DataConsumer<T> {
        /**
         * callback处理程序
         * @param values
         * @return 如果callback成功处理, 则返回true, 如果异常或失败, 返回false
         */
        public boolean process(List<T> values);
    }

}

