package com.bokesoft.distro.tech.commons.basis.io;

import com.bokesoft.distro.tech.commons.basis.io.internal.FolderExpiredCalcTool;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;

/**
 * yyyy/MM/dd 模式的文件目录序列管理
 */
public class CalendarSerialFolderManager {
    private static final Logger log = LoggerFactory.getLogger(CalendarSerialFolderManager.class);

    private static final String DATA_FOMART_YYYYMMDD = "yyyy/MM/dd";

    /**
     * 以当前日期创建 yyyy/MM/dd 模式的文件目录序列
     * @param basePath 基础路径
     * @return 创建的目录路径
     * @throws IOException
     */
    public static String prepareFolder(String basePath) throws IOException {
        return prepareFolder(basePath, new Date());
    }

    /**
     * 按指定日期创建 yyyy/MM/dd 模式的文件目录序列
     * @param basePath 基础路径
     * @param date 日期参数
     * @return 创建的目录路径
     * @throws IOException
     */
    public static String prepareFolder(String basePath, Date date) throws IOException {
        DateFormat df = new SimpleDateFormat(DATA_FOMART_YYYYMMDD);
        Path folderPath = Paths.get(basePath + "/"+df.format(date));
        FileUtils.forceMkdir(folderPath.toFile());
        return folderPath.toAbsolutePath().toString();
    }

    /**
     * 在 yyyy/MM/dd 模式的文件目录序列中, 清除过期的文件和目录
     * @param basePath 基础路径
     * @param expiredDays 过期天数, 删除这些天之前的数据, 小于等于 0 则删除全部数据
     * @throws IOException
     */
    public static void clearExpiredFiles(String basePath, int expiredDays) throws IOException {
        File folder = new File(basePath);
        if(!folder.exists()){
            log.warn("日期根目录 [{}] 不存在, 跳过处理",basePath);
            return;
        }
        List<String> children = getSubTreePath(basePath);
        if(children.isEmpty()){
            log.warn("日期根目录 [{}] 为空目录, 跳过处理", basePath);
        }
        List<String> needRemoveList = getExpiredFolderPath(children, getExpiredCalendar(expiredDays));
        log.info("日期根目录 [{}] - 清除过期({}天)的子目录: {} ...", basePath, expiredDays, needRemoveList);
        for (String needRemovePath : needRemoveList) {
            FileUtils.deleteDirectory(new File(basePath, needRemovePath));
        }
    }

    /**
     * 给定一组目录地址,根据 yyyy/MM/dd 模式判断 这些目录是否过期
     * @param allPosterityFolder 路径数组
     * @param expiryDateC 过期日期
     * @return
     */
    protected static List<String> getExpiredFolderPath(List<String> allPosterityFolder, Calendar expiryDateC) {
        List<String> result = new LinkedList<>();
        if(null != allPosterityFolder && !allPosterityFolder.isEmpty()) {
            LocalDate expiryDateL = LocalDateTime.ofInstant(
                    expiryDateC.toInstant(), expiryDateC.getTimeZone().toZoneId()).toLocalDate();
            for (String posterityFolder : allPosterityFolder) {
                LocalDate dirDate = FolderExpiredCalcTool.parseDateFromPath(posterityFolder);
                if (expiryDateL.isAfter(dirDate)) {
                    result.add(posterityFolder);
                }
            }
            // 降序排列保证删除的顺序
            result.sort(new Comparator<String>() {
                @Override
                public int compare(String s1, String s2) {
                    if (s1 == null && s2 == null) return 0;
                    if (s1 == null) return -1;
                    if (s2 == null) return 1;
                    return s2.compareTo(s1); // 按降序排序
                }
            });
        }
        return result;
    }

    /**
     * 获取指定目录下所有层级的子目录
     * @param basePath 根目录地址
     * @return
     */
    private static List<String> getSubTreePath(String basePath) throws IOException {
        Path rootDir = Paths.get(basePath);
        List<String> relativePaths = new ArrayList<>();
        Files.walkFileTree(rootDir, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                Path relativePath = rootDir.relativize(dir); // 获取相对路径
                if (!relativePath.toString().isEmpty()) { // 排除根目录本身
                    relativePaths.add(relativePath.toString().replaceAll("\\\\","/"));
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return relativePaths;
    }

    /**
     * 根据限制天数生成对应日期calendar实例
     * @param expiredDays 限制天数
     * @return
     */
    private static Calendar getExpiredCalendar(int expiredDays){
        Calendar calendar = Calendar.getInstance();
        // 将时间设置为当天的 00:00:00
        calendar.add(Calendar.DATE, -1*expiredDays+1);
        return calendar;
    }

}
