package com.bokesoft.distro.tech.commons.basis.jdbc;

import com.bokesoft.distro.tech.commons.basis.jdbc.mssql.SQLServerUtils;
import com.bokesoft.distro.tech.commons.basis.jdbc.struc.DBTypeCollection;
import com.bokesoft.distro.tech.commons.basis.jdbc.struc.DBTypeInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;

import java.util.*;

/**
 * 与数据库类别相关的一些功能
 */
public class DBTypeUtils {
	private static Logger logger = LoggerFactory.getLogger(DBTypeUtils.class);
	private static List<DBTypeInfo> dbTypeInfoCollection=getByDefault();
	private static final String DBTYPE_INFO_FILE = "dbTypeInfo.yml";

	private static List<DBTypeInfo> getByDefault(){
		Yaml yaml=new Yaml();
		logger.info("数据库类型映射开始加载: {} ...",DBTYPE_INFO_FILE);
		DBTypeCollection defaultDBTypeCollection = yaml.loadAs(
				DBTypeUtils.class.getResourceAsStream(DBTYPE_INFO_FILE), DBTypeCollection.class);
		logger.info("数据库类型映射加载完成.");
		return defaultDBTypeCollection.getDbTypeInfos();
	}

	/**
	 * 根据JDBC URL获取数据库类型定义
	 * @param jdbcUrl jdbc请求路径
	 * @return 数据库类型定义实例
	 */
	public static String getDBType(String jdbcUrl){
		return getDBTypeInfo(jdbcUrl).getDbType();
	}

	/**
	 * 根据JDBC URL获取数据库子类型
	 * @param jdbcUrl jdbc请求路径
	 * @return 数据库子类型
	 */
	public static String getDBSubType(String jdbcUrl){
		return getDBTypeInfo(jdbcUrl).getSubType();
	}


	/**
	 * 根据JDBC URL获取数据库驱动类
	 * @param jdbcUrl jdbc请求路径
	 * @return 数据库驱动类
	 */
	public static String getDBDriver(String jdbcUrl){
		return getDBTypeInfo(jdbcUrl).getDbDriver();
	}

	public static DBTypeInfo getDBTypeInfo(String jdbcUrl){
		for(DBTypeInfo dbTypeInfo: getAllDBtypTeInfos()){
			if(jdbcUrl.startsWith(dbTypeInfo.getJdbcUrlPattern())){
				return dbTypeInfo;
			}
		}
		throw new IllegalArgumentException("暂时不支持 JDBC URL 为 '"+jdbcUrl+ "' 的数据库");
	}

	private static boolean isOracle(String url){
		return (null!=url) && (getDBType(url).equals("Oracle"));
	}
	private static boolean isMySQL(String url){
		return (null!=url) && (getDBType(url).equals("MySQL"));
	}
	private static boolean isSQLServer(String url){
		return (null!=url) && (getDBType(url).equals("SqlServer"));
	}


	/**
	 * 根据原有的数据查询 SQL 语句获取获取其数据总数的 SQL 数据
	 * @param sql 原有的数据查询 SQL
	 * @param countColumnName SQL 语句中对于总数的字段名
	 * @param jdbcUrl jdbc请求路径
	 * @return  查询总数的 SQL
	 */
	public static String getCountSql(String sql, String countColumnName, String jdbcUrl){
		if (isSQLServer(jdbcUrl)){
			return SQLServerUtils.buildCountSqlWithoutOrderBy(sql, countColumnName);
		}else{
			return "SELECT COUNT(*) AS "+countColumnName+" FROM ("+sql+") ALL_DATA";
		}
	}

	/**
	 * 包装 SQL 语句产生用于分页查询的 SQL 语句
	 * @param sql  原有的数据查询 SQL
	 * @param from 开始位置
	 * @param size 分页的大小
	 * @param jdbcUrl jdbc请求路径
	 * @return 分页查询的 SQL
	 */
	public static String adaptSqlRange(String sql, int from, int size, String jdbcUrl){
		if (isOracle(jdbcUrl)){
			return "SELECT * FROM (SELECT datas_.*, rownum rownum_ FROM (\n"
					+ "\t " + sql
					+"\n) datas_ WHERE rownum <= " + (from+size) + ") WHERE rownum_ > " + from;
		}
		if (isMySQL(jdbcUrl)){
			return sql + "\n\t Limit " + from + ", " + size;
		}
		if (isSQLServer(jdbcUrl)){
			return SQLServerUtils.adaptSqlRange(sql, from, size);
		}
		throw new UnsupportedOperationException("Unsupported jdbc connection type: " + jdbcUrl);
	}

	/**
	 * 获取所有数据库类型
	 * @return 所有数据库类型集合
	 */
	public static List<DBTypeInfo> getAllDBtypTeInfos(){
		return dbTypeInfoCollection;
	}
}
