package com.bokesoft.distro.tech.commons.basis.flightrecord.impl;

import javax.sql.DataSource;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.reflect.MethodUtils;

import com.bokesoft.distro.tech.commons.basis.flightrecord.intf.IDataSourceCfgResolver;
import com.bokesoft.distro.tech.commons.basis.flightrecord.struc.database.DatasourceCfgVO;
import com.bokesoft.distro.tech.commons.basis.flightrecord.utils.YFRUtils;

/**
 * 默认的数据库连接信息获取方法
 */
public class DefaultDataSourceCfgResolver implements IDataSourceCfgResolver {

    @Override
    public DatasourceCfgVO build(DataSource dataSource) {
        DatasourceCfgVO result;
        String datasourceClass = dataSource.getClass().getName();
        try {
            if("org.apache.commons.dbcp.BasicDataSource".equals(datasourceClass)) {
                result = buildDatasourceCfgEventByDBCP(dataSource);
                return result;
            }
            if("com.zaxxer.hikari.HikariDataSource".equals(datasourceClass)){
                result = buildDatasourceCfgEventByHikraiCP(dataSource);
                return result;
            }
            if("com.alibaba.druid.spring.boot.autoconfigure.DruidDataSourceWrapper".equals(datasourceClass)){
                result = buildDatasourceCfgEventByDruid(dataSource);
                return result;
            }
        } catch (ReflectiveOperationException e) {
            return ExceptionUtils.rethrow(e);
        }

        throw new RuntimeException("不支持的 DataSource 类型: " + datasourceClass);
    }

    private DatasourceCfgVO buildDatasourceCfgEventByDBCP(DataSource dataSource) throws ReflectiveOperationException {
        String datasourceId = YFRUtils.getDatasourceId(dataSource);
        String jdbcUrl = (String) MethodUtils.invokeMethod(dataSource,"getUrl");
        String driverClassName = (String) MethodUtils.invokeMethod(dataSource,"getDriverClassName");
        int initialSize = (int) MethodUtils.invokeMethod(dataSource,"getInitialSize");
        int minIdle = (int) MethodUtils.invokeMethod(dataSource,"getMinIdle");
        int maxIdle = (int) MethodUtils.invokeMethod(dataSource,"getMaxIdle");
        long maxActive = (int) MethodUtils.invokeMethod(dataSource,"getMaxActive");
        return new DatasourceCfgVO(datasourceId, jdbcUrl, driverClassName, initialSize, minIdle, maxIdle, maxActive);
    }

    private DatasourceCfgVO buildDatasourceCfgEventByDruid(DataSource dataSource) throws ReflectiveOperationException {
        String datasourceId = YFRUtils.getDatasourceId(dataSource);
        String jdbcUrl = (String) MethodUtils.invokeMethod(dataSource,"getRawJdbcUrl");
        String driverClassName = (String) MethodUtils.invokeMethod(dataSource,"getDriverClassName");
        int initialSize = (int) MethodUtils.invokeMethod(dataSource,"getInitialSize");
        int minIdle = (int) MethodUtils.invokeMethod(dataSource,"getMinIdle");
        int maxIdle = (int) MethodUtils.invokeMethod(dataSource,"getMaxIdle");
        long maxActive = (int) MethodUtils.invokeMethod(dataSource,"getMaxActive");
        return new DatasourceCfgVO(datasourceId, jdbcUrl, driverClassName, initialSize, minIdle, maxIdle, maxActive);
    }

    private DatasourceCfgVO buildDatasourceCfgEventByHikraiCP(DataSource dataSource) throws ReflectiveOperationException {
        String datasourceId = YFRUtils.getDatasourceId(dataSource);
        String jdbcUrl = (String) MethodUtils.invokeMethod(dataSource,"getJdbcUrl");
        String driverClassName = (String) MethodUtils.invokeMethod(dataSource,"getDriverClassName");
        int initialSize = (int) MethodUtils.invokeMethod(dataSource,"getMinimumIdle");
        int minIdle = initialSize;
        int maxIdle = (int) MethodUtils.invokeMethod(dataSource,"getMaximumPoolSize");
        long maxActive = (long) MethodUtils.invokeMethod(dataSource,"getMaxLifetime");
        return new DatasourceCfgVO(datasourceId, jdbcUrl, driverClassName, initialSize, minIdle, maxIdle, maxActive);
    }

}
