package com.bokesoft.distro.tech.commons.basis.io.internal;

import com.bokesoft.distro.tech.commons.basis.io.CalendarSerialFolderManager;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class FileUnit {

    private static final Logger log = LoggerFactory.getLogger(FileUnit.class);

    private static final Map<String,Integer> errFileCount = new ConcurrentHashMap<>();

    /**
     *
     * @param fileName 文件名
     * @param clazz 文件对应javabean类
     * @param tmpDir tmp区目录
     * @param readyDir ready区目录
     * @param errDir 错误区目录
     * @param <T> clazz对应泛型
     * @throws IOException
     */
    public static <T> void transfTmpFiletoReady(String fileName, Class<T> clazz,
        String tmpDir, String readyDir,String errDir) throws IOException {

        Path srcPath = Paths.get(tmpDir, fileName);
        Path jsonFolder = Paths.get(tmpDir, ".json");
        jsonFolder.toFile().mkdirs();
        Path jsonFilePath = Paths.get(jsonFolder.toFile().getAbsolutePath(), fileName);
        Path tagPath = Paths.get(readyDir, fileName);
        File tmpFile = srcPath.toFile();
        File jsonFile = jsonFilePath.toFile();
        File targetFile = tagPath.toFile();
        // 将tmp区文件转化为json格式的字符串
        List<T> dataList = (List<T>) FileUnit.loadTmpObject(tmpFile, readyDir, errDir, clazz);
        if(null != dataList && !dataList.isEmpty()) {
            for (T data : dataList) {
                ObjectMapper objectMapper = new ObjectMapper();
                String jsonStr = objectMapper.writeValueAsString(data);
                FileUtils.writeStringToFile(jsonFile, jsonStr + "\r\n", "UTF-8", true);
            }
        }
        FileUtils.delete(tmpFile);
        FileUtils.moveFile(jsonFile, targetFile);
        if (log.isDebugEnabled()){
            log.debug("文件 {} 已迁移完毕, 从{} -> {}  .", fileName, tmpDir, readyDir);
        }
    }
    
    /**
     * 根据文件名,读取准备区目录中的文件数据,并按对象泛型转化
     * @param objFile objectStream序列化的文件
     * @param targetDir json化之后的文件存储地址
     * @param errDir 无法恢复的处理错误文件存储地址
     * @param clazz json化对应class类型
     * @return
     * @throws IOException 
     */
    public static <T> List<T> loadTmpObject(File objFile, String targetDir, String errDir, Class<T> clazz) throws IOException {
        int MAX_RETRY = 3;
        String fileName = objFile.getName();
        List<T> result = new LinkedList<>();
        try (FileInputStream fio = new FileInputStream(objFile);
             ObjectInputStream ois = new ObjectInputStream(fio);){
            while(fio.available()>0) {
                @SuppressWarnings("unchecked")
                T value = (T) ois.readObject();
                result.add(value);
            }
            return result;
        }catch(IOException | ClassNotFoundException e){
            String filePath = objFile.getAbsolutePath();
            log.error("文件 ["+filePath+"] 读取错误 .", e);
            int failCount = errFileCount.computeIfAbsent(fileName, (key) -> {
                return 0;
            });
            if( failCount > MAX_RETRY ){
                if (log.isDebugEnabled()){
                    log.debug("文件 [{}] 读取错误已累计 {} 次, 将迁移到 err 目录 [{}] ...", filePath, MAX_RETRY, errDir);
                }
                File errFile = new File(fileName);
                moveFileToErrFolder(errFile, errDir);
                errFileCount.remove(fileName);
                log.info("文件 [{}] 读取错误已累计 {} 次, 已迁移到 err 目录 [{}] .", filePath, MAX_RETRY, errDir);
            }else {
                failCount = failCount + 1;
                if (log.isDebugEnabled()){
                    log.debug("文件 [{}] 读取错误, 已累计 {} 次.", objFile.getAbsolutePath(), failCount);
                }
                errFileCount.put(fileName, failCount);
                ExceptionUtils.rethrow(e);
            }
            return null;
        }
    }

    /**
     * 无法解析的文件,迁移至err区目录
     * @param errFile 无法解析的文件
     * @param errDir 无法解析文件存储的地址
     * @throws IOException
     */
    private static void moveFileToErrFolder(File errFile, String errDir) throws IOException {
        Path srcPath = Paths.get(errFile.getAbsolutePath());
        Path tagPath = Paths.get(CalendarSerialFolderManager.prepareFolder(errDir),errFile.getName());
        Files.move(srcPath,tagPath);
    }
    
}
